/** @file dcmuid.c
    @brief DICOM Transfer Syntax UID.
 */
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <time.h>
#include <string.h>
/*****************************************************************************/
#include "tpcextensions.h"
/*****************************************************************************/
#include "tpcdcm.h"
/*****************************************************************************/

/*****************************************************************************/
/*! @cond PRIVATE */
/** One item for table of DICOM Transfer Syntax UIDs. */
typedef struct DCM_TRUID_ITEM {
  /** Enumerated Id */
  dcmtruid id;
  /** Pointer to the NULL terminated UID string. */
  char uid[64];
  /** Description of the VR; NULL terminated string. */
  char descr[64];
} DCM_TRUID_ITEM;

/** Table of DICOM Transfer Syntax UIDs. 

    Items must be the same and in the same order as the dcmtruid enums 
    in tpcdcm.h. Table can not be accessed directly outside the c file.
*/
static DCM_TRUID_ITEM dcm_truid[]={
  {DCM_TRUID_UNKNOWN, "1.2.840.10008.1.2", "unknown"}, //< default
  {DCM_TRUID_LEI, "1.2.840.10008.1.2", "implicit VR little endian"}, //< default for DICOM
  {DCM_TRUID_LEE, "1.2.840.10008.1.2.1", "explicit VR little endian"},
  {DCM_TRUID_LEED, "1.2.840.10008.1.2.1.99", "deflated explicit VR little endian"},
  {DCM_TRUID_BEE, "1.2.840.10008.1.2.2", "explicit VR big endian"}, //< Retired
  {DCM_TRUID_JPEG50, "1.2.840.10008.1.2.4.50", "lossy JPEG 8-bit compression"},
  {DCM_TRUID_JPEG51, "1.2.840.10008.1.2.4.51", "lossy JPEG 12-bit compression"},
  {DCM_TRUID_JPEG70, "1.2.840.10008.1.2.4.70", "lossless JPEG"},
  {DCM_TRUID_JPEG80, "1.2.840.10008.1.2.4.80", "lossless JPEG-LS"},
  {DCM_TRUID_JPEG81, "1.2.840.10008.1.2.4.81", "lossy JPEG-LS"},
  {DCM_TRUID_JPEG90, "1.2.840.10008.1.2.4.90", "lossless JPEG 2000"},
  {DCM_TRUID_JPEG91, "1.2.840.10008.1.2.4.91", "JPEG 2000"},
  {DCM_TRUID_JPEG92, "1.2.840.10008.1.2.4.92", "lossless multicomponent JPEG 2000"},
  {DCM_TRUID_JPEG93, "1.2.840.10008.1.2.4.93", "multicomponent JPEG 2000"},
  {DCM_TRUID_MPEG100, "1.2.840.10008.1.2.4.100", "MPEG-2"},
  {DCM_TRUID_MPEG102, "1.2.840.10008.1.2.4.102", "MPEG-4"},
  {DCM_TRUID_MPEG103, "1.2.840.10008.1.2.4.103", "MPEG-4 BD-compatible"},
  {DCM_TRUID_RLE, "1.2.840.10008.1.2.5", "lossless RLE"},
  {DCM_TRUID_RFC, "1.2.840.10008.1.2.6.1", "RFC 2557"},
  {DCM_TRUID_XML, "1.2.840.10008.1.2.6.2", "XML"},
  {DCM_TRUID_INVALID, "", "invalid"}
};
/*! @endcond */
/*****************************************************************************/

/*****************************************************************************/
/** Identify the DICOM Transfer Syntax UID.
 *  @return Returns the enumerated id.
 *  @sa dcmTrUIDDescr
 */
dcmtruid dcmTrUID(
  /** UID string. */
  const char *s
) {
  if(s==NULL || strnlen(s, 5)<5) return(DCM_TRUID_INVALID);

  /* Identify the UID */
  unsigned short int i=1;  // 1 because 0 is unknown
  while(dcm_truid[i].id!=DCM_TRUID_INVALID) {
    if(strcmp(dcm_truid[i].uid, s)==0) return(dcm_truid[i].id);
    i++;
  }
  return(DCM_TRUID_UNKNOWN);
}
/*****************************************************************************/

/*****************************************************************************/
/** Get the DICOM Transfer Syntax UID description.
 *  @return Returns pointer to the description string.
 *  @sa dcmTrUID
 */
char *dcmTrUIDDescr(
  /** Transfer Syntax UID id (DCM_TRUID_LEI, ...). */ 
  dcmtruid id
) {
  unsigned short int i=0;
  while(dcm_truid[i].id!=DCM_TRUID_INVALID) {
    if(id==dcm_truid[i].id) return(dcm_truid[i].descr);
    i++;
  }
  return(dcm_truid[DCM_TRUID_INVALID].descr);
}
/*****************************************************************************/

/*****************************************************************************/
/** Get the DICOM Transfer Syntax UID string.
 *  @return Returns pointer to the description string.
 *  @sa dcmTrUID
 */
char *dcmTrUIDString(
  /** Transfer Syntax UID id (DCM_TRUID_LEI, ...). */ 
  dcmtruid id
) {
  unsigned short int i=0;
  while(dcm_truid[i].id!=DCM_TRUID_INVALID) {
    if(id==dcm_truid[i].id) return(dcm_truid[i].uid);
    i++;
  }
  return(dcm_truid[DCM_TRUID_INVALID].uid);
}
/*****************************************************************************/

/*****************************************************************************/
