/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include "tpcextensions.h"
#include "test_tpcdcm.h"
/*****************************************************************************/
/* Local functions */
int test_makeDcmFile(const char *filename);
int test_makeDcmFile2(const char *filename, TPCSTATUS *status);
int test_makeDcmFile3(const char *filename, TPCSTATUS *status);
/*****************************************************************************/

/*****************************************************************************/
int test_dcmVerifyMagic(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }

  int ret;
  char fname[256];

  if(verbose>1) printf("\ntry with NULL input...\n");
  ret=dcmVerifyMagic(NULL, NULL);
  if(verbose>2) printf("  ret := %d\n", ret);
  if(ret!=0) return(1);

  strcpy(fname, "");
  ret=dcmVerifyMagic(fname, NULL);
  if(verbose>2) printf("  ret := %d\n", ret);
  if(ret!=0) return(2);

  if(verbose>1) printf("\ntry with nonexisting file...\n");
  strcpy(fname, "nonexistingfile");
  ret=dcmVerifyMagic(fname, NULL);
  if(verbose>2) printf("  ret := %d\n", ret);
  if(ret!=0) return(3);

  if(verbose>1) printf("\ncreate test file...\n");
  strcpy(fname, "test1.dcm");
  ret=test_makeDcmFile(fname);
  if(ret!=0) return(10);

  if(verbose>1) printf("\ntry with test file...\n");
  ret=dcmVerifyMagic(fname, NULL);
  if(verbose>2) printf("  ret := %d\n", ret);
  if(ret==0) return(11);

  if(verbose>1) printf("\ntry with file pointer to test file...\n");
  {
    FILE *fp;
    fp=fopen(fname, "rb"); if(fp==NULL) return(12);
    ret=dcmVerifyMagic(NULL, fp);
    if(verbose>2) printf("  ret := %d\n", ret);
    if(ret==0) {fclose(fp); return(13);}
    if(verbose>1) printf("\ntry another time, after file position was moved...\n");
    ret=dcmVerifyMagic(NULL, fp);
    if(verbose>2) printf("  ret := %d\n", ret);
    if(ret==0) {fclose(fp); return(14);}
    fclose(fp);
  }

  statusSet(status, __func__, __FILE__, __LINE__, 0);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
int test_dcmReadTransferSyntaxUID(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }

  dcmtruid uid;

  if(verbose>1) printf("\n try with NULL input...\n");
  uid=dcmReadTransferSyntaxUID(NULL);
  if(uid!=DCM_TRUID_INVALID) return(1);

  statusSet(status, __func__, __FILE__, __LINE__, 0);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
int test_dcmReadFileTag(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }

  int ret=0;
  DCMTAG tag;

  if(verbose>1) printf("\n try with NULL input...\n");
  ret=dcmReadFileTag(NULL, &tag);
  if(ret==0) return(1);

  // See also test_dcmWriteFileTag

  statusSet(status, __func__, __FILE__, __LINE__, 0);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
int test_dcmWriteFileTag(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }

  char *fname="dcmwritefiletagtest.dat";
  int ret=0;
  DCMTAG tag, tag2;

  if(verbose>1) printf("\n try with NULL filepointer...\n");
  ret=dcmWriteFileTag(NULL, &tag);
  if(ret!=TPCERROR_FAIL) return(1);

  FILE *fp=fopen(fname, "wb"); if(fp==NULL) return(2);

  if(verbose>1) printf("\n try with NULL tag...\n");
  ret=dcmWriteFileTag(fp, NULL);
  if(ret!=TPCERROR_FAIL) {fclose(fp); return(11);}

  if(verbose>1) printf("\n try with valid input...\n");
  tag.group=0x002;
  tag.element=0x010;
  ret=dcmWriteFileTag(fp, &tag);
  if(ret!=TPCERROR_OK) {fclose(fp); return(12);}

  fclose(fp);

  if(verbose>1) printf("\n try to read...\n");
  fp=fopen(fname, "rb"); if(fp==NULL) return(21);
  ret=dcmReadFileTag(fp, &tag2);
  if(ret!=0) {fclose(fp); return(22);}
  if(tag.group!=tag2.group || tag.element!=tag2.element) {fclose(fp); return(23);}
  fclose(fp);

  if(verbose>1) printf("\n ... ok\n");
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
int test_dcmWriteFileSQDelimItem(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }

  char *fname="dcmwritefilesqdelimtest.dat";
  int ret=0;
  DCMTAG tag, tag2;

  if(verbose>1) printf("\n try with NULL filepointer...\n");
  ret=dcmWriteFileSQDelimItem(NULL);
  if(ret!=TPCERROR_FAIL) return(1);

  if(verbose>1) printf("\n try with valid input...\n");
  FILE *fp=fopen(fname, "wb"); if(fp==NULL) return(10);

  ret=dcmWriteFileSQDelimItem(fp);
  if(ret!=TPCERROR_OK) return(11);

  fclose(fp);

  if(verbose>1) printf("\n try to read...\n");
  fp=fopen(fname, "rb"); if(fp==NULL) return(21);
  ret=dcmReadFileTag(fp, &tag);
  if(ret==0) ret=dcmReadFileTag(fp, &tag2);
  if(ret!=0) {fclose(fp); return(22);}
  if(tag.group!=0xFFFE || tag.element!=0xE0DD) {fclose(fp); return(23);}
  if(tag2.group!=0x0000 || tag2.element!=0x0000) {fclose(fp); return(24);}
  ret=dcmReadFileTag(fp, &tag); if(ret==0) {fclose(fp); return(25);}
  fclose(fp);

  if(verbose>1) printf("\n ... ok\n");
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
int test_dcmWriteFileSQItemDelimTag(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }

  char *fname="dcmwritefilesqitemdelimtest.dat";
  int ret=0;
  DCMTAG tag, tag2;

  if(verbose>1) printf("\n try with NULL filepointer...\n");
  ret=dcmWriteFileSQItemDelimTag(NULL);
  if(ret!=TPCERROR_FAIL) return(1);

  if(verbose>1) printf("\n try with valid input...\n");
  FILE *fp=fopen(fname, "wb"); if(fp==NULL) return(10);

  ret=dcmWriteFileSQItemDelimTag(fp);
  if(ret!=TPCERROR_OK) return(11);

  fclose(fp);

  if(verbose>1) printf("\n try to read...\n");
  fp=fopen(fname, "rb"); if(fp==NULL) return(21);
  ret=dcmReadFileTag(fp, &tag);
  if(ret==0) ret=dcmReadFileTag(fp, &tag2);
  if(ret!=0) {fclose(fp); return(22);}
  if(tag.group!=0xFFFE || tag.element!=0xE00D) {fclose(fp); return(23);}
  if(tag2.group!=0x0000 || tag2.element!=0x0000) {fclose(fp); return(24);}
  ret=dcmReadFileTag(fp, &tag); if(ret==0) {fclose(fp); return(25);}
  fclose(fp);

  if(verbose>1) printf("\n ... ok\n");
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
int test_dcmReadFileVRVL(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }

  if(verbose>1) printf("\n try with NULL input...\n");
  if(dcmReadFileVRVL(NULL, NULL, NULL, NULL)==TPCERROR_OK) return(1);

  // See also test_dcmWriteFileVRVL

  if(verbose>1) printf("\n ... ok\n");
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
int test_dcmWriteFileVRVL(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }


  char *fname="dcmwritefilevrvltest.dat";
  int ret=0;
  dcmvr vr, vr2;
  unsigned int vl=0, n, vl2, n2;


  if(verbose>1) printf("\n try with NULL filepointer...\n");
  vr=DCM_VR_FD;
  ret=dcmWriteFileVRVL(NULL, vr, vl, &n);
  if(ret!=TPCERROR_FAIL) return(1);

  FILE *fp=fopen(fname, "wb"); if(fp==NULL) return(2);

  if(verbose>1) printf("\n try with unsupported VR...\n");
  vr=DCM_VR_INVALID;
  ret=dcmWriteFileVRVL(fp, vr, vl, &n);
  if(ret!=TPCERROR_FAIL) {fclose(fp); return(3);}

  if(verbose>1) printf("\n try with supported VR... 2 bytes\n");
  vr=DCM_VR_FD; vl=8;
  ret=dcmWriteFileVRVL(fp, vr, vl, &n);
  if(ret!=TPCERROR_OK) {fclose(fp); return(11);}
  if(n!=4) {fclose(fp); return(12);}

  fclose(fp);

  if(verbose>1) printf("\n try to read...\n");
  fp=fopen(fname, "rb"); if(fp==NULL) return(21);
  ret=dcmReadFileVRVL(fp, &vr2, &vl2, &n2);
  if(ret!=TPCERROR_OK) {fclose(fp); return(22);}
  if(vr!=vr2 || vl!=vl2 || n!=n2) {fclose(fp); return(23);}
  fclose(fp);


  if(verbose>1) printf("\n try with supported VR... 4 bytes\n");
  fp=fopen(fname, "wb"); if(fp==NULL) return(30);

  vr=DCM_VR_UR; vl=64;
  ret=dcmWriteFileVRVL(fp, vr, vl, &n);
  if(ret!=TPCERROR_OK) {fclose(fp); return(31);}
  if(n!=8) {fclose(fp); return(32);}

  fclose(fp);

  if(verbose>1) printf("\n try to read...\n");
  fp=fopen(fname, "rb"); if(fp==NULL) return(41);
  ret=dcmReadFileVRVL(fp, &vr2, &vl2, &n2);
  if(ret!=TPCERROR_OK) {fclose(fp); return(43);}
  if(vr!=vr2 || vl!=vl2 || n!=n2) {fclose(fp); return(44);}
  fclose(fp);


  if(verbose>1) printf("\n ... ok\n");
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
int test_dcmFileRead(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }

  int ret;
  char fname[256];

  DCMFILE dcm; dcmfileInit(&dcm);

  if(verbose>1) printf("\ntry with NULL input...\n");
  ret=dcmFileRead(NULL, NULL, 0, NULL);
  if(verbose>2) printf("  ret := %d\n", ret);
  if(ret==0) return(1);

  strcpy(fname, "");
  ret=dcmFileRead(fname, &dcm, 0, NULL);
  if(verbose>2) printf("  ret := %d\n", ret);
  if(ret==0) return(2);

  if(verbose>1) printf("\ntry with nonexisting file...\n");
  strcpy(fname, "nonexistingfile");
  dcmfileFree(&dcm);
  ret=dcmFileRead(fname, &dcm, 0, NULL);
  if(verbose>2) printf("  ret := %d\n", ret);
  if(ret==0) return(3);

  if(verbose>1) printf("\ncreate test file...\n");
  strcpy(fname, "test1.dcm");
  ret=test_makeDcmFile(fname);
  if(ret!=0) return(10);

  if(verbose>1) printf("\ntry with test file...\n");
  if(verbose>1) printf("this must fail for now, since test file is not ready!\n");
  dcmfileFree(&dcm);
  ret=dcmFileRead(fname, &dcm, 0, NULL);
  if(verbose>2) printf("  ret := %d\n", ret);
  dcmfileFree(&dcm);
  if(ret==0) return(11);

  if(verbose>1) printf("\ntry with an Inveon DICOM file...\n");
  strcpy(fname, "f170.dcm");
  dcmfileFree(&dcm);
  if(verbose>3) ret=dcmFileRead(fname, &dcm, 0, status);
  else ret=dcmFileRead(fname, &dcm, 0, NULL);
  if(verbose>2) printf("  ret := %d\n", ret);
  dcmfileFree(&dcm);
  if(ret!=0) return(21);

  /* More testing inside test_dcmFileWrite() */

  statusSet(status, __func__, __FILE__, __LINE__, 0);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
int test_dcmFileWrite(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }

  int ret;
  char fname[256];

  DCMFILE dcm; dcmfileInit(&dcm);

  if(verbose>1) printf("\ntry with NULL input...\n");
  ret=dcmFileWrite(NULL, NULL, NULL);
  if(verbose>2) printf("  ret := %d\n", ret);
  if(ret==0) return(1);

  strcpy(fname, "");
  ret=dcmFileWrite(fname, &dcm, NULL);
  if(verbose>2) printf("  ret := %d\n", ret);
  if(ret==0) return(2);

  strcpy(fname, "test.dcm");
  ret=dcmFileWrite(fname, &dcm, NULL);
  if(verbose>2) printf("  ret := %d\n", ret);
  if(ret==0) return(3);


  if(verbose>1) printf("\ntry to read and write an Inveon DICOM file...\n");
  strcpy(fname, "f170.dcm");
  dcmfileFree(&dcm);
  ret=dcmFileRead(fname, &dcm, 0, NULL);
  if(verbose>2) printf("  ret := %d\n", ret);
  if(ret!=0) {dcmfileFree(&dcm); return(10);}

  /* Change Study ID to prevent other tests from believing that this file belongs
     to the same study as the parent file */
  {
    DCMTAG tag;
    tag.group=0x0020; tag.element=0x0010;
    DCMITEM *iptr;
    iptr=dcmFindTag(dcm.item, 0, &tag, 0);
    if(iptr!=NULL && iptr->rd!=NULL) iptr->rd[0]='2';
  }

  strcpy(fname, "test2.dcm");
  ret=dcmFileWrite(fname, &dcm, status);
  if(verbose>2) printf("  ret := %d\n", ret);
  if(ret!=0) {dcmfileFree(&dcm); return(11);}

  dcmfileFree(&dcm);



  statusSet(status, __func__, __FILE__, __LINE__, 0);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
int test_dcmTestFile(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }

  int ret=0;

  ret=test_makeDcmFile2("testfile.dcm", status);
  if(ret!=0) {
    if(verbose>0) printf("  ret := %d\n", ret);
  }

  ret=test_makeDcmFile3("testfile3D", status);
  if(ret!=0) {
    if(verbose>0) printf("  ret := %d\n", ret);
  }
  return(ret);
}
/*****************************************************************************/

/*****************************************************************************/
/** Make simple DICOM file for testing. */
int test_makeDcmFile(
  /** Name of file. */
  const char *filename
) {
  FILE *fp;

  if(filename==NULL || !filename[0]) return(1);
  fp=fopen(filename, "wb");
  if(fp==NULL) return(1);

  /* Write zero into the first 128 bytes */
  char buf[128];
  size_t n;
 
  for(int i=0; i<128; i++) buf[i]=(char)0;
  n=fwrite(&buf, 1, 128, fp);
  if(n!=(size_t)128) {fclose(fp); return(2);}
  
  /* Write the magic number */
  strcpy(buf, "DICM");
  n=fwrite(&buf, 1, 4, fp);
  if(n!=(size_t)4) {fclose(fp); return(3);}

  fclose(fp);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Make 2D DICOM file for testing. 
    Image size is 128x128 pixels.
    This image has values between 0 and 8700 Bq/mL.
 */
int test_makeDcmFile2(
  /** Name of file. */
  const char *filename,
  /** Pointer to status data; enter NULL if not needed. */
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }

  if(filename==NULL || !filename[0]) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL); return(1);}

  int ret=0;

  /*
   *  Make DCMFILE data structure
   */
  DCMFILE dcm; dcmfileInit(&dcm);

  /* Set filename and Transfer UID */
  strlcpy(dcm.filename, filename, FILENAME_MAX);
  dcm.truid=DCM_TRUID_LEE;

  /* Add items */
  DCMTAG tag;

  unsigned int ul;

  dcmTagSet(&tag, 0x0002, 0x0000);
  ul=0;
  if(dcmAddItem(&dcm, NULL, 0, tag, DCM_VR_UL, 0xFFFFFFFF, (char*)&ul, verbose-1)!=0) {
    dcmfileFree(&dcm); statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL); return(100);}
  dcmTagSet(&tag, 0x0002, 0x0001);
  if(dcmAddItem(&dcm, NULL, 0, tag, DCM_VR_OB, 2, "\0\1", verbose-1)!=0) {
    dcmfileFree(&dcm); statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL); return(100);}

  dcmTagSet(&tag, 0x0002, 0x0002);
  if(dcmAddItem(&dcm, NULL, 0, tag, DCM_VR_UI, 0xFFFFFFFF, "1.2.840.10008.5.1.4.1.1.128", verbose-1)!=0) {
    dcmfileFree(&dcm); statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL); return(101);}
  dcmTagSet(&tag, 0x0002, 0x0003);
  if(dcmAddItem(&dcm, NULL, 0, tag, DCM_VR_UI, 0xFFFFFFFF, "1.2.826.0.1.3417726.3.0.20180222", verbose-1)!=0) {
    dcmfileFree(&dcm); statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL); return(102);}

  dcmTagSet(&tag, 0x0002, 0x0010);
  if(dcmAddItem(&dcm, NULL, 0, tag, DCM_VR_UI, 0xFFFFFFFF, dcmTrUIDString(dcm.truid), verbose-1)!=0) {
    dcmfileFree(&dcm); statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL); return(111);}

  dcmTagSet(&tag, 0x0002, 0x0012);
  if(dcmAddItem(&dcm, NULL, 0, tag, DCM_VR_UI, 0xFFFFFFFF, "1.3.6.1.4.1.25403.1.1.1", verbose-1)!=0) {
    dcmfileFree(&dcm); statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL); return(121);}
  dcmTagSet(&tag, 0x0002, 0x0013);
  if(dcmAddItem(&dcm, NULL, 0, tag, DCM_VR_SH, 0xFFFFFFFF, "Dicom 0.1", verbose-1)!=0) {
    dcmfileFree(&dcm); statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL); return(122);}


  dcmTagSet(&tag, 0x0008, 0x0000);
  ul=0;
  if(dcmAddItem(&dcm, NULL, 0, tag, DCM_VR_UL, 0xFFFFFFFF, (char*)&ul, verbose-1)!=0) {
    dcmfileFree(&dcm); statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL); return(200);}

  dcmTagSet(&tag, 0x0008, 0x0008);
  if(dcmAddItem(&dcm, NULL, 0, tag, DCM_VR_CS, 17, "ORIGINAL/PRIMARY", verbose-1)!=0) {
    dcmfileFree(&dcm); statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL); return(201);}
  dcmTagSet(&tag, 0x0008, 0x0012);
  if(dcmAddItem(&dcm, NULL, 0, tag, DCM_VR_DA, 0xFFFFFFFF, "20180222", verbose-1)!=0) {
    dcmfileFree(&dcm); statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL); return(202);}
  dcmTagSet(&tag, 0x0008, 0x0016);
  if(dcmAddItem(&dcm, NULL, 0, tag, DCM_VR_UI, 0xFFFFFFFF, "1.2.840.10008.5.1.4.1.1.128", verbose-1)!=0) {
    dcmfileFree(&dcm); statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL); return(203);}
  dcmTagSet(&tag, 0x0008, 0x0018); // SOP Instance UID
  if(dcmAddItem(&dcm, NULL, 0, tag, DCM_VR_UI, 0xFFFFFFFF, "1.2.826.0.1.3417726.3.0.20180222", verbose-1)!=0) {
    dcmfileFree(&dcm); statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL); return(204);}
  dcmTagSet(&tag, 0x0008, 0x0060);
  if(dcmAddItem(&dcm, NULL, 0, tag, DCM_VR_CS, 3, "PT", verbose-1)!=0) {
    dcmfileFree(&dcm); statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL); return(205);}

  dcmTagSet(&tag, 0x0008, 0x0104);
  if(dcmAddItem(&dcm, NULL, 0, tag, DCM_VR_LO, 0xFFFFFFFF, "^18^Fluorine", verbose-1)!=0) {
    dcmfileFree(&dcm); statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL); return(211);}


  dcmTagSet(&tag, 0x0018, 0x0000);
  ul=0;
  if(dcmAddItem(&dcm, NULL, 0, tag, DCM_VR_UL, 0xFFFFFFFF, (char*)&ul, verbose-1)!=0) {
    dcmfileFree(&dcm); statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL); return(300);}

  dcmTagSet(&tag, 0x0018, 0x0050);  // Slice thickness (mm)
  if(dcmAddItem(&dcm, NULL, 0, tag, DCM_VR_DS, 0xFFFFFFFF, "0.796", verbose-1)!=0) {
    dcmfileFree(&dcm); statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL); return(301);}
  dcmTagSet(&tag, 0x0018, 0x1242);
  if(dcmAddItem(&dcm, NULL, 0, tag, DCM_VR_IS, 0xFFFFFFFF, "12000", verbose-1)!=0) {
    dcmfileFree(&dcm); statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL); return(302);}

  dcmTagSet(&tag, 0x0018, 0x1075); // half-life
  if(dcmAddItem(&dcm, NULL, 0, tag, DCM_VR_DS, 0xFFFFFFFF, "6586.2", verbose-1)!=0) {
    dcmfileFree(&dcm); statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL); return(303);}
  dcmTagSet(&tag, 0x0018, 0x1076); // branching
  if(dcmAddItem(&dcm, NULL, 0, tag, DCM_VR_DS, 0xFFFFFFFF, "0.967", verbose-1)!=0) {
    dcmfileFree(&dcm); statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL); return(304);}



  dcmTagSet(&tag, 0x0020, 0x0000);
  ul=0;
  if(dcmAddItem(&dcm, NULL, 0, tag, DCM_VR_UL, 0xFFFFFFFF, (char*)&ul, verbose-1)!=0) {
    dcmfileFree(&dcm); statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL); return(400);}

  dcmTagSet(&tag, 0x0020, 0x000d); // Study Instance UID
  if(dcmAddItem(&dcm, NULL, 0, tag, DCM_VR_UI, 0xFFFFFFFF, "1.2.826.0.2.332211.1.4478", verbose-1)!=0) {
    dcmfileFree(&dcm); statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL); return(401);}
  dcmTagSet(&tag, 0x0020, 0x000e); // Series Instance UID
  if(dcmAddItem(&dcm, NULL, 0, tag, DCM_VR_UI, 0xFFFFFFFF, "1.2.826.0.2.332211.1.5589.2018", verbose-1)!=0) {
    dcmfileFree(&dcm); statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL); return(402);}
  dcmTagSet(&tag, 0x0020, 0x0010); // Study ID
  if(dcmAddItem(&dcm, NULL, 0, tag, DCM_VR_SH, 0xFFFFFFFF, "1.1.0902.0401", verbose-1)!=0) {
    dcmfileFree(&dcm); statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL); return(403);}

  dcmTagSet(&tag, 0x0020, 0x0011); // series number
  if(dcmAddItem(&dcm, NULL, 0, tag, DCM_VR_IS, 0xFFFFFFFF, "1", verbose-1)!=0) {
    dcmfileFree(&dcm); statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL); return(411);}
  dcmTagSet(&tag, 0x0020, 0x0012); // Acquisition number
  if(dcmAddItem(&dcm, NULL, 0, tag, DCM_VR_IS, 0xFFFFFFFF, "170", verbose-1)!=0) {
    dcmfileFree(&dcm); statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL); return(412);}
  dcmTagSet(&tag, 0x0020, 0x0013); // Instance (image) number
  if(dcmAddItem(&dcm, NULL, 0, tag, DCM_VR_IS, 0xFFFFFFFF, "170", verbose-1)!=0) {
    dcmfileFree(&dcm); statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL); return(413);}
  dcmTagSet(&tag, 0x0020, 0x0032); // Image position (patient)
  if(dcmAddItem(&dcm, NULL, 0, tag, DCM_VR_DS, 25, "49.30032\\49.30032\\54.924", verbose-1)!=0) {
    dcmfileFree(&dcm); statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL); return(421);}
  dcmTagSet(&tag, 0x0020, 0x0037); // Image orientation (patient)
  if(dcmAddItem(&dcm, NULL, 0, tag, DCM_VR_DS, 24, "1.0\\0.0\\0.0\\0.0\\1.0\\0.0", verbose-1)!=0) {
    dcmfileFree(&dcm); statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL); return(422);}


  unsigned short int us;

  dcmTagSet(&tag, 0x0028, 0x0000);
  ul=0;
  if(dcmAddItem(&dcm, NULL, 0, tag, DCM_VR_UL, 0xFFFFFFFF, (char*)&ul, verbose-1)!=0) {
    dcmfileFree(&dcm); statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL); return(500);}

  dcmTagSet(&tag, 0x0028, 0x0002); // samples per pixel
  us=1;
  if(dcmAddItem(&dcm, NULL, 0, tag, DCM_VR_US, 0xFFFFFFFF, (char*)&us, verbose-1)!=0) {
    dcmfileFree(&dcm); statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL); return(501);}
  dcmTagSet(&tag, 0x0028, 0x0010);
  us=128;
  if(dcmAddItem(&dcm, NULL, 0, tag, DCM_VR_US, 0xFFFFFFFF, (char*)&us, verbose-1)!=0) {
    dcmfileFree(&dcm); statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL); return(502);}
  dcmTagSet(&tag, 0x0028, 0x0011);
  us=128;
  if(dcmAddItem(&dcm, NULL, 0, tag, DCM_VR_US, 0xFFFFFFFF, (char*)&us, verbose-1)!=0) {
    dcmfileFree(&dcm); statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL); return(503);}
  dcmTagSet(&tag, 0x0028, 0x0030);
  if(dcmAddItem(&dcm, NULL, 0, tag, DCM_VR_DS, 18, "0.776383\\0.776383", verbose-1)!=0) {
    dcmfileFree(&dcm); statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL); return(504);}
  dcmTagSet(&tag, 0x0028, 0x0100);
  us=16;
  if(dcmAddItem(&dcm, NULL, 0, tag, DCM_VR_US, 0xFFFFFFFF, (char*)&us, verbose-1)!=0) {
    dcmfileFree(&dcm); statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL); return(511);}
  dcmTagSet(&tag, 0x0028, 0x0101);
  us=16;
  if(dcmAddItem(&dcm, NULL, 0, tag, DCM_VR_US, 0xFFFFFFFF, (char*)&us, verbose-1)!=0) {
    dcmfileFree(&dcm); statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL); return(512);}
  dcmTagSet(&tag, 0x0028, 0x0102);
  us=15;
  if(dcmAddItem(&dcm, NULL, 0, tag, DCM_VR_US, 0xFFFFFFFF, (char*)&us, verbose-1)!=0) {
    dcmfileFree(&dcm); statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL); return(513);}
  dcmTagSet(&tag, 0x0028, 0x0103); // pixel representation: 1=signed, 0=unsigned
  us=1;
  if(dcmAddItem(&dcm, NULL, 0, tag, DCM_VR_US, 0xFFFFFFFF, (char*)&us, verbose-1)!=0) {
    dcmfileFree(&dcm); statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL); return(514);}
  dcmTagSet(&tag, 0x0028, 0x01052); // rescale ic
  if(dcmAddItem(&dcm, NULL, 0, tag, DCM_VR_DS, 0xFFFFFFFF, "0.0", verbose-1)!=0) {
    dcmfileFree(&dcm); statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL); return(515);}
  dcmTagSet(&tag, 0x0028, 0x01053); // rescale slope
  if(dcmAddItem(&dcm, NULL, 0, tag, DCM_VR_DS, 0xFFFFFFFF, "100.0", verbose-1)!=0) {
    dcmfileFree(&dcm); statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL); return(516);}


  dcmTagSet(&tag, 0x0054, 0x0000);
  ul=0;
  if(dcmAddItem(&dcm, NULL, 0, tag, DCM_VR_UL, 0xFFFFFFFF, (char*)&ul, verbose-1)!=0) {
    dcmfileFree(&dcm); statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL); return(600);}

  dcmTagSet(&tag, 0x0054, 0x0081);
  us=159; // nr of slices
  if(dcmAddItem(&dcm, NULL, 0, tag, DCM_VR_US, 0xFFFFFFFF, (char*)&us, verbose-1)!=0) {
    dcmfileFree(&dcm); statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL); return(601);}
  dcmTagSet(&tag, 0x0054, 0x0101);
  us=24; // nr of frames
  if(dcmAddItem(&dcm, NULL, 0, tag, DCM_VR_US, 0xFFFFFFFF, (char*)&us, verbose-1)!=0) {
    dcmfileFree(&dcm); statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL); return(602);}
  dcmTagSet(&tag, 0x0054, 0x1000);
  if(dcmAddItem(&dcm, NULL, 0, tag, DCM_VR_CS, 14, "DYNAMIC\\IMAGE", verbose-1)!=0) {
    dcmfileFree(&dcm); statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL); return(603);}
  dcmTagSet(&tag, 0x0054, 0x1001);
  if(dcmAddItem(&dcm, NULL, 0, tag, DCM_VR_CS, 5, "BQML", verbose-1)!=0) {
    dcmfileFree(&dcm); statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL); return(604);}
  dcmTagSet(&tag, 0x0054, 0x1002);
  if(dcmAddItem(&dcm, NULL, 0, tag, DCM_VR_CS, 9, "EMISSION", verbose-1)!=0) {
    dcmfileFree(&dcm); statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL); return(605);}
  dcmTagSet(&tag, 0x0054, 0x1102); // decay correction to
  if(dcmAddItem(&dcm, NULL, 0, tag, DCM_VR_CS, 6, "START", verbose-1)!=0) {
    dcmfileFree(&dcm); statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL); return(606);}
  dcmTagSet(&tag, 0x0054, 0x01300); // frame reference time
  if(dcmAddItem(&dcm, NULL, 0, tag, DCM_VR_DS, 0xFFFFFFFF, "18000.0", verbose-1)!=0) {
    dcmfileFree(&dcm); statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL); return(607);}
  dcmTagSet(&tag, 0x0054, 0x01321); // decay correction factor
  if(dcmAddItem(&dcm, NULL, 0, tag, DCM_VR_DS, 0xFFFFFFFF, "1.0019", verbose-1)!=0) {
    dcmfileFree(&dcm); statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL); return(608);}
  dcmTagSet(&tag, 0x0054, 0x1330);
  us=170; // image index
  if(dcmAddItem(&dcm, NULL, 0, tag, DCM_VR_US, 0xFFFFFFFF, (char*)&us, verbose-1)!=0) {
    dcmfileFree(&dcm); statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL); return(611);}


  /* Pixel data */
  dcmTagSet(&tag, 0x7FE0, 0x0000);
  ul=32780;
  if(dcmAddItem(&dcm, NULL, 0, tag, DCM_VR_UL, 0xFFFFFFFF, (char*)&ul, verbose-1)!=0) {
    dcmfileFree(&dcm); statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL); return(600);}

  dcmTagSet(&tag, 0x7FE0, 0x0010);
  unsigned int si, xi, yi, xdim=128, ydim=128;
  short int m[ydim][xdim];
  for(yi=0; yi<ydim; yi++)
    for(xi=0; xi<xdim; xi++)
      m[yi][xi]=xi/5 + yi/2;
  short int pxls[xdim*ydim];
  si=0;
  for(yi=0; yi<ydim; yi++)
    for(xi=0; xi<xdim; xi++)
      pxls[si++]=m[yi][xi];
  if(dcmAddItem(&dcm, NULL, 0, tag, DCM_VR_OW, xdim*ydim*sizeof(short int), (char*)pxls, verbose-1)!=0) {
    dcmfileFree(&dcm); statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL); return(1001);}


  /*
   *  Write DCMFILE into a file
   */
  ret=dcmFileWrite(filename, &dcm, status);
  if(verbose>1) printf("  ret := %d\n", ret);
  if(ret!=0) {dcmfileFree(&dcm); return(3);}


  /* Free the contents of DCMFILE */
  dcmfileFree(&dcm);

  statusSet(status, __func__, __FILE__, __LINE__, 0);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Make 3D DICOM file for testing. 
    Image size is 64x64x51 pixels.
    This image has values between 0 and 6200 Bq/mL.
 */
int test_makeDcmFile3(
  /** Initial part of DICOM file name; sequential numbers and .dcm will be added. */
  const char *filename,
  /** Pointer to status data; enter NULL if not needed. */
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }

  if(filename==NULL || !filename[0]) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL); return(1);}

  int ret=0;
  char buf[256];

  unsigned int dimx, dimy, dimz;
  dimx=dimy=64; dimz=51;
  double planewidth=0.5;

  /*
   *  Make DCMFILE data structure
   */
  DCMFILE dcm; dcmfileInit(&dcm);

  for(unsigned int plane=1; plane<=dimz; plane++) {

    /* Set filename and Transfer UID */
    sprintf(buf, "%s%d.dcm", filename, plane);
    strlcpy(dcm.filename, buf, FILENAME_MAX);
    dcm.truid=DCM_TRUID_LEE;

    /* Add items */
    DCMTAG tag;

    unsigned int ul;

    dcmTagSet(&tag, 0x0002, 0x0000);
    ul=0;
    if(dcmAddItem(&dcm, NULL, 0, tag, DCM_VR_UL, 0xFFFFFFFF, (char*)&ul, verbose-1)!=0) {
      dcmfileFree(&dcm); statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL); return(100);}
    dcmTagSet(&tag, 0x0002, 0x0001);
    if(dcmAddItem(&dcm, NULL, 0, tag, DCM_VR_OB, 2, "\0\1", verbose-1)!=0) {
      dcmfileFree(&dcm); statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL); return(100);}

    dcmTagSet(&tag, 0x0002, 0x0002);
    if(dcmAddItem(&dcm, NULL, 0, tag, DCM_VR_UI, 0xFFFFFFFF, "1.2.840.10008.5.1.4.1.1.128", verbose-1)!=0) {
      dcmfileFree(&dcm); statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL); return(101);}
    dcmTagSet(&tag, 0x0002, 0x0003); // Media Stored SOP Instance UID
    if(dcmAddItem(&dcm, NULL, 0, tag, DCM_VR_UI, 0xFFFFFFFF, "1.2.826.0.1.3417726.3.0.20180616", verbose-1)!=0) {
      dcmfileFree(&dcm); statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL); return(102);}

    dcmTagSet(&tag, 0x0002, 0x0010);
    if(dcmAddItem(&dcm, NULL, 0, tag, DCM_VR_UI, 0xFFFFFFFF, dcmTrUIDString(dcm.truid), verbose-1)!=0) {
      dcmfileFree(&dcm); statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL); return(111);}

    dcmTagSet(&tag, 0x0002, 0x0012);
    if(dcmAddItem(&dcm, NULL, 0, tag, DCM_VR_UI, 0xFFFFFFFF, "1.3.6.1.4.1.25403.1.1.1", verbose-1)!=0) {
      dcmfileFree(&dcm); statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL); return(121);}
    dcmTagSet(&tag, 0x0002, 0x0013);
    if(dcmAddItem(&dcm, NULL, 0, tag, DCM_VR_SH, 0xFFFFFFFF, "Dicom 0.1", verbose-1)!=0) {
      dcmfileFree(&dcm); statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL); return(122);}


    dcmTagSet(&tag, 0x0008, 0x0000);
    ul=0;
    if(dcmAddItem(&dcm, NULL, 0, tag, DCM_VR_UL, 0xFFFFFFFF, (char*)&ul, verbose-1)!=0) {
      dcmfileFree(&dcm); statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL); return(200);}

    dcmTagSet(&tag, 0x0008, 0x0008);
    if(dcmAddItem(&dcm, NULL, 0, tag, DCM_VR_CS, 17, "ORIGINAL/PRIMARY", verbose-1)!=0) {
      dcmfileFree(&dcm); statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL); return(201);}
    dcmTagSet(&tag, 0x0008, 0x0012);
    if(dcmAddItem(&dcm, NULL, 0, tag, DCM_VR_DA, 0xFFFFFFFF, "20180616", verbose-1)!=0) {
      dcmfileFree(&dcm); statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL); return(202);}
    dcmTagSet(&tag, 0x0008, 0x0016);
    if(dcmAddItem(&dcm, NULL, 0, tag, DCM_VR_UI, 0xFFFFFFFF, "1.2.840.10008.5.1.4.1.1.128", verbose-1)!=0) {
      dcmfileFree(&dcm); statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL); return(203);}
    dcmTagSet(&tag, 0x0008, 0x0018); // SOP Instance UID
    if(dcmAddItem(&dcm, NULL, 0, tag, DCM_VR_UI, 0xFFFFFFFF, "1.2.826.0.1.3417726.3.0.20180616", verbose-1)!=0) {
      dcmfileFree(&dcm); statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL); return(204);}
    dcmTagSet(&tag, 0x0008, 0x0060);
    if(dcmAddItem(&dcm, NULL, 0, tag, DCM_VR_CS, 3, "PT", verbose-1)!=0) {
      dcmfileFree(&dcm); statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL); return(205);}

    dcmTagSet(&tag, 0x0008, 0x0104);
    if(dcmAddItem(&dcm, NULL, 0, tag, DCM_VR_LO, 0xFFFFFFFF, "^18^Fluorine", verbose-1)!=0) {
      dcmfileFree(&dcm); statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL); return(211);}


    dcmTagSet(&tag, 0x0018, 0x0000);
    ul=0;
    if(dcmAddItem(&dcm, NULL, 0, tag, DCM_VR_UL, 0xFFFFFFFF, (char*)&ul, verbose-1)!=0) {
      dcmfileFree(&dcm); statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL); return(300);}

    dcmTagSet(&tag, 0x0018, 0x0050);  // Slice thickness (mm)
    sprintf(buf, "%g", planewidth);
    if(dcmAddItem(&dcm, NULL, 0, tag, DCM_VR_DS, 0xFFFFFFFF, buf, verbose-1)!=0) {
      dcmfileFree(&dcm); statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL); return(301);}
    dcmTagSet(&tag, 0x0018, 0x1242);
    if(dcmAddItem(&dcm, NULL, 0, tag, DCM_VR_IS, 0xFFFFFFFF, "12000", verbose-1)!=0) {
      dcmfileFree(&dcm); statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL); return(302);}

    dcmTagSet(&tag, 0x0018, 0x1075); // half-life
    if(dcmAddItem(&dcm, NULL, 0, tag, DCM_VR_DS, 0xFFFFFFFF, "6586.2", verbose-1)!=0) {
      dcmfileFree(&dcm); statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL); return(303);}
    dcmTagSet(&tag, 0x0018, 0x1076); // branching
    if(dcmAddItem(&dcm, NULL, 0, tag, DCM_VR_DS, 0xFFFFFFFF, "0.967", verbose-1)!=0) {
      dcmfileFree(&dcm); statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL); return(304);}



    dcmTagSet(&tag, 0x0020, 0x0000);
    ul=0;
    if(dcmAddItem(&dcm, NULL, 0, tag, DCM_VR_UL, 0xFFFFFFFF, (char*)&ul, verbose-1)!=0) {
      dcmfileFree(&dcm); statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL); return(400);}

    dcmTagSet(&tag, 0x0020, 0x000d); // Study Instance UID
    if(dcmAddItem(&dcm, NULL, 0, tag, DCM_VR_UI, 0xFFFFFFFF, "1.2.826.0.2.332211.1.4479", verbose-1)!=0) {
      dcmfileFree(&dcm); statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL); return(401);}
    dcmTagSet(&tag, 0x0020, 0x000e); // Series Instance UID
    if(dcmAddItem(&dcm, NULL, 0, tag, DCM_VR_UI, 0xFFFFFFFF, "1.2.826.0.2.332211.1.5590.2018", verbose-1)!=0) {
      dcmfileFree(&dcm); statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL); return(402);}
    dcmTagSet(&tag, 0x0020, 0x0010); // Study ID
    if(dcmAddItem(&dcm, NULL, 0, tag, DCM_VR_SH, 0xFFFFFFFF, "1.1.0902.0402", verbose-1)!=0) {
      dcmfileFree(&dcm); statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL); return(403);}

    dcmTagSet(&tag, 0x0020, 0x0011); // series number
    if(dcmAddItem(&dcm, NULL, 0, tag, DCM_VR_IS, 0xFFFFFFFF, "1", verbose-1)!=0) {
      dcmfileFree(&dcm); statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL); return(411);}
    dcmTagSet(&tag, 0x0020, 0x0012); // Acquisition number
    if(dcmAddItem(&dcm, NULL, 0, tag, DCM_VR_IS, 0xFFFFFFFF, "170", verbose-1)!=0) {
      dcmfileFree(&dcm); statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL); return(412);}
    dcmTagSet(&tag, 0x0020, 0x0013); // Instance (image) number (sequential for each file in DICOM)
    sprintf(buf, "%d", plane);
    if(dcmAddItem(&dcm, NULL, 0, tag, DCM_VR_IS, 0xFFFFFFFF, buf, verbose-1)!=0) {
      dcmfileFree(&dcm); statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL); return(413);}
    dcmTagSet(&tag, 0x0020, 0x0032); // Image position (patient)
    sprintf(buf, "49.30032\\49.30032\\%.5f", planewidth*(plane-1));
    if(dcmAddItem(&dcm, NULL, 0, tag, DCM_VR_DS, 25, buf, verbose-1)!=0) {
      dcmfileFree(&dcm); statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL); return(421);}
    dcmTagSet(&tag, 0x0020, 0x0037); // Image orientation (patient)
    if(dcmAddItem(&dcm, NULL, 0, tag, DCM_VR_DS, 24, "1.0\\0.0\\0.0\\0.0\\1.0\\0.0", verbose-1)!=0) {
      dcmfileFree(&dcm); statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL); return(422);}


    unsigned short int us;

    dcmTagSet(&tag, 0x0028, 0x0000);
    ul=0;
    if(dcmAddItem(&dcm, NULL, 0, tag, DCM_VR_UL, 0xFFFFFFFF, (char*)&ul, verbose-1)!=0) {
      dcmfileFree(&dcm); statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL); return(500);}

    dcmTagSet(&tag, 0x0028, 0x0002); // samples per pixel
    us=1;
    if(dcmAddItem(&dcm, NULL, 0, tag, DCM_VR_US, 0xFFFFFFFF, (char*)&us, verbose-1)!=0) {
      dcmfileFree(&dcm); statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL); return(501);}
    dcmTagSet(&tag, 0x0028, 0x0010); // rows
    us=dimy;
    if(dcmAddItem(&dcm, NULL, 0, tag, DCM_VR_US, 0xFFFFFFFF, (char*)&us, verbose-1)!=0) {
      dcmfileFree(&dcm); statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL); return(502);}
    dcmTagSet(&tag, 0x0028, 0x0011); // columns
    us=dimx;
    if(dcmAddItem(&dcm, NULL, 0, tag, DCM_VR_US, 0xFFFFFFFF, (char*)&us, verbose-1)!=0) {
      dcmfileFree(&dcm); statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL); return(503);}
    dcmTagSet(&tag, 0x0028, 0x0030); // Pixel Spacing/Size (mm)
    if(dcmAddItem(&dcm, NULL, 0, tag, DCM_VR_DS, 18, "0.776383\\0.776383", verbose-1)!=0) {
      dcmfileFree(&dcm); statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL); return(504);}
    dcmTagSet(&tag, 0x0028, 0x0100);
    us=16;
    if(dcmAddItem(&dcm, NULL, 0, tag, DCM_VR_US, 0xFFFFFFFF, (char*)&us, verbose-1)!=0) {
      dcmfileFree(&dcm); statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL); return(511);}
    dcmTagSet(&tag, 0x0028, 0x0101);
    us=16;
    if(dcmAddItem(&dcm, NULL, 0, tag, DCM_VR_US, 0xFFFFFFFF, (char*)&us, verbose-1)!=0) {
      dcmfileFree(&dcm); statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL); return(512);}
    dcmTagSet(&tag, 0x0028, 0x0102);
    us=15;
    if(dcmAddItem(&dcm, NULL, 0, tag, DCM_VR_US, 0xFFFFFFFF, (char*)&us, verbose-1)!=0) {
      dcmfileFree(&dcm); statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL); return(513);}
    dcmTagSet(&tag, 0x0028, 0x0103); // pixel representation: 1=signed, 0=unsigned
    us=1;
    if(dcmAddItem(&dcm, NULL, 0, tag, DCM_VR_US, 0xFFFFFFFF, (char*)&us, verbose-1)!=0) {
      dcmfileFree(&dcm); statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL); return(514);}
    dcmTagSet(&tag, 0x0028, 0x01052); // rescale ic
    if(dcmAddItem(&dcm, NULL, 0, tag, DCM_VR_DS, 0xFFFFFFFF, "0.0", verbose-1)!=0) {
      dcmfileFree(&dcm); statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL); return(515);}
    dcmTagSet(&tag, 0x0028, 0x01053); // rescale slope
    if(dcmAddItem(&dcm, NULL, 0, tag, DCM_VR_DS, 0xFFFFFFFF, "100.0", verbose-1)!=0) {
      dcmfileFree(&dcm); statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL); return(516);}


    dcmTagSet(&tag, 0x0054, 0x0000);
    ul=0;
    if(dcmAddItem(&dcm, NULL, 0, tag, DCM_VR_UL, 0xFFFFFFFF, (char*)&ul, verbose-1)!=0) {
      dcmfileFree(&dcm); statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL); return(600);}

    dcmTagSet(&tag, 0x0054, 0x0081);
    us=dimz; // nr of slices
    if(dcmAddItem(&dcm, NULL, 0, tag, DCM_VR_US, 0xFFFFFFFF, (char*)&us, verbose-1)!=0) {
      dcmfileFree(&dcm); statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL); return(601);}
    dcmTagSet(&tag, 0x0054, 0x0101);
    us=1; // nr of frames
    if(dcmAddItem(&dcm, NULL, 0, tag, DCM_VR_US, 0xFFFFFFFF, (char*)&us, verbose-1)!=0) {
      dcmfileFree(&dcm); statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL); return(602);}
    dcmTagSet(&tag, 0x0054, 0x1000);
    if(dcmAddItem(&dcm, NULL, 0, tag, DCM_VR_CS, 14, "DYNAMIC\\IMAGE", verbose-1)!=0) {
      dcmfileFree(&dcm); statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL); return(603);}
    dcmTagSet(&tag, 0x0054, 0x1001);
    if(dcmAddItem(&dcm, NULL, 0, tag, DCM_VR_CS, 5, "BQML", verbose-1)!=0) {
      dcmfileFree(&dcm); statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL); return(604);}
    dcmTagSet(&tag, 0x0054, 0x1002);
    if(dcmAddItem(&dcm, NULL, 0, tag, DCM_VR_CS, 9, "EMISSION", verbose-1)!=0) {
      dcmfileFree(&dcm); statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL); return(605);}
    dcmTagSet(&tag, 0x0054, 0x1102); // decay correction to
    if(dcmAddItem(&dcm, NULL, 0, tag, DCM_VR_CS, 6, "START", verbose-1)!=0) {
      dcmfileFree(&dcm); statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL); return(606);}
    dcmTagSet(&tag, 0x0054, 0x01300); // frame reference time
    if(dcmAddItem(&dcm, NULL, 0, tag, DCM_VR_DS, 0xFFFFFFFF, "18000.0", verbose-1)!=0) {
      dcmfileFree(&dcm); statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL); return(607);}
    dcmTagSet(&tag, 0x0054, 0x01321); // decay correction factor
    if(dcmAddItem(&dcm, NULL, 0, tag, DCM_VR_DS, 0xFFFFFFFF, "1.0019", verbose-1)!=0) {
      dcmfileFree(&dcm); statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL); return(608);}
    dcmTagSet(&tag, 0x0054, 0x1330);
    us=plane; // image index
    if(dcmAddItem(&dcm, NULL, 0, tag, DCM_VR_US, 0xFFFFFFFF, (char*)&us, verbose-1)!=0) {
      dcmfileFree(&dcm); statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL); return(611);}


    /* Pixel data */
    dcmTagSet(&tag, 0x7FE0, 0x0000);
    ul=dimx*dimy;
    if(dcmAddItem(&dcm, NULL, 0, tag, DCM_VR_UL, 0xFFFFFFFF, (char*)&ul, verbose-1)!=0) {
      dcmfileFree(&dcm); statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL); return(600);}

    dcmTagSet(&tag, 0x7FE0, 0x0010);
    unsigned int si, xi, yi;
    short int m[dimy][dimx];
    for(yi=0; yi<dimy; yi++)
      for(xi=0; xi<dimx; xi++)
        m[yi][xi]=xi/3 + yi/2 + (plane-1)/5;
    short int pxls[dimx*dimy];
    si=0;
    for(yi=0; yi<dimy; yi++)
      for(xi=0; xi<dimx; xi++)
        pxls[si++]=m[yi][xi];
    if(dcmAddItem(&dcm, NULL, 0, tag, DCM_VR_OW, dimx*dimy*sizeof(short int), (char*)pxls, verbose-1)!=0) {
      dcmfileFree(&dcm); statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL); return(1001);}


    /*
     *  Write DCMFILE into a file
     */
    ret=dcmFileWrite(dcm.filename, &dcm, status);
    if(verbose>1) printf("  ret := %d\n", ret);
    if(ret!=0) {dcmfileFree(&dcm); return(3);}


    /* Free the contents of DCMFILE */
    dcmfileFree(&dcm);

  } // next plane

  statusSet(status, __func__, __FILE__, __LINE__, 0);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
