/** @file ecatio.c
    @brief Basic IO functions for ECAT files.
 */
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <time.h>
#include <string.h>
/*****************************************************************************/
#include "tpcextensions.h"
/*****************************************************************************/
#include "tpcecat.h"
/*****************************************************************************/

/*****************************************************************************/
/** Read specified data block from an ECAT file.
    @sa ecatVerifyMagic
    @return enum tpcerror (TPCERROR_OK when successful).
 */
int ecatReadBlock(
  /** Name of file to open and read specified data block; file is closed after reading;
      enter NULL to use the file pointer (next argument) instead. */
  const char *filename,
  /** File pointer of file, opened with fp=fopen(filename, "rb"), to read the specified data block
      from; enter NULL to open file (previous argument) locally. 
      This function leaves the file pointer to the end of data block. */ 
  FILE *fp,
  /** Block number to read; enter 0 to read the block from current file pointer position,
      if file pointer was given. */
  const unsigned int blocknumber, 
  /** Pointer to write block data in; must be allocated for at least ECATBLKSIZE bytes. */
  unsigned char *data
) {
  if(filename==NULL && fp==NULL) return(TPCERROR_FAIL);
  if(data==NULL) return(TPCERROR_FAIL);

  if(fp!=NULL) { // read from given file pointer

    /* If block number was given, then seek the block */
    if(blocknumber>0) {
      if(fseeko(fp, (off_t)(blocknumber-1)*ECATBLKSIZE, SEEK_SET) != 0) 
        return(TPCERROR_NO_DATA);
    }
    /* Read the block */
    if(fread(data, ECATBLKSIZE, 1, fp)<1) return(TPCERROR_CANNOT_READ);

  } else { // read from given file name

    /* Open file for read */
    FILE *localfp=fopen(filename, "rb");
    if(localfp==NULL) return(TPCERROR_CANNOT_OPEN);

    /* If block number was given, then seek the block */
    if(blocknumber>0) {
      if(fseeko(localfp, (off_t)(blocknumber-1)*ECATBLKSIZE, SEEK_SET) != 0)
        {fclose(localfp); return(TPCERROR_NO_DATA);}
    }
    /* Read the block */
    if(fread(data, ECATBLKSIZE, 1, localfp)<1) {fclose(localfp); return(TPCERROR_CANNOT_READ);}
    fclose(localfp);

  }

  /* That's it then */
  return(TPCERROR_OK);
}
/*****************************************************************************/

/*****************************************************************************/
/** Verify that given file (either file name or file pointer) appears to be
    ECAT file, based on the magic number.

    ECAT 7 has defined magic numbers, which are checked.
    ECAT 6 format does not a defined magic number, therefore here the file is only checked to
    not indicate anything else than ECAT 6.

    @return Returns 0 if magic number is definitely not ECAT; 7 if ECAT 7 magic number 
     is found, and 6 if magic number does not indicate anything other than ECAT 6.
    @author Vesa Oikonen
 */
int ecatVerifyMagic(
  /** Name of file to open and to verify for the magic number; file is closed after reading;
      enter NULL to use the file pointer (next argument) instead. */
  const char *filename,
  /** File pointer of file to check, opened with fp=fopen(filename, "rb"); 
      enter NULL to open file (previous argument) locally. 
      This function rewinds the file pointer to start before and after reading the data.
  */ 
  FILE *fp
) {

  /* Read the first data block (512 bytes) */
  unsigned char buf[ECATBLKSIZE];
  int ret=ecatReadBlock(filename, fp, 0, buf);
  if(fp!=NULL) rewind(fp);
  if(ret!=TPCERROR_OK) return(0);

  /* Check if the start matches the ECAT 7 magic numbers */
  if(strncmp((char*)buf, "MATRIX72v", 9)==0 || strncmp((char*)buf, "MATRIX7011", 10)==0) return(7);

  /* ECAT 6 does not have defined magic numbers, but check anyway */
  if(strncmp((char*)buf, "MATRIX6", 7)==0 || strncmp((char*)buf, "ECAT6", 5)==0) return(6);

  /* Check that we do NOT have DICOM magic number */
  if(strncmp((char*)buf+128, "DICM", 4)==0) return(0);

  /* Check that we do NOT have NIfTI magic number */
  if(strncmp((char*)buf+344, "ni1", 3)==0 || strncmp((char*)buf, "n+1", 3)==0) return(0);

  /* Check that we do NOT have Interfile magic number */
  if(strncmp((char*)buf, "!INTERFILE", 10)==0) return(0);

  /* There is no indication that this IS NOT an ECAT 6 file */
  return(6);
}
/*****************************************************************************/

/*****************************************************************************/
