/** @file decpoint.c
    @brief Decimal point functions.
    @details Decimal point in data files and user-specified arguments can be
    either dot (UK/USA) or comma (EU), independent on what is the defined
    locale on the current platform. Therefore we usually need to try to
    correctly read decimal numbers whether they are stored in either format,
    and to convert decimal dots to commas and vice versa.
    This file contains functions used for that purpose.
    @author Vesa Oikonen
 */
/*****************************************************************************/

/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <string.h>
#include <ctype.h>
#include <unistd.h>
#include <math.h>
#include "tpcextensions.h"
/*****************************************************************************/

/*****************************************************************************/
/** Verifies that given string seems like a valid representation of integer
    or floating point number in decimal or exponential format.
    @return 1 if string is valid number, and 0 if not.
    @author Vesa Oikonen
    @sa atofVerified, atoiCheck, tpcYesNo
 */
int strIsValidNumber(
  /** Pointer to the string to be verified; it must not contain any leading
      or trailing space characters etc, or it will never verify. */
  const char *s
) {
  if(s==NULL || strnlen(s, 256)<1) return 0;

  char *p=(char*)s;
  int i, nn=0;
  /* It can start with + or minus */
  if(*p=='+' || *p=='-') p++;
  /* Then jump over all digits, counting them */
  i=strspn(p, "0123456789"); nn+=i; p+=i;
  if(!*p) {if(nn>0) return 1; else return 0;} // end of string
  /* Then there can be a decimal point or comma, but only one per string */
  if(*p=='.' || *p==',') p++;
  if(!*p) {if(nn>0) return 1; else return 0;} // end of string
  /* Again jump over all digits, counting them */
  i=strspn(p, "0123456789"); nn+=i; p+=i;
  /* At this point we must have got at least one digit */
  if(nn<1) return 0;
  if(!*p) return 1; // end of string
  /* String continues, the next character must be E or e */
  if(*p=='E' || *p=='e') p++; else return 0;
  /* next can be + or - but not necessarily */
  if(*p=='+' || *p=='-') p++;
  /* then we must have at least one digit */
  i=strspn(p, "0123456789"); if(i<1) return 0;
  nn+=i; p+=i;
  /* now we must be at the end of string */
  if(!*p) return 1; else return 0; 
}
/*****************************************************************************/

/*****************************************************************************/
/** Version of atof() which verifies that argument string at least seems like a valid number. 

    Both decimal point and comma are accepted.
    Result value is set to NaN if string was not valid value.
    @sa atofCheck, doubleGetWithUnit, atoiCheck, strIsValidNumber, tpcYesNo
    @return converted double value, or NaN in case of an error.
 */
double atofVerified(
  /** String which is converted to a double; string must not contain any space characters. */
  const char *s
) {
  if(s==NULL || !strIsValidNumber(s)) return nan("");
  char *p; p=strchr(s, ','); if(p==NULL) return atof(s);
  char *s2=strdup(s); p=strchr(s2, ','); *p='.';
  double t=atof(s2); free(s2); return t;
}
/*****************************************************************************/

/*****************************************************************************/
/** Verifies that argument string at least seems like a valid number. 

    Both decimal point and comma are accepted.
    Optional result double value is set to NaN if string was not valid value.
    @sa atofVerified, atoiCheck, doubleGetWithUnit, atofList, strCleanSpaces, strIsValidNumber
    @return 0 if successful, and 1 in case of an error.
 */
int atofCheck(
  /** String which is converted to a double; string must not contain any space characters. */
  const char *s,
  /** Pointer to the double float; enter NULL, if not needed. */
  double *v
) {
  if(v!=NULL) *v=nan(""); 
  if(s==NULL) return 1;
  if(!strIsValidNumber(s)) return 1;
  char *p; p=strchr(s, ','); if(p==NULL) {*v=atof(s); return 0;}
  char *s2=strdup(s); p=strchr(s2, ','); *p='.';
  *v=atof(s2); free(s2); return 0;
}
/*****************************************************************************/

/*****************************************************************************/
/** Checks whether argument string contains a decimal comma instead of dot.
    @author Vesa Oikonen
    @return 1 if decimal comma is found and 0 if not.
    @sa strHaveDecimalSeparator, atofVerified, strIsValidNumber
 */
int strHaveDecimalComma(
  /** Pointer to string. */
  const char *s
) {
  if(s==NULL) return(0);
  if(strchr(s, '.')!=NULL) return(0);
  if(strchr(s, ',')!=NULL) return(1);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Checks whether argument string contains a decimal comma or dot, or neither.
    @author Vesa Oikonen
    @return 0, if neither is found, 1 if dot, and 2 if comma is found.
    @sa strHaveDecimalComma, atofVerified, strIsValidNumber
 */
int strHaveDecimalSeparator(
  /** Pointer to string. */
  const char *s
) {
  if(s==NULL) return(0);
  if(strchr(s, '.')!=NULL) return(1);
  if(strchr(s, ',')!=NULL) return(2);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Read a list of double values from given string with given delimiters.
    @return The number of double values, or <0 in case of an error.
    @author Vesa Oikonen
    @sa strTokenNr, strTokenNCpy, intlistAddFromString, atofCheck, tpcYesNo, doubleCopy,
        doubleSpanPositives
 */
int atofList(
  /** Pointer to string from which double values are read. */
  const char *s1,
  /** String containing character delimiters */
  const char *s2,
  /** Pointer to a pre-allocated array of doubles. */
  double *x,
  /** Size of double list; obligatory. If string contains more values,
      only maxn values will be read. */
  int maxn
) {
  if(x==NULL || maxn<=0) return(-1);
  for(int i=0; i<maxn; i++) x[i]=nan("");
  if(s1==NULL || s2==NULL) return(0);
  
  /* Get the nr of tokens */
  int i, n;
  n=strTokenNr((char*)s1, s2); if(n<1) return(0);
  /* Read the values */
  char tmp[256];
  double v; 
  for(i=0; i<n && i<maxn; i++) {
    if(strTokenNCpy(s1, s2, 1+i, tmp, 256)<1) return(-2);
    v=atofVerified(tmp); if(isnan(v)) return(-3);
    x[i]=v;
  }
  return(i);
}
/*****************************************************************************/

/*****************************************************************************/
