/** @file endian.c
    @brief Byte swapping between little and big endian.
    @author Vesa Oikonen
 */
/*****************************************************************************/

/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <string.h>
#include <ctype.h>
#include <unistd.h>
#include <math.h>
#include "tpcextensions.h"
/*****************************************************************************/

/*****************************************************************************/
/** Detect endianness at run-time.
    From https://sourceforge.net/p/predef/wiki/Endianness/
    @return Returns ENDIAN_UNKNOWN, ENDIAN_BIG, ENDIAN_LITTLE, ENDIAN_BIG_WORD,
    or ENDIAN_LITTLE_WORD.
 */
int endianness()
{
  union {
    uint32_t value;
    uint8_t data[sizeof(uint32_t)];
  } number;

  number.data[0] = 0x00;
  number.data[1] = 0x01;
  number.data[2] = 0x02;
  number.data[3] = 0x03;

  switch (number.value) {
    case UINT32_C(0x00010203): return ENDIAN_BIG;
    case UINT32_C(0x03020100): return ENDIAN_LITTLE;
    case UINT32_C(0x02030001): return ENDIAN_BIG_WORD;
    case UINT32_C(0x01000302): return ENDIAN_LITTLE_WORD;
    default:                   return ENDIAN_UNKNOWN;
  }
}
/*****************************************************************************/

/*****************************************************************************/
/** Check whether current platform uses little endian byte order.
   @sa swap, swabip, swawbip, swawip
   @return Returns 1, if current platform is little endian, and 0 if not.
 */
int endianLittle()
{
  //int x=1;
  //if(*(char *)&x==1) return(1); else return(0);
  if(endianness()==ENDIAN_LITTLE) return(1); else return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Swap the specified short int, int, long int, float, or double
    from little endian to big endian or vice versa.

    Arguments are allowed to overlap.

   @sa swawbip, swabip, swawip, endianLittle
 */
void swap(
  /** Pointer to a short int, int, long int, float, or double variable. */
  void *from,
  /** Pointer to a short int, int, long int, float, or double variable. */
  void *to,
  /** Size of from and to (byte nr) must be 1, 2, 4 or 8. */
  int size
) {
  if(from==NULL || to==NULL) return;

  unsigned char c;
  unsigned short int s;
  unsigned long l;
  
  switch(size) {
    case 1:
      *(char *)to=*(char *)from;
      break;
    case 2:
      c=*(unsigned char *)from;
      *(unsigned char *)to = *((unsigned char *)from+1);
      *((unsigned char *)to+1) = c; 
      /*swab(from, to, size); // NOT ANSI */
      break;
    case 4:
      s=*(unsigned short *)from;
      *(unsigned short *)to = *((unsigned short *)from+1);
      *((unsigned short *)to+1) = s;
      swap((char*)to, (char*)to, 2);
      swap((char*)((unsigned short *)to+1), (char*)((unsigned short *)to+1), 2);
      break;
    case 8:
      l=*(unsigned long *)from;
      *(unsigned long *)to = *((unsigned long *)from+1);
      *((unsigned long *)to+1) = l;
      swap((char *)to, (char *)to, 4);
      swap((char*)((unsigned long *)to+1), (char*)((unsigned long *)to+1), 4);
      break;
  }
}
/*****************************************************************************/

/*****************************************************************************/
/** In-place swab, replaces the non-ANSI function swab(), which may not work if data is overlapping.
   @sa swawbip, swawip, swap, endianLittle
 */
void swabip(
  /** Pointer to memory */
  void *buf,
  /** Size of buf in bytes */
  int size
) {
  if(buf==NULL) return;

  int i;
  unsigned char c;

  for(i=1; i<size; i+=2) {
    c=*((unsigned char *)buf+i);
    *((unsigned char *)buf+i)=*((unsigned char *)buf+(i-1));
    *((unsigned char *)buf+(i-1))=c;
  }
}
/*****************************************************************************/

/*****************************************************************************/
/** In-place swab and swaw, switches words and bytes from an array of 4-byte ints or floats.
   @sa swawip, swabip, swap, endianLittle
 */
void swawbip(
  /** Pointer to memory */
  void *buf,
  /** Size of buf in bytes */
  int size
) {
  if(buf==NULL) return;

  int i;
  unsigned char c, *cptr;

  cptr=(unsigned char*)buf;
  for(i=0; i<size; i+=4, cptr+=4) {
    c=cptr[0]; cptr[0]=cptr[3]; cptr[3]=c;
    c=cptr[1]; cptr[1]=cptr[2]; cptr[2]=c;
  }
}
/*****************************************************************************/

/*****************************************************************************/
/** In-place swaw, switches words (but not bytes) from an array of 4-byte ints or floats.
   @sa swawbip, swabip, swap, endianLittle
 */
void swawip(
  /** Pointer to memory */
  void *buf,
  /** Size of buf in bytes */
  int size
) {
  if(buf==NULL) return;

  int i;
  unsigned short int s, *sptr;

  sptr=(unsigned short int*)buf;
  for(i=0; i<size; i+=4, sptr+=2) {
    s=sptr[0]; sptr[0]=sptr[1]; sptr[1]=s;
  }
}
/*****************************************************************************/

/*****************************************************************************/
/** In-place big-little endian swapping of an array of 64 bit (8 byte)
    integers or floating point values.
   @sa swap32ip, swap16ip, endianLittle
 */
void swap64ip(
  /** Pointer to memory. */
  void *buf,
  /** Nr of 64 bit values in the array. */
  unsigned long long size
) {
  if(buf==NULL || size==0) return;

  unsigned long long int i;
  unsigned char c, *cptr;

  cptr=(unsigned char*)buf;
  for(i=0; i<size; i++, cptr+=8) {
    c=cptr[0]; cptr[0]=cptr[7]; cptr[7]=c;
    c=cptr[1]; cptr[1]=cptr[6]; cptr[6]=c;
    c=cptr[2]; cptr[2]=cptr[5]; cptr[5]=c;
    c=cptr[3]; cptr[3]=cptr[4]; cptr[4]=c;
  }
}
/*****************************************************************************/

/*****************************************************************************/
/** In-place big-little endian swapping of an array of 32 bit (4 byte)
    integers or floating point values.
   @sa swap16ip, swap64ip, endianLittle
 */
void swap32ip(
  /** Pointer to memory. */
  void *buf,
  /** Nr of 32 bit values in the array. */
  unsigned long long size
) {
  if(buf==NULL || size==0) return;

  unsigned long long int i;
  unsigned char c, *cptr;

  cptr=(unsigned char*)buf;
  for(i=0; i<size; i++, cptr+=4) {
    c=cptr[0]; cptr[0]=cptr[3]; cptr[3]=c;
    c=cptr[1]; cptr[1]=cptr[2]; cptr[2]=c;
  }
}
/*****************************************************************************/

/*****************************************************************************/
/** In-place big-little endian swapping of an array of 16 bit (2 byte)
    integers.
   @sa swap32ip, swap64ip, endianLittle
 */
void swap16ip(
  /** Pointer to memory. */
  void *buf,
  /** Nr of 16 bit values in the array. */
  unsigned long long size
) {
  if(buf==NULL || size==0) return;

  unsigned long long int i;
  unsigned char c, *cptr;

  cptr=(unsigned char*)buf;
  for(i=0; i<size; i++, cptr+=2) {
    c=cptr[0]; cptr[0]=cptr[1]; cptr[1]=c;
  }
}
/*****************************************************************************/

/*****************************************************************************/
