/** @file roiname.c
 *  @brief Functions for ROI names.
 */
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "tpcextensions.h"
/*****************************************************************************/

/*****************************************************************************/
/** Divide ROI name into its sub-parts, based on user-defined delimiter characters 
    (usually underscore and/or space).
    @author Vesa Oikonen
    @return Returns pointer to the required sub-part, or NULL if not found.
    @sa roinameEditByTemplate, roinameAddField, roinameExists, roinameMatch
 */
char *roinameSubpart(
  /** Pointer to ROI name. Not modified. */
  const char *roiname,
  /** String containing the sub-part delimiters. For example, " _". */
  const char *dlm,
  /** Index [0..n-1] of sub-part to search for. */
  const unsigned int si,
  /** Pointer to allocated string for sub-part. */
  char *subpart,
  /** Length of the sub-part string, including terminating null. 
      Resulting sub-part length will be at maximum slen-1. */
  const unsigned int slen
) {
  if(subpart==NULL || slen<1) return(subpart);
  subpart[0]=(char)0;
  if(roiname==NULL || strnlen(roiname, 1)==0) return((char*)NULL);
  if(dlm==NULL || strnlen(dlm, 1)==0) return((char*)NULL);

  unsigned int i=0;
  char *s=strdup(roiname);
  char *sp=s; //printf(" si=%d\n", si);
  do {
    //printf("  sp='%s'\n", sp);
    char *p=strpbrk(sp, dlm);
    //printf("  i=%d  p='%s'\n", i, p);
    if(!p) {
      if(si==i) {strlcat(subpart, sp, slen); free(s); return(subpart);}
      free(s); return((char*)NULL);
    }
    *p=(char)0;
    if(si==i) {strlcat(subpart, sp, slen); free(s); return(subpart);}
    sp=p+1; i++;
  } while(*sp);  

  if(si==i) {strlcat(subpart, sp, slen); free(s); return(subpart);}
  free(s);
  return((char*)NULL);
}
/*****************************************************************************/

/*****************************************************************************/
/** Make a new ROI name based on template and old ROI name.
    @return Returns pointer to the new ROI name, or NULL in case of an error.
    @sa roinameSubpart, roinameAddField, roinameExists, roinameMatch
 */ 
char *roinameEditByTemplate(
  /** Template ROI name, where '_' is used to separate sub-parts, and '@' is replaced by 
      the corresponding sub-part from the 'currname'. */  
  const char *template,
  /** Current ROI name, from which sub-parts will be added as replacement of '@' in the template.
      Space character and '_' are assumed to separate the sub-parts. */
  const char *currname,
  /** Pointer to pre-allocated string of length count, into which the new ROI name will be written. */  
  char *newname,
  /** Max length of new name, including terminal null. */
  const unsigned int count 
) {
  if(newname==NULL || count<1) return((char*)NULL);
  newname[0]=(char)0;
  if(template==NULL || currname==NULL) return((char*)NULL);
  
  unsigned int i, len;
  len=strlen(template); i=strlen(currname); if(i>len) len=i;
  if(len<1) return(newname);
  char subpart[++len], *c;
  
  i=0;
  c=roinameSubpart(template, "_", i, subpart, len);
  while(c) {
    /* If not '@' or if '@' can be replaced with sub-part from original tacname, then copy */
    if(strcmp(subpart, "@")!=0 || roinameSubpart(currname, " _", i, subpart, 256)) {
      if(i>0) strlcat(newname, "_", count);
      strlcat(newname, subpart, count);
    }
    c=roinameSubpart(template, "_", ++i, subpart, 256);
  }
  return(newname);
}
/*****************************************************************************/

/*****************************************************************************/
/** Add a new field into given position inside an existing ROI name.
    Space character and '_' are accepted as field separators.
    @return Returns pointer to the edited ROI name, or NULL in case of an error.
    @sa roinameSubpart, roinameEditByTemplate, roinameExists, roinameMatch, strdelstr
 */
char *roinameAddField(
  /** ROI name string which will be edited. */
  char *roiname, 
  /** String which will be added as a new field in roiname; can be empty in order to add empty 
      field, but NULL is not accepted. */
  const char *field, 
  /** Field index [0..n] for which position new field will be added; 
      enter a large number to add field to the end. */
  const unsigned int in,
  /** Max length of the roiname, including space for terminal null. 
      Function returns NULL (error) in case there is not enough space. */
  const unsigned int count
) {
  if(roiname==NULL) return(roiname);
  if(field==NULL) return((char*)NULL);
  unsigned int slen=strlen(roiname);
  unsigned int flen=strlen(field);
  if(count < (slen+flen+2)) return((char*)NULL);
  if(slen==0) {
    if(flen==0) {strlcpy(roiname, "_", count); return(roiname);}
    else {strlcpy(roiname, field, count); return(roiname);}
  }

  char *c, subpart[slen+1], tmp[slen+1];
  unsigned int i=0;
  strcpy(tmp, roiname); strcpy(roiname, "");
  c=roinameSubpart(tmp, " _", i, subpart, slen+1);
  while(c) {
    if(i==in) {
      if(i>0) strlcat(roiname, "_", count);
      strlcat(roiname, field, count);
    }  
    if(i>0 || i==in) strlcat(roiname, "_", count);
    strlcat(roiname, subpart, count);
    c=roinameSubpart(tmp, " _", ++i, subpart, slen+1);
  }
  if(i<=in) {
    if(i>0) strlcat(roiname, "_", count);
    strlcat(roiname, field, count);
  }
  return(roiname);
}  
/*****************************************************************************/

/*****************************************************************************/
/** @brief Verifies whether TAC name exists or not.

    TAC name string may contain only delimiters like '.', '_', '-', or spaces.
    Those cases are interpreted as no name in this function.
    @return 1 if valid name exists, 0 if not.
    @sa roinameSubpart, roinameEditByTemplate, roinameAddField, roinameMatch
 */
int roinameExists(
  /** ROI name string; not edited. */
  char *roiname
) {
  if(roiname==NULL) return(0);
  size_t len, n;
  len=strlen(roiname); if(len==0) return(0);
  n=strspn(roiname, " _-.\t"); if(n==len) return(0);
  return(1);
}
/*****************************************************************************/

/*****************************************************************************/
/** Test whether ROI (TAC) name matches with a test string.

    Test string can contain wild cards. 
    If both names contain ROI name separators (' ', '_', or '-'), 
    the corresponding sub-names are tested against each other.
    If test_str does not contain separators but ROI name does, then names are considered matching if 
    test_str matches with any of sub-names.
    @return 1, in case of match, or 0 if not matched.
    @author Vesa Oikonen
    @sa roinameSubpart, roinameEditByTemplate, roinameAddField, roinameExists, fnmatch, 
        strstrNoQuotation
 */
int roinameMatch(
  /** Pointer to ROI name to be tested. */
  const char *roiname,
  /** Test string. */
  const char *test_str,
  /** Pointer to status data; enter NULL if not needed. */
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  if(verbose>0) printf("%s()\n", __func__);
  statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  /* If both string are empty, then that is a match */
  if((roiname==NULL || strlen(roiname)<1) && (test_str==NULL || strlen(test_str)<1)) {
    return(1);
  } 
  /* If just either of those is empty, then that is a no match */
  if(roiname==NULL || strlen(roiname)<1) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_DATA);
    return(0);
  }
  if(test_str==NULL || strlen(test_str)<1) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_DATA);
    return(0);
  }

  /* Check the number of name sub-parts */
  int i, len;
  len=strlen(roiname); i=strlen(test_str); if(i>len) len=i;
  char subn1[len+1], subn2[len+1];
  int roisubnr=0, refsubnr=0;
  for(i=0; i<6; i++) {
    if(roinameSubpart(roiname, " _-", i, subn1, len+1)==NULL) break; 
    roisubnr++;
  }
  for(i=0; i<6; i++) {
    if(roinameSubpart(test_str, " _-", i, subn2, len+1)==NULL) break; 
    refsubnr++;
  }
  if(verbose>3) printf("roisubnr := %d\nrefsubnr := %d\n", roisubnr, refsubnr);
  if(roisubnr==0 && refsubnr==0) return(1);
  if(roisubnr==0 || refsubnr==0) return(0);
  if(refsubnr>1 && roisubnr!=refsubnr) return(0);
  
  /* If more than one sub-name to test for, then test against corresponding sub-names with 
     wild cards */
  if(refsubnr>1) {
    for(i=0; i<refsubnr; i++) {
      roinameSubpart(roiname, " _-", i, subn1, len+1); 
      if(!strcmp(subn1, ".")) strcpy(subn1, "");
      roinameSubpart(test_str, " _-", i, subn2, len+1);
      if(!strcmp(subn2, ".")) strcpy(subn2, "");
      if(verbose>3) printf(" '%s' vs '%s'\n", subn1, subn2);
      if(strnlen(subn1, 1)==0 && strnlen(subn2, 1)==0) continue;
      if(strnlen(subn2, 1)<1 || fncasematch(subn1, subn2)==0 ) return(0);
    }
    return(1);
  }

  /* So, we have just one sub-name in the test string */
  /* If it is found in any of sub names, accepting wild cards, then we consider it a match */
  if(roinameSubpart(test_str, " _-", 0, subn2, len+1)==NULL) return(0);
  if(!strcmp(subn2, ".")) strcpy(subn2, "");
  for(i=0; i<roisubnr; i++) {
    if(roinameSubpart(roiname, " _-", i, subn1, len+1)==NULL) continue;
    if(!strcmp(subn1, ".")) strcpy(subn1, "");
    if(fncasematch(subn1, subn2)) return(1);
  }
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
