/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include "tpcextensions.h"
#include "test_tpcextensions.h"
/*****************************************************************************/

/*****************************************************************************/
int test_doubleMatch(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }
  double v1, v2, lim;
  int ret;

  v1=v2=lim=nan("");
  if(verbose>1) printf("%g vs %g within %g\n", v1, v2, lim);
  ret=doubleMatch(v1, v2, lim); if(verbose>2) printf("  -> %d\n", ret);
  if(ret!=1) return 1;

  v1=0; v2=lim=nan("");
  if(verbose>1) printf("%g vs %g within %g\n", v1, v2, lim);
  ret=doubleMatch(v1, v2, lim); if(verbose>2) printf("  -> %d\n", ret);
  if(ret!=0) return 2;

  v1=0; v2=0; lim=nan("");
  if(verbose>1) printf("%g vs %g within %g\n", v1, v2, lim);
  ret=doubleMatch(v1, v2, lim); if(verbose>2) printf("  -> %d\n", ret);
  if(ret!=1) return 3;

  v1=1.001; v2=1.002; lim=0.002;
  if(verbose>1) printf("%g vs %g within %g\n", v1, v2, lim);
  ret=doubleMatch(v1, v2, lim); if(verbose>2) printf("  -> %d\n", ret);
  if(ret!=1) return 4;

  v1=1.01; v2=1.02; lim=0.002;
  if(verbose>1) printf("%g vs %g within %g\n", v1, v2, lim);
  ret=doubleMatch(v1, v2, lim); if(verbose>2) printf("  -> %d\n", ret);
  if(ret!=0) return 5;

  v1=-1.01; v2=-1.02; lim=0.002;
  if(verbose>1) printf("%g vs %g within %g\n", v1, v2, lim);
  ret=doubleMatch(v1, v2, lim); if(verbose>2) printf("  -> %d\n", ret);
  if(ret!=0) return 6;

  v1=-1.001; v2=-1.002; lim=0.002;
  if(verbose>1) printf("%g vs %g within %g\n", v1, v2, lim);
  ret=doubleMatch(v1, v2, lim); if(verbose>2) printf("  -> %d\n", ret);
  if(ret!=1) return 7;

  v1=-1.001; v2=-1.002; lim=0;
  if(verbose>1) printf("%g vs %g within %g\n", v1, v2, lim);
  ret=doubleMatch(v1, v2, lim); if(verbose>2) printf("  -> %d\n", ret);
  if(ret!=0) return 8;

  statusSet(status, __func__, __FILE__, __LINE__, 0);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
int test_doubleArrayMatch(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }
  unsigned int i, n=10;
  double a[n], b[n], lim;
  int ret;

  for(i=0; i<n; i++) a[i]=b[i]=(double)i;
  lim=0.0;
  ret=doubleArrayMatch(a, b, n, lim);
  if(ret!=1) return 1;
  a[1]=nan("");
  ret=doubleArrayMatch(a, b, n, lim);
  if(ret!=0) return 2;
  b[1]=nan("");
  ret=doubleArrayMatch(a, b, n, lim);
  if(ret!=1) return 3;

  for(i=0; i<n; i++) b[i]+=1.0E-06;
  lim=0.0;
  ret=doubleArrayMatch(a, b, n, lim);
  if(ret!=0) return 4;
  lim=1.0E-05;
  ret=doubleArrayMatch(a, b, n, lim);
  if(ret!=1) return 5;

  statusSet(status, __func__, __FILE__, __LINE__, 0);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
int test_doubleMatchRel(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }
  double v1, v2, lim;
  int ret;

  v1=v2=lim=nan("");
  if(verbose>1) printf("%g vs %g within %g\n", v1, v2, lim);
  ret=doubleMatchRel(v1, v2, lim); if(verbose>2) printf("  -> %d\n", ret);
  if(ret!=1) return 1;

  v1=0; v2=lim=nan("");
  if(verbose>1) printf("%g vs %g within %g\n", v1, v2, lim);
  ret=doubleMatchRel(v1, v2, lim); if(verbose>2) printf("  -> %d\n", ret);
  if(ret!=0) return 2;

  v1=0; v2=0; lim=nan("");
  if(verbose>1) printf("%g vs %g within %g\n", v1, v2, lim);
  ret=doubleMatchRel(v1, v2, lim); if(verbose>2) printf("  -> %d\n", ret);
  if(ret!=1) return 3;

  v1=1.001; v2=1.002; lim=0.002;
  if(verbose>1) printf("%g vs %g within %g\n", v1, v2, lim);
  ret=doubleMatchRel(v1, v2, lim); if(verbose>2) printf("  -> %d\n", ret);
  if(ret!=1) return 4;

  v1=1.01; v2=1.02; lim=0.002;
  if(verbose>1) printf("%g vs %g within %g\n", v1, v2, lim);
  ret=doubleMatchRel(v1, v2, lim); if(verbose>2) printf("  -> %d\n", ret);
  if(ret!=0) return 5;

  v1=-1.01; v2=-1.02; lim=0.002;
  if(verbose>1) printf("%g vs %g within %g\n", v1, v2, lim);
  ret=doubleMatchRel(v1, v2, lim); if(verbose>2) printf("  -> %d\n", ret);
  if(ret!=0) return 6;

  v1=-1.001; v2=-1.002; lim=0.002;
  if(verbose>1) printf("%g vs %g within %g\n", v1, v2, lim);
  ret=doubleMatchRel(v1, v2, lim); if(verbose>2) printf("  -> %d\n", ret);
  if(ret!=1) return 7;

  v1=-1.001; v2=-1.002; lim=0;
  if(verbose>1) printf("%g vs %g within %g\n", v1, v2, lim);
  ret=doubleMatchRel(v1, v2, lim); if(verbose>2) printf("  -> %d\n", ret);
  if(ret!=0) return 8;

  v1=0.0; v2=0.0; lim=0.001;
  if(verbose>1) printf("%g vs %g within %g\n", v1, v2, lim);
  ret=doubleMatchRel(v1, v2, lim); if(verbose>2) printf("  -> %d\n", ret);
  if(ret!=1) return 9;

  v1=-1.0; v2=1.0; lim=0.001;
  if(verbose>1) printf("%g vs %g within %g\n", v1, v2, lim);
  ret=doubleMatchRel(v1, v2, lim); if(verbose>2) printf("  -> %d\n", ret);
  if(ret!=0) return 10;

  statusSet(status, __func__, __FILE__, __LINE__, 0);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
int test_doubleMachEps(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }

  double v=doubleMachEps();
  if(verbose>1) printf("MachEps := %e\n", v);
  if(!isnormal(v)) return(1);
  if(!isfinite(v)) return(2);
  if(v<=0.0) return(3);

  statusSet(status, __func__, __FILE__, __LINE__, 0);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
int test_doubleCopy(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }
  double t[10], s[10];
  unsigned int n, i;

  if(verbose>1) printf("NULL data\n");
  doubleCopy(NULL, NULL, 10); 
  if(verbose>2) printf("  -> Good, not crashed\n");

  if(verbose>1) printf("Ok data\n");
  n=10; for(i=0; i<n; i++) {s[i]=(double)(1+i); t[i]=0.0;}
  doubleCopy(t, s, 10);
  for(i=0; i<n; i++) if(!doubleMatch(s[i], t[i], 1.0E-08)) return 10;

  statusSet(status, __func__, __FILE__, __LINE__, 0);
  return 0;
}
/*****************************************************************************/

/*****************************************************************************/
int test_doubleCopyFinite(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }
  double t[10], s[10];
  unsigned int n, i;

  if(verbose>1) printf("NULL data\n");
  if(doubleCopyFinite(NULL, NULL, 10)!=0) return(1);
  if(verbose>2) printf("  -> Good, not crashed\n");

  if(verbose>1) printf("Ok data\n");
  n=10; for(i=0; i<n; i++) {s[i]=(double)(1+i); t[i]=0.0;}
  doubleCopyFinite(t, s, 10);
  for(i=0; i<n; i++) if(!doubleMatch(s[i], t[i], 1.0E-08)) return 10;

  if(verbose>1) printf("all NaNs in data\n");
  n=10; for(i=0; i<n; i++) {s[i]=nan(""); t[i]=0.0;}
  if(doubleCopyFinite(t, s, 10)!=0) return(20);

  if(verbose>1) printf("not all NaNs in data\n");
  s[8]=123.0;
  if(doubleCopyFinite(t, s, 10)!=1) return(31);
  if(t[0]!=s[8]) return(32);

  statusSet(status, __func__, __FILE__, __LINE__, 0);
  return 0;
}
/*****************************************************************************/

/*****************************************************************************/
int test_doubleNaNs(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }
  double a[10];
  unsigned int n, i;

  if(verbose>1) printf("NULL data\n");
  if(doubleNaNs(NULL, 0)!=0) return(1);
  if(verbose>2) printf("  -> Good, not crashed\n");
  if(doubleNaNs(NULL, 10)!=0) return(2);
  if(verbose>2) printf("  -> Good, not crashed\n");

  if(verbose>1) printf("Ok data\n");
  n=10; for(i=0; i<n; i++) a[i]=(double)(1+i);
  if(doubleNaNs(a, n)!=0) return(11);

  if(verbose>1) printf("all NaNs in data\n");
  n=10; for(i=0; i<n; i++) a[i]=nan("");
  if(doubleNaNs(a, n)!=n) return(12);

  if(verbose>1) printf("not all NaNs in data\n");
  a[8]=123.0;
  if(doubleNaNs(a, n)!=n-1) return(31);

  statusSet(status, __func__, __FILE__, __LINE__, 0);
  return 0;
}
/*****************************************************************************/

/*****************************************************************************/
int test_doubleRange(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }
  double a[10], amin, amax;
  unsigned int n, i;

  if(verbose>1) printf("NULL data\n");
  if(doubleRange(NULL, 0, NULL, NULL)!=0) return(1);
  if(verbose>2) printf("  -> Good, not crashed\n");
  if(doubleRange(NULL, 10, &amin, &amax)!=0) return(2);
  if(verbose>2) printf("  -> Good, not crashed\n");
  if(!isnan(amin) || !isnan(amax)) return(3);
  if(doubleRange(a, 0, &amin, &amax)!=0) return(4);
  if(verbose>2) printf("  -> Good, not crashed\n");
  if(!isnan(amin) || !isnan(amax)) return(5);

  if(verbose>1) printf("Ok data with no NaNs\n");
  n=10; for(i=0; i<n; i++) a[i]=(double)(1+i);
  if(doubleRange(a, n, &amin, &amax)!=n) return(11);
  if(amin!=a[0] || amax!=a[n-1]) return(12);
  if(doubleRange(a, n, NULL, NULL)!=n) return(13);

  if(verbose>1) printf("all NaNs in data\n");
  n=10; for(i=0; i<n; i++) a[i]=nan("");
  if(doubleRange(a, n, &amin, &amax)!=0) return(21);
  if(!isnan(amin) || !isnan(amax)) return(22);

  if(verbose>1) printf("not all NaNs in data\n");
  a[8]=123.0;
  if(doubleRange(a, n, &amin, &amax)!=1) return(31);
  if(amin!=a[8] || amax!=a[8]) return(32);
  a[1]=0.0;
  if(doubleRange(a, n, &amin, &amax)!=2) return(33);
  if(amin!=a[1] || amax!=a[8]) return(34);

  statusSet(status, __func__, __FILE__, __LINE__, 0);
  return 0;
}
/*****************************************************************************/

/*****************************************************************************/
int test_doubleSum(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }

  double v;

  if(verbose>1) printf("NULL data\n");
  v=doubleSum(NULL, 0);
  if(verbose>2) printf("  -> Good, not crashed\n");
  if(!doubleMatch(v, 0.0, doubleMachEps())) return(1);
  v=doubleSum(NULL, 10);
  if(verbose>2) printf("  -> Good, not crashed\n");
  if(!doubleMatch(v, 0.0, doubleMachEps())) return(2);

  double a[]={3,2,1,0,1,2,3};

  if(verbose>1) printf("Decent data\n");
  v=doubleSum(a, 7);
  if(verbose>2) printf("  v := %g\n", v);
  if(!doubleMatch(v, 12.0, doubleMachEps())) return(11);
  v=doubleSum(a+1, 6);
  if(verbose>2) printf("  v := %g\n", v);
  if(!doubleMatch(v, 9.0, doubleMachEps())) return(12);

  if(verbose>1) printf("Data with NaN\n");
  a[0]=nan("");
  v=doubleSum(a, 7);
  if(verbose>2) printf("  v := %g\n", v);
  if(!doubleMatch(v, 9.0, doubleMachEps())) return(21);

  statusSet(status, __func__, __FILE__, __LINE__, 0);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
int test_doubleMean(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }

  double v;

  if(verbose>1) printf("NULL data\n");
  v=doubleMean(NULL, 0);
  if(verbose>2) printf("  -> Good, not crashed\n");
  if(!isnan(v)) return(1);
  v=doubleMean(NULL, 10);
  if(verbose>2) printf("  -> Good, not crashed\n");
  if(!isnan(v)) return(2);

  double a[]={3,2,1,1,2,3};

  if(verbose>1) printf("Decent data\n");
  v=doubleMean(a, 6);
  if(verbose>2) printf("  v := %g\n", v);
  if(!doubleMatch(v, 2.0, doubleMachEps())) return(11);
  v=doubleMean(a+1, 5);
  if(verbose>2) printf("  v := %g\n", v);
  if(!doubleMatch(v, 1.8, doubleMachEps())) return(12);

  if(verbose>1) printf("Data with NaN\n");
  a[0]=nan("");
  v=doubleMean(a, 6);
  if(verbose>2) printf("  v := %g\n", v);
  if(!doubleMatch(v, 1.8, doubleMachEps())) return(21);

  statusSet(status, __func__, __FILE__, __LINE__, 0);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
int test_doubleWMean(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }

  double v;

  if(verbose>1) printf("NULL data\n");
  v=doubleWMean(NULL, NULL, 0);
  if(verbose>2) printf("  -> Good, not crashed\n");
  if(!isnan(v)) return(1);
  v=doubleWMean(NULL, NULL, 10);
  if(verbose>2) printf("  -> Good, not crashed\n");
  if(!isnan(v)) return(2);

  double a[]={3,2,1,1,2,3};
  double w[]={1,1,1,1,1,1};

  if(verbose>1) printf("Decent data\n");
  v=doubleWMean(a, w, 6);
  if(verbose>2) printf("  v := %g\n", v);
  if(!doubleMatch(v, 2.0, doubleMachEps())) return(11);
  v=doubleWMean(a+1, w+1, 5);
  if(verbose>2) printf("  v := %g\n", v);
  if(!doubleMatch(v, 1.8, doubleMachEps())) return(12);

  if(verbose>1) printf("Data with NaN\n");
  a[0]=nan("");
  v=doubleWMean(a, w, 6);
  if(verbose>2) printf("  v := %g\n", v);
  if(!doubleMatch(v, 1.8, doubleMachEps())) return(21);
  a[0]=3.0;

  if(verbose>1) printf("Weights are NaN\n");
  for(int i=0; i<6; i++) w[i]=nan("");
  v=doubleWMean(a, w, 6);
  if(!isnan(v)) return(31);
  if(verbose>1) printf("Weights are all 0\n");
  for(int i=0; i<6; i++) w[i]=0.0;
  v=doubleWMean(a, w, 6);
  if(!isnan(v)) return(32);

  if(verbose>1) printf("Just one weight is non-zero\n");
  w[1]=1.0;
  v=doubleWMean(a, w, 6);
  if(!doubleMatch(v, 2.0, doubleMachEps())) return(41);
  w[1]=0.5;
  v=doubleWMean(a, w, 6);
  if(!doubleMatch(v, 2.0, doubleMachEps())) return(42);

  if(verbose>1) printf("Decent weights\n");
  for(int i=0; i<6; i++) w[i]=2+i;
  v=doubleWMean(a, w, 6);
  if(!doubleMatch(v, 2.0, doubleMachEps())) return(43);


  statusSet(status, __func__, __FILE__, __LINE__, 0);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
int test_doubleGetWithUnit(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }
  char str[64];
  double v;
  int unit, ret;

  if(verbose>1) printf("str := NULL\n");
  ret=doubleGetWithUnit(NULL, NULL, NULL);
  if(ret==0) return(1);

  strcpy(str, ""); if(verbose>1) printf("str := '%s'\n", str);
  ret=doubleGetWithUnit(str, &v, NULL); if(verbose>2) printf("  -> %g\n", v);
  if(ret==0 || !isnan(v)) return(2);

  strcpy(str, "NaN"); if(verbose>1) printf("str := '%s'\n", str);
  ret=doubleGetWithUnit(str, NULL, NULL);
  if(ret==0) return(3);

  strcpy(str, " 0"); if(verbose>1) printf("str := '%s'\n", str);
  ret=doubleGetWithUnit(str, &v, NULL); if(verbose>2) printf("  -> %g\n", v);
  if(ret==0 || !isnan(v)) return(4);

  strcpy(str, "1.2345"); if(verbose>1) printf("str := '%s'\n", str);
  ret=doubleGetWithUnit(str, &v, NULL); if(verbose>2) printf("  -> %g\n", v);
  if(ret!=0 || fabs(v-1.2345)>1.0E-010) return(11);
  strcpy(str, "1,2345"); if(verbose>1) printf("str := '%s'\n", str);
  ret=doubleGetWithUnit(str, &v, NULL); if(verbose>2) printf("  -> %g\n", v);
  if(ret!=0 || fabs(v-1.2345)>1.0E-010) return(12);
  strcpy(str, "+1.2345E4"); if(verbose>1) printf("str := '%s'\n", str);
  ret=doubleGetWithUnit(str, &v, NULL); if(verbose>2) printf("  -> %g\n", v);
  if(ret!=0 || fabs(v-12345)>1.0E-010) return(13);
  strcpy(str, "-1,2345E4"); if(verbose>1) printf("str := '%s'\n", str);
  ret=doubleGetWithUnit(str, &v, NULL); if(verbose>2) printf("  -> %g\n", v);
  if(ret!=0 || fabs(v-(-12345))>1.0E-010) return(14);

  strcpy(str, "1.2345 sec"); if(verbose>1) printf("str := '%s'\n", str);
  ret=doubleGetWithUnit(str, &v, &unit); 
  if(verbose>2) printf("  -> %g %s\n", v, unitName(unit));
  if(ret!=0 || fabs(v-1.2345)>1.0E-010 || unit!=UNIT_SEC) return(21);
  strcpy(str, "1,2345 viallinen"); if(verbose>1) printf("str := '%s'\n", str);
  ret=doubleGetWithUnit(str, &v, &unit); 
  if(verbose>2) printf("  -> %g %s\n", v, unitName(unit));
  if(ret!=0 || fabs(v-1.2345)>1.0E-010 || unit!=UNIT_UNKNOWN) return(22);
  strcpy(str, "+1.2345E4  min"); if(verbose>1) printf("str := '%s'\n", str);
  ret=doubleGetWithUnit(str, &v, &unit); if(verbose>2) printf("  -> %g\n", v);
  if(ret!=0 || fabs(v-12345)>1.0E-010 || unit!=UNIT_MIN) return(23);
  strcpy(str, "-1,2345E4 1/min"); if(verbose>1) printf("str := '%s'\n", str);
  ret=doubleGetWithUnit(str, &v, &unit); if(verbose>2) printf("  -> %g\n", v);
  if(ret!=0 || fabs(v-(-12345))>1.0E-010 || unit!=UNIT_PER_MIN) return(24);

  statusSet(status, __func__, __FILE__, __LINE__, 0);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
int test_doubleSpanPositives(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }

  int gi, gn;
  double g[20], *gp;

  g[0]=0.0;
  g[1]=1.0;
  g[2]=2.0;
  g[3]=3.0;
  g[4]=4.0;
  g[5]=0.0;
  g[6]=-1.0;
  g[7]=nan("");
  g[8]=0.0;
  g[9]=5.0;
  g[10]=nan("");
  g[11]=6.0;
  g[12]=7.0;
  g[13]=-1.0;
  g[14]=0.0;
  g[15]=0.0;
  g[16]=0.0;
  g[17]=0.0;
  g[18]=0.0;
  g[19]=0.0;

  if(verbose>1) printf("\nNULL data\n");
  gi=doubleSpanPositives(NULL, 0); if(verbose>3) printf("  -> %d\n", gi);
  if(gi!=0) return(1);
  gi=doubleSpanPositives(NULL, 10); if(verbose>3) printf("  -> %d\n", gi);
  if(gi!=0) return(2);
  gi=doubleSpanPositives(g, 0); if(verbose>3) printf("  -> %d\n", gi);
  if(gi!=0) return(3);

  if(verbose>1) printf("\nArray starts with zero\n");
  gn=20; gp=g; 
  if(verbose>3) for(int i=0; i<gn; i++) printf(" a[%d]=%g\n", i, gp[i]);
  gi=doubleSpanPositives(gp, gn); if(verbose>3) printf("  -> %d\n", gi);
  if(gi!=0) return(10);

  if(verbose>1) printf("\nArray starts with NaN\n");
  gn=7; gp=&g[10]; 
  if(verbose>3) for(int i=0; i<gn; i++) printf(" a[%d]=%g\n", i, gp[i]);
  gi=doubleSpanPositives(gp, gn); if(verbose>3) printf("  -> %d\n", gi);
  if(gi!=0) return(11);

  if(verbose>1) printf("\nArray stops with zero\n");
  gn=19; gp=&g[1]; 
  if(verbose>3) for(int i=0; i<gn; i++) printf(" a[%d]=%g\n", i, gp[i]);
  gi=doubleSpanPositives(gp, gn); if(verbose>3) printf("  -> %d\n", gi);
  if(gi!=4) return(12);

  if(verbose>1) printf("\nArray stops with negative\n");
  gn=5; gp=&g[11]; 
  if(verbose>3) for(int i=0; i<gn; i++) printf(" a[%d]=%g\n", i, gp[i]);
  gi=doubleSpanPositives(gp, gn); if(verbose>3) printf("  -> %d\n", gi);
  if(gi!=2) return(13);

  if(verbose>1) printf("\nArray stops with N\n");
  gn=3; gp=&g[1]; 
  if(verbose>3) for(int i=0; i<gn; i++) printf(" a[%d]=%g\n", i, gp[i]);
  gi=doubleSpanPositives(gp, gn); if(verbose>3) printf("  -> %d\n", gi);
  if(gi!=3) return(14);

  if(verbose>1) printf("\nArray has nothing but zeroes\n");
  gn=4; gp=&g[16]; 
  if(verbose>3) for(int i=0; i<gn; i++) printf(" a[%d]=%g\n", i, gp[i]);
  gi=doubleSpanPositives(gp, gn); if(verbose>3) printf("  -> %d\n", gi);
  if(gi!=0) return(15);

  statusSet(status, __func__, __FILE__, __LINE__, 0);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
int test_doubleCSpanPositives(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }

  int gi, gn;
  double g[20], *gp;

  g[0]=0.0;
  g[1]=1.0;
  g[2]=2.0;
  g[3]=3.0;
  g[4]=4.0;
  g[5]=0.0;
  g[6]=-1.0;
  g[7]=nan("");
  g[8]=0.0;
  g[9]=5.0;
  g[10]=nan("");
  g[11]=6.0;
  g[12]=7.0;
  g[13]=-1.0;
  g[14]=0.0;
  g[15]=0.0;
  g[16]=0.0;
  g[17]=0.0;
  g[18]=0.0;
  g[19]=0.0;

  if(verbose>1) printf("\nNULL data\n");
  gi=doubleCSpanPositives(NULL, 0); if(verbose>3) printf("  -> %d\n", gi);
  if(gi!=0) return(1);
  gi=doubleCSpanPositives(NULL, 10); if(verbose>3) printf("  -> %d\n", gi);
  if(gi!=0) return(2);
  gi=doubleCSpanPositives(g, 0); if(verbose>3) printf("  -> %d\n", gi);
  if(gi!=0) return(3);

  if(verbose>1) printf("\nArray starts with zero\n");
  gn=20; gp=g; 
  if(verbose>3) for(int i=0; i<gn; i++) printf(" a[%d]=%g\n", i, gp[i]);
  gi=doubleCSpanPositives(gp, gn); if(verbose>3) printf("  -> %d\n", gi);
  if(gi!=1) return(10);

  if(verbose>1) printf("\nArray starts with NaN followed by zero\n");
  gn=10; gp=&g[7]; 
  if(verbose>3) for(int i=0; i<gn; i++) printf(" a[%d]=%g\n", i, gp[i]);
  gi=doubleCSpanPositives(gp, gn); if(verbose>3) printf("  -> %d\n", gi);
  if(gi!=2) return(11);

  if(verbose>1) printf("\nArray has negative, NaN and zero\n");
  gn=11; gp=&g[6]; 
  if(verbose>3) for(int i=0; i<gn; i++) printf(" a[%d]=%g\n", i, gp[i]);
  gi=doubleCSpanPositives(gp, gn); if(verbose>3) printf("  -> %d\n", gi);
  if(gi!=3) return(12);

  if(verbose>1) printf("\nArray stops with N\n");
  gn=6; gp=&g[14]; 
  if(verbose>3) for(int i=0; i<gn; i++) printf(" a[%d]=%g\n", i, gp[i]);
  gi=doubleCSpanPositives(gp, gn); if(verbose>3) printf("  -> %d\n", gi);
  if(gi!=6) return(13);

  statusSet(status, __func__, __FILE__, __LINE__, 0);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
int test_doubleNonzeroes(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }

  unsigned int gi, gn;
  double g[20], *gp;

  g[0]=0.0;
  g[1]=1.0;
  g[2]=2.0;
  g[3]=3.0;
  g[4]=4.0;
  g[5]=0.0;
  g[6]=-1.0;
  g[7]=nan("");
  g[8]=0.0;
  g[9]=5.0;
  g[10]=nan("");
  g[11]=6.0;
  g[12]=7.0;
  g[13]=-1.0;
  g[14]=0.0;
  g[15]=0.0;
  g[16]=0.0;
  g[17]=0.0;
  g[18]=0.0;
  g[19]=0.0;

  if(verbose>1) printf("\nNULL data\n");
  gi=doubleNonzeroes(NULL, 0); if(verbose>3) printf("  -> %d\n", gi);
  if(gi!=0) return(1);
  gi=doubleNonzeroes(NULL, 10); if(verbose>3) printf("  -> %d\n", gi);
  if(gi!=0) return(2);
  gi=doubleNonzeroes(g, 0); if(verbose>3) printf("  -> %d\n", gi);
  if(gi!=0) return(3);

  if(verbose>1) printf("\nArray contains only zeroes\n");
  gn=1; gp=g; 
  if(verbose>3) for(unsigned int i=0; i<gn; i++) printf(" a[%d]=%g\n", i, gp[i]);
  gi=doubleNonzeroes(gp, gn); if(verbose>3) printf("  -> %d\n", gi);
  if(gi!=0) return(10);
  gn=5; gp=g+14; 
  if(verbose>3) for(unsigned int i=0; i<gn; i++) printf(" a[%d]=%g\n", i, gp[i]);
  gi=doubleNonzeroes(gp, gn); if(verbose>3) printf("  -> %d\n", gi);
  if(gi!=0) return(11);

  if(verbose>1) printf("\nArray contains ... what not\n");
  gn=20; gp=g; 
  if(verbose>3) for(unsigned int i=0; i<gn; i++) printf(" a[%d]=%g\n", i, gp[i]);
  gi=doubleNonzeroes(gp, gn); if(verbose>3) printf("  -> %d\n", gi);
  if(gi!=9) return(20);

  statusSet(status, __func__, __FILE__, __LINE__, 0);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
int test_doubleMaxIndex(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }

  unsigned int i;
  double a[]={3,2,1,0,1,2,3};

  if(verbose>1) printf("NULL data\n");
  i=doubleMaxIndex(NULL, 0);
  if(verbose>2) printf("  -> Good, not crashed\n");
  if(i!=0) return(1);
  i=doubleMaxIndex(NULL, 10);
  if(verbose>2) printf("  -> Good, not crashed\n");
  if(i!=0) return(2);

  if(verbose>1) printf("Decent data\n");
  i=doubleMaxIndex(a, 7);
  if(verbose>2) printf("  i := %d\n", i);
  if(i!=0) return(11);
  i=doubleMaxIndex(a+1, 6);
  if(verbose>2) printf("  i := %d\n", i);
  if(i!=5) return(12);

  if(verbose>1) printf("Data with NaN\n");
  a[0]=nan("");
  i=doubleMaxIndex(a, 7);
  if(verbose>2) printf("  i := %d\n", i);
  if(i!=6) return(21);

  statusSet(status, __func__, __FILE__, __LINE__, 0);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
int test_doubleAbsMaxIndex(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }

  unsigned int i;
  double a[]={3,2,1,0,1,2,3};

  if(verbose>1) printf("NULL data\n");
  i=doubleAbsMaxIndex(NULL, 0);
  if(verbose>2) printf("  -> Good, not crashed\n");
  if(i!=0) return(1);
  i=doubleAbsMaxIndex(NULL, 10);
  if(verbose>2) printf("  -> Good, not crashed\n");
  if(i!=0) return(2);

  if(verbose>1) printf("Decent data\n");
  i=doubleAbsMaxIndex(a, 7);
  if(verbose>2) printf("  i := %d\n", i);
  if(i!=0) return(11);
  i=doubleAbsMaxIndex(a+1, 6);
  if(verbose>2) printf("  i := %d\n", i);
  if(i!=5) return(12);

  if(verbose>1) printf("Data with NaN\n");
  a[0]=nan("");
  i=doubleAbsMaxIndex(a, 7);
  if(verbose>2) printf("  i := %d\n", i);
  if(i!=6) return(21);
  a[0]=3.0;

  if(verbose>1) printf("Data with negatives\n");
  a[0]=a[6]=-3.0;
  i=doubleAbsMaxIndex(a, 7);
  if(verbose>2) printf("  i := %d\n", i);
  if(i!=0) return(101);
  a[0]=nan("");
  i=doubleAbsMaxIndex(a, 7);
  if(verbose>2) printf("  i := %d\n", i);
  if(i!=6) return(21);
  a[0]=3.0; a[1]=-30.0;
  i=doubleAbsMaxIndex(a, 7);
  if(verbose>2) printf("  i := %d\n", i);
  if(i!=1) return(21);

  statusSet(status, __func__, __FILE__, __LINE__, 0);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
int test_doubleMinIndex(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }

  unsigned int i;
  double a[]={3,2,1,0,1,2,3};

  if(verbose>1) printf("NULL data\n");
  i=doubleMinIndex(NULL, 0);
  if(verbose>2) printf("  -> Good, not crashed\n");
  if(i!=0) return(1);
  i=doubleMinIndex(NULL, 10);
  if(verbose>2) printf("  -> Good, not crashed\n");
  if(i!=0) return(2);

  if(verbose>1) printf("Decent data\n");
  i=doubleMinIndex(a, 7);
  if(verbose>2) printf("  i := %d\n", i);
  if(i!=3) return(11);
  i=doubleMinIndex(a, 3);
  if(verbose>2) printf("  i := %d\n", i);
  if(i!=2) return(12);

  statusSet(status, __func__, __FILE__, __LINE__, 0);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
int test_doubleAbsMinIndex(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }

  unsigned int i;
  double a[]={3,2,1,0,1,2,3};

  if(verbose>1) printf("NULL data\n");
  i=doubleAbsMinIndex(NULL, 0);
  if(verbose>2) printf("  -> Good, not crashed\n");
  if(i!=0) return(1);
  i=doubleAbsMinIndex(NULL, 10);
  if(verbose>2) printf("  -> Good, not crashed\n");
  if(i!=0) return(2);

  if(verbose>1) printf("Decent data\n");
  i=doubleAbsMinIndex(a, 7);
  if(verbose>2) printf("  i := %d\n", i);
  if(i!=3) return(11);
  i=doubleAbsMinIndex(a, 3);
  if(verbose>2) printf("  i := %d\n", i);
  if(i!=2) return(12);

  if(verbose>1) printf("Decent negative data\n");
  for(int i=0; i<7; i++) a[i]=-a[i];
  i=doubleAbsMinIndex(a, 7);
  if(verbose>2) printf("  i := %d\n", i);
  if(i!=3) return(21);
  i=doubleAbsMinIndex(a, 3);
  if(verbose>2) printf("  i := %d\n", i);
  if(i!=2) return(22);

  a[0]=-10.0; for(int i=1; i<7; i++) a[i]=a[i-1]+1;
  i=doubleAbsMinIndex(a, 7);
  if(verbose>2) printf("  i := %d\n", i);
  if(i!=6) return(31);
  i=doubleAbsMinIndex(a, 3);
  if(verbose>2) printf("  i := %d\n", i);
  if(i!=2) return(32);

  statusSet(status, __func__, __FILE__, __LINE__, 0);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
int test_doubleGEIndex(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }

  unsigned int i=doubleGEIndex(NULL, 0, 0.0);
  if(i!=0) return(1);
  {
    double a[1];
    a[0]=0.0; i=doubleGEIndex(a, 1, 1.0); if(i!=1) return(2);
    a[0]=1.0; i=doubleGEIndex(a, 1, 1.0); if(i!=0) return(3);
    a[0]=0.0; i=doubleGEIndex(a, 1, -0.1); if(i!=0) return(4);
    a[0]=0.0; i=doubleGEIndex(a, 1, 0.0); if(i!=0) return(5);
  }

  {
    double a[2];
    a[0]=0.0; a[1]=0.0; i=doubleGEIndex(a, 2, 1.0); if(i!=2) return(11);
    a[0]=1.0; a[1]=2.0; i=doubleGEIndex(a, 2, 1.0); if(i!=0) return(12);
    a[0]=0.0; a[1]=0.0; i=doubleGEIndex(a, 2, -0.1); if(i!=0) return(13);
    a[0]=0.0; a[1]=0.0; i=doubleGEIndex(a, 2, 0.0); if(i!=0) return(14);
    a[0]=1.0; a[1]=2.0; i=doubleGEIndex(a, 2, 2.0); if(i!=1) return(15);
    a[0]=nan(""); a[1]=nan(""); i=doubleGEIndex(a, 2, 0.0); if(i!=2) return(16);
    a[0]=nan(""); a[1]=1.0; i=doubleGEIndex(a, 2, 0.1); if(i!=1) return(17);
    a[0]=1.0; a[1]=nan(""); i=doubleGEIndex(a, 2, 0.0); if(i!=0) return(18);
  }

  {
    double a[3];
    a[0]=0.0; a[1]=0.0; a[2]=0.0; i=doubleGEIndex(a, 3, 1.0); if(i!=3) return(21);
    a[0]=0.0; a[1]=1.0; a[2]=2.0; i=doubleGEIndex(a, 3, 0.1); if(i!=1) return(22);
  }

  statusSet(status, __func__, __FILE__, __LINE__, 0);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
int test_doubleGTIndex(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }

  unsigned int i=doubleGTIndex(NULL, 0, 0.0);
  if(i!=0) return(1);
  {
    double a[1];
    a[0]=0.0; i=doubleGTIndex(a, 1, 1.0); if(i!=1) return(2);
    a[0]=1.0; i=doubleGTIndex(a, 1, 1.0); if(i!=1) return(3);
    a[0]=0.0; i=doubleGTIndex(a, 1, -0.1); if(i!=0) return(4);
    a[0]=1.0; i=doubleGTIndex(a, 1, 0.0); if(i!=0) return(5);
  }

  {
    double a[2];
    a[0]=0.0; a[1]=0.0; i=doubleGTIndex(a, 2, 1.0); if(i!=2) return(11);
    a[0]=1.0; a[1]=2.0; i=doubleGTIndex(a, 2, 1.0); if(i!=1) return(12);
    a[0]=0.0; a[1]=1.0; i=doubleGTIndex(a, 2, -0.1); if(i!=0) return(13);
    a[0]=0.0; a[1]=0.0; i=doubleGTIndex(a, 2, 0.0); if(i!=2) return(14);
    a[0]=1.0; a[1]=2.0; i=doubleGTIndex(a, 2, 2.0); if(i!=2) return(15);
    a[0]=nan(""); a[1]=nan(""); i=doubleGTIndex(a, 2, 0.0); if(i!=2) return(16);
    a[0]=nan(""); a[1]=1.0; i=doubleGTIndex(a, 2, 0.1); if(i!=1) return(17);
    a[0]=1.0; a[1]=nan(""); i=doubleGTIndex(a, 2, 0.0); if(i!=0) return(18);
  }

  {
    double a[3];
    a[0]=0.0; a[1]=0.0; a[2]=0.0; i=doubleGEIndex(a, 3, 1.0); if(i!=3) return(21);
    a[0]=0.0; a[1]=1.0; a[2]=2.0; i=doubleGEIndex(a, 3, 0.1); if(i!=1) return(22);
    a[0]=0.0; a[1]=1.0; a[2]=2.0; i=doubleGEIndex(a, 3, 1.01); if(i!=2) return(22);
  }

  statusSet(status, __func__, __FILE__, __LINE__, 0);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
int test_inverfc(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }

  double x, v, correctv;

  x=0.3;
  correctv=0.7328690779592168522188;
  v=inverfc(x);
  if(verbose>1) printf("inverfc(%g)=%g\n", x, v);
  if(!doubleMatch(correctv, v, 1.0E-08)) return(11);

  x=0.01;
  correctv=1.82138636771844967304;
  v=inverfc(x);
  if(verbose>1) printf("inverfc(%g)=%g\n", x, v);
  if(!doubleMatch(correctv, v, 1.0E-08)) return(12);

  x=0.99;
  correctv=0.0088625012809505979078;
  v=inverfc(x);
  if(verbose>1) printf("inverfc(%g)=%g\n", x, v);
  if(!doubleMatch(correctv, v, 1.0E-08)) return(13);

  x=0.00000001;
  correctv=4.052237243871389205231;
  v=inverfc(x);
  if(verbose>1) printf("inverfc(%g)=%g\n", x, v);
  if(!doubleMatch(correctv, v, 1.0E-08)) return(14);

  x=0.99999999;
  correctv=8.86226925452758E-09;
  v=inverfc(x);
  if(verbose>1) printf("inverfc(%g)=%g\n", x, v);
  if(!doubleMatch(correctv, v, 1.0E-12)) return(15);

  x=0.97;
  correctv=0.02659307523408841059059;
  v=inverfc(x);
  if(verbose>1) printf("inverfc(%g)=%g\n", x, v);
  if(!doubleMatch(correctv, v, 1.0E-08)) return(16);

  x=0.0;
  v=inverfc(x);
  if(verbose>1) printf("inverfc(%g)=%g\n", x, v);
  if(!isnan(v)) return(21);

  x=1.0;
  v=inverfc(x);
  correctv=0.0;
  v=inverfc(x);
  if(verbose>1) printf("inverfc(%g)=%g\n", x, v);
  if(!doubleMatch(correctv, v, 1.0E-08)) return(22);


  statusSet(status, __func__, __FILE__, __LINE__, 0);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
