/// @file tpcextensions.h
/// @brief Header file for library libtpcextensions.
/// @author Vesa Oikonen
/// @copyright (c) Turku PET Centre
///
#ifndef _TPCEXTENSIONS_H_
#define _TPCEXTENSIONS_H_
/*****************************************************************************/

/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <errno.h>
#include <stdio.h>
#include <stdint.h>
#include <stdlib.h>
#include <string.h>
#include <strings.h>
#include <ctype.h>
#include <math.h>
#include <float.h>
#include <time.h>
/*****************************************************************************/

/*****************************************************************************/
/// Define max units string length
#ifndef MAX_UNITS_LEN
#define MAX_UNITS_LEN 32 ///<  Max string length for PET units
#endif
/*****************************************************************************/

/*****************************************************************************/
/// Max length of TAC ID name (not including trailing zero)
#ifndef MAX_TACNAME_LEN
#define MAX_TACNAME_LEN 31 ///<  Max string length for TAC name
#endif
/** Max length of parameter names and units */
#ifndef MAX_PARNAME_LEN
#define MAX_PARNAME_LEN MAX_TACNAME_LEN ///<  Max string length for PAR name
#endif
/*****************************************************************************/

/*****************************************************************************/
/// Define max study number length
#ifndef MAX_STUDYNR_LEN
#define MAX_STUDYNR_LEN 255 ///<  Max string length for PET study number
#endif
/*****************************************************************************/

/*****************************************************************************/
/// @public Is data weighted, or are weight factors available with data?
typedef enum {
  WEIGHTING_UNKNOWN,    ///< Not known; usually assumed that not weighted.
  WEIGHTING_OFF,        ///< Not weighted or weights not available (weights for all included samples are 1.0).
  WEIGHTING_ON_GENERAL, ///< Weighted or weights are available, but not specified.
  WEIGHTING_ON_COUNTS,  ///< Weights based on counts (Mazoyer et al, 1986).
  WEIGHTING_ON_F,       ///< Weights based on sample frequency or frame length.
  WEIGHTING_ON_FD,      ///< Weights based on decay and sample frequency or frame length (Thiele et al, 2008).
  // possibility to add more weighting schemes
  WEIGHTING_LAST
} weights;
/*****************************************************************************/

/*****************************************************************************/
/// @public TAC content types
typedef enum {
  TACTYPE_UNKNOWN,      ///< Content type not known
  TACTYPE_PLASMA,       ///< Plasma TAC 
  TACTYPE_BLOOD,        ///< Blood TAC
  TACTYPE_PARENT,       ///< Parent tracer in plasma
  TACTYPE_METABOLITE,   ///< Metabolite in plasma
  TACTYPE_VOI,          ///< Image volume-of-interest TAC
  TACTYPE_REF,          ///< Reference region TAC
  TACTYPE_PIXEL,        ///< Image pixel TAC
  TACTYPE_LAST
} tactype;
/*****************************************************************************/

/*****************************************************************************/
/// @enum unit
/// @public unit code, same as unit position in the table (table can not
/// be accessed directly outside the c file).
/// @sa units.c unitName, unitIdentify, unitConversionFactor
typedef enum {
  UNIT_UNKNOWN,              ///< Unknown unit
  UNIT_UNITLESS,             ///< Unitless
  UNIT_HU,                   ///< Hounsfield Unit
  UNIT_MSEC,                 ///< milliseconds
  UNIT_SEC,                  ///< seconds
  UNIT_MIN,                  ///< minutes
  UNIT_HOUR,                 ///< hours
  UNIT_DAY,                  ///< days
  UNIT_MONTH,                ///< months
  UNIT_YEAR,                 ///< years
  UNIT_UM,                   ///< micrometer
  UNIT_MM,                   ///< millimeter
  UNIT_CM,                   ///< centimeter
  UNIT_M,                    ///< meter
  UNIT_UL,                   ///< microliter
  UNIT_ML,                   ///< milliliter
  UNIT_DL,                   ///< desiliter
  UNIT_L,                    ///< liter
  UNIT_UG,                   ///< microgram
  UNIT_MG,                   ///< milligram
  UNIT_G,                    ///< gram
  UNIT_100G,                 ///< 100 grams
  UNIT_KG,                   ///< kilogram
  UNIT_PMOL,                 ///< picomole
  UNIT_NMOL,                 ///< nanomole
  UNIT_UMOL,                 ///< micromole
  UNIT_MMOL,                 ///< millimole
  UNIT_MOL,                  ///< mole
  UNIT_COUNTS,               ///< counts
  UNIT_KCOUNTS,              ///< kilocounts
  UNIT_BQ,                   ///< Becquerel
  UNIT_KBQ,                  ///< kiloBecquerel
  UNIT_MBQ,                  ///< megaBecquerel
  UNIT_GBQ,                  ///< gigaBecquerel
  UNIT_NCI,                  ///< nanoCurie
  UNIT_UCI,                  ///< microCurie
  UNIT_MCI,                  ///< milliCurie
  UNIT_CI,                   ///< Curie
  UNIT_CPS,                  ///< counts/s
  UNIT_KCPS,                 ///< kilocounts/s
  UNIT_CPM,                  ///< counts/min
  UNIT_KCPM,                 ///< kilocounts/min
  UNIT_BQ_PER_ML,            ///< Bq/mL
  UNIT_KBQ_PER_ML,           ///< kBq/mL
  UNIT_MBQ_PER_ML,           ///< MBq/mL
  UNIT_NCI_PER_ML,           ///< nCi/mL
  UNIT_UCI_PER_ML,           ///< uCi/mL
  UNIT_BQ_PER_G,             ///< Bq/g
  UNIT_KBQ_PER_G,            ///< kBq/g
  UNIT_MBQ_PER_G,            ///< MBq/g
  UNIT_NCI_PER_G,            ///< nCi/g
  UNIT_UCI_PER_G,            ///< uCi/g
  UNIT_PMOL_PER_ML,          ///< pmol/mL
  UNIT_NMOL_PER_ML,          ///< nmol/mL
  UNIT_UMOL_PER_ML,          ///< umol/mL
  UNIT_MMOL_PER_ML,          ///< mmol/mL
  UNIT_PMOL_PER_L,           ///< pmol/L
  UNIT_NMOL_PER_L,           ///< nmol/L
  UNIT_UMOL_PER_L,           ///< umol/L
  UNIT_MMOL_PER_L,           ///< mmol/L
  UNIT_MOL_PER_L,            ///< mol/L
  UNIT_SEC_KBQ_PER_ML,       ///< s*kBq/mL
  UNIT_MIN_KBQ_PER_ML,       ///< min*kBq/mL
  UNIT_SEC_BQ_PER_ML,        ///< s*Bq/mL
  UNIT_MIN_BQ_PER_ML,        ///< min*Bq/mL
  UNIT_PERCENTAGE,           ///< Percentage (%)
  UNIT_ML_PER_ML,            ///< mL/mL
  UNIT_ML_PER_DL,            ///< mL/dL
  UNIT_G_PER_ML,             ///< g/mL
  UNIT_ML_PER_G,             ///< mL/g
  UNIT_PER_SEC,              ///< 1/s
  UNIT_PER_MIN,              ///< 1/min
  UNIT_PER_HOUR,             ///< 1/h
  UNIT_ML_PER_ML_SEC,        ///< mL/(mL*sec)
  UNIT_ML_PER_ML_MIN,        ///< mL/(mL*min)
  UNIT_ML_PER_DL_MIN,        ///< mL/(dL*min)
  UNIT_ML_PER_G_MIN,         ///< mL/(g*min)
  UNIT_ML_PER_100G_MIN,      ///< mL/(100g*min)
  UNIT_UMOL_PER_ML_MIN,      ///< umol/(mL*min)
  UNIT_UMOL_PER_DL_MIN,      ///< umol/(dL*min)
  UNIT_UMOL_PER_G_MIN,       ///< umol/(g*min)
  UNIT_UMOL_PER_100G_MIN,    ///< umol/(100g*min)
  UNIT_MMOL_PER_ML_MIN,      ///< mmol/(mL*min)
  UNIT_MMOL_PER_DL_MIN,      ///< mmol/(dL*min)
  UNIT_MMOL_PER_G_MIN,       ///< mmol/(g*min)
  UNIT_MMOL_PER_100G_MIN,    ///< mmol/(100g*min)
  UNIT_MG_PER_100G_MIN,      ///< mg/(100g*min)
  UNIT_MG_PER_DL_MIN,        ///< mg/(dL*min)
  UNIT_PID,                  ///< Percent of injected dose
  UNIT_PID_PER_G,            ///< Percent of injected dose / g
  UNIT_PID_PER_ML,           ///< Percent of injected dose / mL
  UNIT_PID_PER_KG,           ///< Percent of injected dose / kg
  UNIT_PID_PER_L,            ///< Percent of injected dose / L
  UNIT_MBQ_PER_NMOL,         ///< MBq/nmol (specific activity)
  UNIT_GBQ_PER_NMOL,         ///< GBq/nmol (specific activity)
  UNIT_LAST                  ///< end of list
} unit;
/*****************************************************************************/

/*****************************************************************************/
/// @brief Error codes.
/// @public Error code tpcerror, same as error message position in the table 
/// (table can not be accessed directly outside the c file).
/// @sa tpcerrormsg, statusmsg.c, TPCSTATUS
typedef enum {
  TPCERROR_OK,                 ///< No error
  TPCERROR_FAIL,               ///< General error
  TPCERROR_OUT_OF_MEMORY,      ///< Cannot allocate memory
  TPCERROR_NO_FILE,            ///< File does not exist
  TPCERROR_INVALID_FILENAME,   ///< Invalid file name
  TPCERROR_CANNOT_OPEN,        ///< Cannot open file
  TPCERROR_CANNOT_READ,        ///< Cannot read file
  TPCERROR_CANNOT_WRITE,       ///< Cannot write file
  TPCERROR_INVALID_FORMAT,     ///< Invalid file format
  TPCERROR_UNSUPPORTED,        ///< Unsupported file type
  TPCERROR_NO_DATA,            ///< File contains no data
  TPCERROR_DUPLICATE_DATA,     ///< File contains duplicate data
  TPCERROR_NO_WEIGHTS,         ///< File contains no weights
  TPCERROR_NO_DATETIME,        ///< File contains no date or time
  TPCERROR_MISSING_DATA,       ///< File contains missing values
  TPCERROR_TOO_FEW,            ///< File contains too few samples
  TPCERROR_TOO_BIG,            ///< File is too big
  TPCERROR_OVERLAPPING_DATA,   ///< Overlapping data
  TPCERROR_LARGE_GAP,          ///< Large gap in data
  TPCERROR_NO_X,               ///< No sample times
  TPCERROR_INVALID_X,          ///< Invalid sample time
  TPCERROR_INVALID_XRANGE,     ///< Invalid sample time range
  TPCERROR_INCOMPATIBLE_UNIT,  ///< Incompatible units
  TPCERROR_INCOMPATIBLE_DATA,  ///< Incompatible data
  TPCERROR_NO_REFERENCE,       ///< Reference not found
  TPCERROR_NO_KEY,             ///< Key not found
  TPCERROR_NO_VALUE,           ///< Value not found
  TPCERROR_INVALID_VALUE,      ///< Invalid value
  TPCERROR_INVALID_SEPARATOR,  ///< Invalid field delimiter
  TPCERROR_INVALID_PARNR,      ///< Invalid number of parameters
  TPCERROR_INVALID_HEADER,     ///< Invalid header contents
  TPCERROR_MISSING_HEADER,     ///< Missing header
  TPCERROR_UNKNOWN_ISOTOPE,    ///< Unknown isotope
  TPCERROR_UNKNOWN_UNIT,       ///< Unknown data unit
  TPCERROR_NO_SOLUTION,        ///< No solution
  TPCERROR_BAD_FIT,            ///< Fitting not successful
  TPCERROR_LAST                ///< marking the end of list
} tpcerror;
/*****************************************************************************/

/*****************************************************************************/
/** Memory holder for storing and transferring status and information
    between functions for debug and error printing purposes.

    Do not change the contents of these fields except for verbose and error;
    change other fields only using functions statusSet() and statusFree().
    @sa statusInit, statusSet, statusFree, statusPrint, tpcerror, errorMsg
 */
typedef struct TPCSTATUS {
  int verbose;         ///< Verbose level, used by statusPrint() etc.
  FILE *fp;            ///< File pointer for writing log information during development and testing.
  int forgiving;       ///< Force level, 0 for strict tests for data units etc.
  tpcerror error;      ///< Error code.
  char *current_func;  ///< Function where status was last time set.
  char *current_file;  ///< Source file name where status was last time set.
  int current_line;    ///< Source line nr where status was last time set.
  char *last_func;     ///< Function where status was previously set.
  char *last_file;     ///< Source file name where status previously set.
  int last_line;       ///< Source line nr where status was previously set.
} TPCSTATUS;
/*****************************************************************************/

/*****************************************************************************/
/* stringext */
extern int strTokenNr(const char *s1, const char *s2);
extern int strTokenNCpy(const char *s1, const char *s2, int i, char *s3, int count);
extern int strChrCount(const char *s1, const char *s2);
extern int strUppercaseCount(const char *s);
extern void strReplaceChar(char *s, char c1, char c2);
/////#ifndef HAVE_STRCASESTR
extern char *strcasestr(const char *haystack, const char *needle);
/////#endif
#ifndef HAVE_STRDUP
extern char *strdup(const char *s);
#endif
#ifndef HAVE_STRNDUP
extern char *strndup(const char *s, size_t n);
#endif
//#ifndef HAVE_STRNLEN
extern size_t strnlen(const char *s, size_t n);
//#endif
//#ifndef HAVE_STRLCAT
extern size_t strlcat(char *dst, const char *src, size_t dstsize);
//#endif
//#ifndef HAVE_STRLCPY
extern size_t strlcpy(char *dst, const char *src, size_t dstsize);
//#endif
extern char *strstrNoQuotation(const char *haystack, const char *needle);
extern int strncpyCleanSpaces(char *s1, const char *s2, int maxlen);
extern int strCleanSpaces(char *s);
extern int strncpyClean(char *s1, const char *s2, int maxlen);
extern int strClean(char *s);
extern char *strTokenDup(const char *s1, const char *s2, int *next);
extern int strInPars(char *s);
extern void strCleanPars(char *s);
extern char *strncatInt(char *str1, const int n, size_t count);
extern char *strncatDouble(char *str1, const double d, size_t count);
extern char *strncatIntZ(char *str1, const int n, const int maxn, size_t count);
extern int strIsSpaceOnly(char *s);
extern char *strdelstr(char *s1, const char *s2);
extern char *strTrimLeft(char *s, const size_t t);
extern char *strEncodeForXML(const char *s);
extern void strCleanForXML(char *s);
/*****************************************************************************/

/*****************************************************************************/
/* decpoint */
extern int strIsValidNumber(const char *s);
extern double atofVerified(const char *s);
extern int atofCheck(const char *s, double *v);
extern int strHaveDecimalComma(const char *s);
extern int strHaveDecimalSeparator(const char *s);
extern int atofList(const char *s1, const char *s2, double *x, int maxn);
/*****************************************************************************/

/*****************************************************************************/
/* filename */
extern void filenameRmPath(char *s);
extern void filenameRmFile(char *s);
extern int filenameRmExtension(char *s);
extern void filenameRmExtensions(char *s);
extern int fnmatch(const char *fname, const char *key);
extern int fncasematch(const char *fname, const char *key);
extern char *filenameGetExtension(const char *s);
extern char *filenameGetExtensions(const char *s);
/*****************************************************************************/

/*****************************************************************************/
/* proginfo */
extern int tpcProcessStdOptions(const char *s, int *print_usage, int *print_version, int *verbose_level);
extern void tpcPrintUsage(const char *program, char *text[], FILE *fp);
extern int tpcHtmlUsage(const char *program, char *text[], const char *path);
extern void tpcPrintBuild(const char *program, FILE *fp);
extern void tpcProgramName(const char *program, int version, int copyright, char *prname, int n);
extern int tpcYesNo(const char *s);
/*****************************************************************************/

/*****************************************************************************/
/* units */
extern char *unitName(int unit_code);
extern int unitIdentify(const char *s);
extern int unitIdentifyFilename(const char *s);
extern int unitIsDistance(int u);
extern int unitIsTime(int u);
extern int unitIsVolume(int u);
extern int unitIsMass(int u);
extern int unitIsMole(int u);
extern int unitIsRadioactivity(int u);
extern int unitIsRAConc(int u);
extern int unitIsCombinatorial(int u);
extern double unitConversionFactor(const int u1, const int u2);
extern int unitInverse(int u);
extern int unitDividerHasVolume(int u);
extern int unitDividerHasMass(int u);
extern int unitDividendHasRadioactivity(int u);
extern int unitCombination(const int u1, const int u2, const int v1, const int v2);
extern int unitDividerMassVolumeConversion(int u);
extern int unitMultiply(int ua, int ub);
/*****************************************************************************/

/*****************************************************************************/
/* statusmsg */
extern void statusPrint(FILE *fp, TPCSTATUS *s);
extern void statusInit(TPCSTATUS *s);
extern void statusFree(TPCSTATUS *s);
extern void statusSet(TPCSTATUS *s, const char *func, const char *srcfile, int srcline, tpcerror error);
extern char *errorMsg(tpcerror e);
/*****************************************************************************/

/*****************************************************************************/
/* readasciifile */
extern size_t asciiFileSize(FILE *fp, int *nonprintable);
extern char *asciiFileRead(FILE *fp, char *data, size_t maxlen);
extern int asciiCommentLine(const char *line, int *cont);
/*****************************************************************************/

/*****************************************************************************/
/* datetime */
#ifndef HAVE_GMTIME_R
extern struct tm* gmtime_r(const time_t* t, struct tm* tm);
#endif
#ifndef HAVE_LOCALTIME_R
extern struct tm* localtime_r(const time_t* t, struct tm* tm);
#endif
//#ifndef HAVE_TIMEGM
extern time_t timegm(struct tm *tm); // always needed at least in macOS
//#endif
extern char* ctime_r_int(const time_t *t, char *buf);
extern int strDateValid(const char *str);
extern int strDateValid2(const char *str, char *intdate);
extern int strDateValid3(const char *str, char *intdate);
extern int strDateValid4(int dateint, char *intdate, int *year, int *month, int *day);
extern int strTimeValid(const char *str);
extern int strDateTimeValid(const char *str, char *intdate);
extern int strDateTimeRead(const char *str, struct tm *date);
extern int strDateRead(const char *str, struct tm *date);
extern void time_to_tm(time_t totalsecs, int offset, struct tm *result);
extern double tmDifference(struct tm *tm1, struct tm *tm0);
extern void tmAdd(int s, struct tm *d);
extern double strDateTimeDifference(const char *dt1, const char *dt0);
extern int strDateTimeAdd(int s, char *dt);
/*****************************************************************************/

/*****************************************************************************/
/* studynr */
extern int studynrVerify(const char *s, int zero_ok);
extern int studynrStandardize(char *s);
extern int studynrFromFilename(const char *fname, char *studynr, int force);
/*****************************************************************************/

/*****************************************************************************/
/* doubleutil */
extern int doubleMatch(const double v1, const double v2, const double lim);
extern int doubleArrayMatch(const double *a1, const double *a2, const unsigned int n, const double lim);
extern int doubleMatchRel(const double v1, const double v2, const double lim);
extern double doubleMachEps();
extern void doubleCopy(double *t, double *s, const unsigned int n);
extern unsigned int doubleCopyFinite(double *t, double *s, const unsigned int n);
extern unsigned int doubleNaNs(double *a, const unsigned int n);
extern unsigned int doubleRange(double *a, const unsigned int n, double *amin, double *amax);
extern double doubleSum(double *a, const unsigned int n);
extern double doubleMean(double *a, const unsigned int n);
extern double doubleWMean(double *a, double *w, const unsigned int n);
extern int doubleGetWithUnit(const char *s, double *v, int *u);
extern int doubleSpanPositives(double *a, const int n);
extern int doubleCSpanPositives(double *a, const int n);
extern unsigned int doubleNonzeroes(double *a, const unsigned int n);
extern unsigned int doubleMaxIndex(double *a, const unsigned int n);
extern unsigned int doubleAbsMaxIndex(double *a, const unsigned int n);
extern unsigned int doubleMinIndex(double *a, const unsigned int n);
extern unsigned int doubleAbsMinIndex(double *a, const unsigned int n);
extern unsigned int doubleGEIndex(double *a, const unsigned int n, double lim);
extern unsigned int doubleGTIndex(double *a, const unsigned int n, double lim);
extern double inverfc(double x);
/*****************************************************************************/

/*****************************************************************************/
/* floatutil */
extern int floatMatch(const float v1, const float v2, const float lim);
extern int floatMatchRel(const float v1, const float v2, const float lim);
extern float floatMachEps();
extern void floatCopy(float *t, float *s, const unsigned int n);
extern unsigned int floatMaxIndex(float *a, const unsigned int n);
extern float floatSum(float *a, const unsigned int n);
extern float floatMean(float *a, const unsigned int n);
extern int floatGetWithUnit(const char *s, float *v, int *u);
extern int floatSpanPositives(float *a, const int n);
extern int floatCSpanPositives(float *a, const int n);
extern unsigned int floatNonzeroes(float *a, const unsigned int n);
/*****************************************************************************/

/*****************************************************************************/
/* intutil */
extern int atoiCheck(const char *s, int *v);
extern unsigned int factorial(unsigned int n);
extern unsigned long long int lfactorial(unsigned long long int n);

/** List of integer values.
    @sa integerListInit, integerListEmpty, integerListAdd, integerListSort. */
typedef struct INTLIST {
  /** Nr of stored integers. */
  int nr;
  /** Allocated list size, >= nr. */
  int _nr;
  /** Array of integers. */
  int *i;
} INTLIST;
extern void intlistInit(INTLIST *l);
extern void intlistFree(INTLIST *l);
extern int intlistAdd(INTLIST *l, const int v, const int ifnew);
extern void intlistSort(INTLIST *l);
extern int intlistAddFromString(const char *s1, const char *s2, INTLIST *l, const int ifnew);
extern int intlistExpandFromString(const char *s1, const char *s2, INTLIST *l, const int ifnew);
/*****************************************************************************/

/*****************************************************************************/
/* roiname */
extern char *roinameSubpart(
  const char *roiname, const char *dlm, const unsigned int si, 
  char *subpart, const unsigned int slen
);
extern char *roinameEditByTemplate(
  const char *template, const char *currname,
  char *newname, const unsigned int count 
);
extern char *roinameAddField(
  char *roiname, const char *field, 
  const unsigned int in, const unsigned int count
);
extern int roinameExists(char *roiname);
extern int roinameMatch(const char *roiname, const char *test_str, TPCSTATUS *status);
/*****************************************************************************/

/*****************************************************************************/
/* endian */
enum { // used by endianness()
  ENDIAN_UNKNOWN,
  ENDIAN_BIG,
  ENDIAN_LITTLE,
  ENDIAN_BIG_WORD,   /* Middle-endian, Honeywell 316 style */
  ENDIAN_LITTLE_WORD /* Middle-endian, PDP-11 style */
};
extern int endianness();
extern int endianLittle();
extern void swap(void *from, void *to, int size);
extern void swabip(void *buf, int size);
extern void swawbip(void *buf, int size);
extern void swawip(void *buf, int size);
extern void swap64ip(void *buf, unsigned long long size);
extern void swap32ip(void *buf, unsigned long long size);
extern void swap16ip(void *buf, unsigned long long size);
/*****************************************************************************/

/*****************************************************************************/
#endif /* TPCEXTENSIONS */

