/** @file pathexist.c
 *  @brief Functions for working with paths.
 */
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include "tpcextensions.h"
/*****************************************************************************/
#include "tpcfileutil.h"
/*****************************************************************************/

/*****************************************************************************/
/** Check if specified directory (path) exists.
   @sa fileExist, pathFileNr, filenameRmFile
   @return Returns 1, if path exists, and 0 if it does not or is not a directory.
 */
int pathExist(
  /** Name of path to check. */
  const char *pathname
) {
  if(pathname==NULL) return(0);
  int len=strlen(pathname);
  if(len<1) return(0);
  char *str=strdup(pathname);
  if(len>1 && str[len-1]=='/') {
    if(len<2) {free(str); return(0);}
    str[len-1]='\0';
  }
  struct stat fst;
  if(stat(str, &fst)!=0) {free(str); return(0);}
  free(str);
  if(S_ISDIR(fst.st_mode)) return(1);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Remove path.
   @sa fileExist, pathFileNr, filenameRmFile
   @return Returns 0, if path could be removed or it did not exist, 1, if it
    exists and cannot be removed.
 */
int pathRemove(
  /** Name of path to remove. */
  const char *pathname
) {
  if(fileExist(pathname)) return(1);
  if(!pathExist(pathname)) return(0);
  if(rmdir(pathname)==0) return(0); else return(1);
}
/*****************************************************************************/

/*****************************************************************************/
/** Create path.
   @sa pathExist, pathRemove, pathFileNr, filenameRmFile
   @return Returns 0, if path could be created or exists already, 1, if it
    does not exist and cannot be created.
 */
int pathCreate(
  /** Name of path to create; can contain subdirectories. */
  const char *pathname
) {
  //printf("pathname='%s'\n", pathname);
  if(pathname==NULL || strnlen(pathname, 2)<1) return(1);
  if(fileExist(pathname)) return(1);
  if(pathExist(pathname)) return(0); // exists already
  // check if this is a subfolder
  char *lpath=strdup(pathname);
  char *cptr=strrchr(lpath, '/'); if(cptr==NULL) cptr=strrchr(lpath, '\\');
  if(cptr!=NULL) { // it is a subfolder, so call recursively with the parent folder name
    *cptr=(char)0;
    int ret=pathCreate(lpath);
    if(ret!=0) {free(lpath); return(1);}
  }
  free(lpath);
  // now try to create the folder
#ifndef HAVE__MKDIR
  // other than Windows
  if(mkdir(pathname, 0777)!=0 && errno!=EEXIST) return(1); else return(0);
#else
  // Windows (MinGW)
  if(_mkdir(pathname)!=0 && errno!=EEXIST) return(1); else return(0);
#endif
}
/*****************************************************************************/

/*****************************************************************************/
