/** @file imageio.c
    @brief IO functions for image data.
    @copyright (c) Turku PET Centre
    @author Vesa Oikonen
 */
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include "tpcimage.h"
/*****************************************************************************/

/*****************************************************************************/
/** Text representations of image file format.
    @sa imgformat, tpcimage.h */
static const char *img_format[] = {
  "Unknown",                    // IMG_FORMAT_UNKNOWN
  "DICOM",                      // IMG_FORMAT_DICOM
  "ECAT6.3",                    // IMG_FORMAT_E63
  "ECAT7",                      // IMG_FORMAT_E7
  "ECAT7-2D",                   // IMG_FORMAT_E7_2D
  "ECAT-polarmap",              // IMG_FORMAT_POLARMAP,
  "Analyze-big",                // IMG_FORMAT_ANA
  "Analyze-little",             // IMG_FORMAT_ANA_L
  "Interfile",                  // IMG_FORMAT_INTERFILE
  "NIfTI-1-dual",               // IMG_FORMAT_NIFTI_1D
  "NIfTI-1-single",             // IMG_FORMAT_NIFTI_1S
  "NIfTI-2-dual",               // IMG_FORMAT_NIFTI_2D
  "NIfTI-2-single",             // IMG_FORMAT_NIFTI_2S
  "MicroPET",                   // IMG_FORMAT_MICROPET
  "Flat",                       // IMG_FORMAT_FLAT
0};
/** Return pointer to image file format description with the format code.
    @return pointer to the image file format string.
    @sa imgformat
 */
char *imgFormatDescr(
  /** IMG format code */
  imgformat c
) {
  if(c<IMG_FORMAT_UNKNOWN || c>=IMG_FORMAT_LAST) return NULL;
  return (char*)img_format[c];
}
/*****************************************************************************/

/*****************************************************************************/
/** Default file name extensions of image file formats. */
static const char *img_fn_ext[] = {
  "",             // IMG_FORMAT_UNKNOWN
  ".dcm",         // IMG_FORMAT_DICOM
  ".img",         // IMG_FORMAT_E63
  ".v",           // IMG_FORMAT_E7
  ".i",           // IMG_FORMAT_E7_2D
  ".pm",          // IMG_FORMAT_POLARMAP,
  "",             // IMG_FORMAT_ANA
  "",             // IMG_FORMAT_ANA_L
  "",             // IMG_FORMAT_INTERFILE
  "",             // IMG_FORMAT_NIFTI_1D
  ".nii",         // IMG_FORMAT_NIFTI_1S
  "",             // IMG_FORMAT_MICROPET
  ".bin",         // IMG_FORMAT_FLAT
0};
/** Return pointer to default image file name extension based on the format code.
    @remark No extension for most formats.
    @return pointer to the file name extension starting with '.', or empty string.
    @sa imgformat
 */
char *imgDefaultExtension(
  /** IMG format code */
  imgformat c
) {
  if(c<IMG_FORMAT_UNKNOWN || c>=IMG_FORMAT_LAST) return NULL;
  return (char*)img_fn_ext[c];
}
/*****************************************************************************/

/*****************************************************************************/
/** Read image file into IMG structure.
    @remark Stub function.
    @sa imgInit, imgWrite, imgFree
    @return enum tpcerror (TPCERROR_OK when successful).
 */
int imgRead(
  /** Pointer to initialized image structure; old contents are deleted.
      @pre Initialize structure using imgInit() before this. */
  IMG *img,
  /** Pointer to the file name, possibly path in case of DICOM. */
  const char *fname,
  /** Pointer to status data; enter NULL if not needed. */
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  if(verbose>0) {printf("%s(img, %s)\n", __func__, fname); fflush(stdout);}

  if(img==NULL || strnlen(fname, 2)<1) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL);
    return(TPCERROR_FAIL);
  }

  /* Delete any old contents in data structure */
  imgFree(img);

  /* What kind of data are we supposed to read? */
  if(pathExist(fname)) {

    if(verbose>2) printf("%s is a path\n", fname);
    unsigned short int fileNr=pathFileNr(fname);
    if(verbose>2) printf("%s contains %u file(s)\n", fname, fileNr);
    if(fileNr==0) {
      statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_FILE);
      return(TPCERROR_NO_FILE);
    }
    /* Check if this is a DICOM folder */
    IFT fl; iftInit(&fl);
    int ret=dcmFileList(fname, &fl, status);
    if(ret==TPCERROR_OK) {
      if(verbose>2) printf("%s is DICOM folder.\n", fname);
      img->format=IMG_FORMAT_DICOM;
    } else {
      if(verbose>2) printf("%s is not a DICOM folder.\n", fname);
      fileNr=pathFileList(fname, &fl); // just read the file list
    }
    if(fileNr==0) {
      iftFree(&fl);
      statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_FILE);
      return(TPCERROR_NO_FILE);
    }
    /* It would be too risky to just open random file from the folder */
    iftFree(&fl);
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_UNSUPPORTED);
    return(TPCERROR_UNSUPPORTED);

  } else if(fileExist(fname)) {

    if(verbose>2) printf("%s is a file\n", fname);
    /* Check if this is DICOM file */
    if(dcmVerifyMagic(fname, NULL)) {
      /*if(verbose>2)*/ printf("%s is in DICOM format.\n", fname);
      img->format=IMG_FORMAT_DICOM;
      /* This may be a single-file DICOM, or user just gave the name of one file in DICOM folder */
      IFT fl; iftInit(&fl);
      int ret=dcmFileList(fname, &fl, status);
      if(ret!=TPCERROR_OK) {iftFree(&fl); return(ret);}
      if(fl.keyNr==1) {
        if(verbose>2) printf("  %s is a single DICOM file\n", fname);
        iftFree(&fl);
        ret=imgReadDICOM(img, fname, status);
        if(verbose>2) printf("  imgReadDICOM() := %s\n", errorMsg(status->error));
        return(ret);
      } else {
        if(verbose>2) printf("  %s is one of %d DICOM files\n", fname, fl.keyNr);

        iftFree(&fl);
        statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_UNSUPPORTED);
        return(TPCERROR_UNSUPPORTED);
      }
    }
    /* Check if we have NIfTI file, which may be in single file format,
       or dual file format which has similar names as microPET and Analyze */
    if(niftiExists(fname, NULL, NULL, NULL, NULL, status)==1) {
      /* Read NIfTI image */
      /*if(verbose>2)*/ printf("%s is in NIfTI format.\n", fname);
      statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_UNSUPPORTED);
      return(TPCERROR_UNSUPPORTED);
    }
    /* Check if we have Analyze file, which may be in dual file format which 
       has similar names as microPET and dual-file NIfTI */
    if(anaExists(fname, NULL, NULL, NULL, NULL, status)==1) {
      /* Read Analyze image */
      /*if(verbose>2)*/ printf("%s is in Analyze format.\n", fname);
      statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_UNSUPPORTED);
      return(TPCERROR_UNSUPPORTED);
    }

  } else { // Not a folder, neither an existing file

    /* Check if this is base name of an image, without extension(s) */
    /* Check if we have NIfTI file, which may be in single file format,
       or dual file format which has similar names as microPET and Analyze */
    if(niftiExists(fname, NULL, NULL, NULL, NULL, status)==1) {
      /* Read NIfTI image */
      /*if(verbose>2)*/ printf("%s is in NIfTI format.\n", fname);
      statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_UNSUPPORTED);
      return(TPCERROR_UNSUPPORTED);
    }
    /* Check if we have Analyze file, which may be in dual file format which 
       has similar names as microPET and dual-file NIfTI */
    if(anaExists(fname, NULL, NULL, NULL, NULL, status)==1) {
      /* Read Analyze image */
      /*if(verbose>2)*/ printf("%s is in Analyze format.\n", fname);
      statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_UNSUPPORTED);
      return(TPCERROR_UNSUPPORTED);
    }
    /* Check for microPET */


    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_FILE);
    return(TPCERROR_NO_FILE);
  }



  statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  return(TPCERROR_OK);
}
/*****************************************************************************/

/*****************************************************************************/
/** Write image file from IMG structure.
    @remark Stub function.
    @sa imgInit, imgRead, imgFree
    @return enum tpcerror (TPCERROR_OK when successful).
 */
int imgWrite(
  /** Pointer to image structure. */
  IMG *img,
  /** Pointer to the file name, possibly path in case of DICOM. */
  const char *fname,
  /** Pointer to status data; enter NULL if not needed. */
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  if(verbose>0) {printf("%s(img, %s)\n", __func__, fname); fflush(stdout);}

  if(img==NULL || strnlen(fname, 2)<1) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL);
    return(TPCERROR_FAIL);
  }
  if(!imgHasData(img)) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_DATA);
    return(TPCERROR_NO_DATA);
  }

  /* Based on file name, guess file format, and use it, or check that specified format is ok */
  imgformat format=imgFormatFromFName(fname, NULL);
  if(format!=IMG_FORMAT_UNKNOWN) {
    if(img->oformat==IMG_FORMAT_UNKNOWN) {
      img->oformat=format;
      if(verbose>1)
        printf("based on file name, output format set to %s\n", imgFormatDescr(img->oformat));
    } else if(format!=img->oformat) {
      if(verbose>1)
        printf("file name is not suitable for format %s\n", imgFormatDescr(img->oformat));
      statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_FILENAME);
      return(TPCERROR_INVALID_FILENAME);
    }
  }
  /* If output format is not yet set, then use input format */
  if(img->oformat==IMG_FORMAT_UNKNOWN && img->format!=IMG_FORMAT_UNKNOWN) {
    img->oformat=img->format;
    if(verbose>1) printf("output set to same as input format (%s)\n", imgFormatDescr(img->oformat));
  }
  /* If output format still unknown, set it to singe-file NIfTI */
  if(img->oformat==IMG_FORMAT_UNKNOWN) {
    img->oformat=IMG_FORMAT_NIFTI_1S;
    if(verbose>1) printf("output format set to default (%s)\n", imgFormatDescr(img->oformat));
  }

  /* Fill output header */
  if(imgFillOHeader(img, status)!=TPCERROR_OK) return(TPCERROR_INVALID_HEADER);

  /* Write image */
  if(img->oformat==IMG_FORMAT_DICOM) {

    if(verbose>1) printf("writing %s\n", imgFormatDescr(img->oformat));

  } else if(img->oformat==IMG_FORMAT_NIFTI_1D || img->oformat==IMG_FORMAT_NIFTI_1S) {

    if(verbose>1) printf("writing %s\n", imgFormatDescr(img->oformat));
    return(imgWriteNifti(img, fname, status));

  } else if(img->oformat==IMG_FORMAT_E7) {

    if(verbose>1) printf("writing %s\n", imgFormatDescr(img->oformat));

    /* Create main header data block */
    unsigned char mh[ECATBLKSIZE];
    int ret=ecatWriteMainheader(&img->oh, mh, status);
    if(ret!=TPCERROR_OK) return(ret);

    /* Open file, write main header block, and initiate matrix list */

    /* Process each frame */
    for(unsigned int ti=0; ti<img->dimt; ti++) {
      /* Create subheader data block */

      /* Create blocks for pixel data */

      /* Write header and data blocks */

    }

    /* Close file */

  } else if(img->oformat==IMG_FORMAT_E63) {

    if(verbose>1) printf("writing %s\n", imgFormatDescr(img->oformat));

    /* Create main header data block */

    /* Open file, write main header block, and initiate matrix list */

    /* Process each plane and frame */
    for(unsigned int ti=0; ti<img->dimt; ti++) {
      for(unsigned int zi=0; zi<img->dimz; zi++) {
        /* Create subheader data block */

        /* Create blocks for pixel data */

        /* Write header and data blocks */

      }
    }

    /* Close file */

  } else if(img->oformat==IMG_FORMAT_FLAT) {

    if(verbose>1) printf("writing %s\n", imgFormatDescr(img->oformat));
    FILE *fp=NULL;
    float *fdata=NULL, *fptr;
    if(verbose>2) printf("opening %s\n", fname);
    if((fp=fopen(fname, "wb")) == NULL) {
      statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_CANNOT_WRITE);
      return(TPCERROR_CANNOT_WRITE);
    }
    /* Allocate memory for float data for one frame */
    unsigned long long pxlNr=img->dimz*img->dimy*img->dimx;
    if(verbose>2) printf("allocating memory for %llu pixels\n", pxlNr);
    fdata=(float*)malloc(pxlNr*sizeof(float));
    if(fdata==NULL) {
      fclose(fp);
      statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OUT_OF_MEMORY);
      return(TPCERROR_OUT_OF_MEMORY);
    }
    /* Write one frame at a time */
    if(verbose>2) printf("writing frame at a time\n");
    for(int ti=0; ti<img->dimt; ti++) {
      fptr=fdata;
      for(int zi=0; zi<img->dimz; zi++)
        for(int yi=0; yi<img->dimy; yi++)
          for(int xi=0; xi<img->dimx; xi++)
            *fptr++=img->m[zi][yi][xi][ti];
      if(fwrite((float*)fdata, 4, pxlNr, fp) != pxlNr) {
        fclose(fp); remove(fname); free(fdata);
        statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_CANNOT_WRITE);
        return(TPCERROR_CANNOT_WRITE);
      }
    } /* next frame */
    /* Close file, free memory */
    fclose(fp); free(fdata);
    if(verbose>2) printf("binary file written\n");
    /* Write matrix information file; no error if not successful. */
    char miffile[strlen(fname)+5]; sprintf(miffile, "%s.mif", fname);
    if(verbose>2) printf("opening matrix information file %s\n", miffile);
    if((fp=fopen(miffile, "w")) == NULL) {
      if(verbose>0) fprintf(stderr, "Error: cannot open MIF for write.\n");
      fclose(fp);
      statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
      return(TPCERROR_OK);
    }
    if(verbose>2) printf("writing matrix information file %s\n", miffile);
    int n=fprintf(fp, "%d %d %d %d\n", img->dimz, img->dimt, img->dimx, img->dimy);
    fclose(fp);
    if(n<7) {
      if(verbose>0) fprintf(stderr, "Error: cannot write in file %s\n", miffile); 
      statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
      return(TPCERROR_OK);
    }
    if(verbose>2) printf("matrix information saved in %s\n", miffile);

  } else {
    if(verbose>0) printf("output format (%s) not supported\n", imgFormatDescr(img->oformat));
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_UNSUPPORTED);
    return(TPCERROR_UNSUPPORTED);
  }

  statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  return(TPCERROR_OK);
}
/*****************************************************************************/

/*****************************************************************************/
/** Determine IMG file format from file name.

    If file name is actually name of an existing path, then DICOM is assumed.
    Non-existing path can be identified as path only if it ends with '/'.

    Note that NIfTI dual file format, Analyze, and microPET files can not be
    separated by file naming.

   @returns enum imgformat (IMG_FORMAT_DICOM, ...), and IMG_FORMAT_UNKNOWN, when not certain.
   @sa imgFormatIdentify, imgRead, imgWrite, micropetExists, niftiExists, anaExists
 */
imgformat imgFormatFromFName(
  /** Name of file that is used to determine format. */
  const char *fname,
  /** Pointer to status data; enter NULL if not needed. */
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  if(verbose>0) {printf("%s(%s)\n", __func__, fname); fflush(stdout);}

  /* Empty name? */
  if(fname==NULL || strnlen(fname, 2)<1) return(IMG_FORMAT_UNKNOWN);

  /* Path as file name? */
  if(pathExist(fname)) {
    if(verbose>1) printf("%s is an existing path\n", fname);
    return(IMG_FORMAT_DICOM);
  }
  if(fname[strlen(fname)-1]=='/' || fname[strlen(fname)-1]=='\\') {
    if(verbose>1) printf("%s is a non-existing path\n", fname);
    return(IMG_FORMAT_DICOM);
  }

  /* Get the extension(s) */
  char *exts=filenameGetExtensions(fname);
  if(exts==NULL) {
    if(verbose>1) printf("file name has no extension\n");
    /* Probably NIfTI, Analyze, or MicroPET, but we cannot be sure */
    return(IMG_FORMAT_UNKNOWN);
  }
  if(verbose>1) printf("file name extension: '%s'\n", exts);

  /* Check certain common extensions */
  if(strcasecmp(exts, ".dcm")==0) return(IMG_FORMAT_DICOM);
  if(strcasecmp(exts, ".dicom")==0) return(IMG_FORMAT_DICOM);

  if(strcasecmp(exts, ".nii")==0) return(IMG_FORMAT_NIFTI_1S);

  if(strcasecmp(exts, ".v")==0) return(IMG_FORMAT_E7);
  if(strcasecmp(exts, ".s")==0) return(IMG_FORMAT_E7);
  if(strcasecmp(exts, ".n")==0) return(IMG_FORMAT_E7);
  if(strcasecmp(exts, ".a")==0) return(IMG_FORMAT_E7);
  if(strcasecmp(exts, ".scn")==0) return(IMG_FORMAT_E63);
  if(strcasecmp(exts, ".nrm")==0) return(IMG_FORMAT_E63);
  if(strcasecmp(exts, ".atn")==0) return(IMG_FORMAT_E63);
  if(strcasestr(exts, "polarmap")!=NULL) return(IMG_FORMAT_POLARMAP);

  if(strcasecmp(exts, ".if")==0) return(IMG_FORMAT_INTERFILE);
  if(strcasestr(exts, ".i.hdr")!=NULL) return(IMG_FORMAT_INTERFILE);
  if(strcasestr(exts, ".i.img")!=NULL) return(IMG_FORMAT_INTERFILE);
  if(strcasecmp(exts, ".i")==0) return(IMG_FORMAT_INTERFILE);

  if(strcasecmp(exts, ".img")==0) return(IMG_FORMAT_E63);

  if(strcasecmp(exts, ".pet.img")==0) return(IMG_FORMAT_MICROPET);
  if(strcasecmp(exts, ".pet.img.hdr")==0) return(IMG_FORMAT_MICROPET);
  if(strcasecmp(exts, ".ct.img")==0) return(IMG_FORMAT_MICROPET);
  if(strcasecmp(exts, ".ct.img.hdr")==0) return(IMG_FORMAT_MICROPET);

  /* Check certain in-house extensions */
  if(strcasecmp(exts, ".bin")==0) return(IMG_FORMAT_FLAT);
  if(strcasecmp(exts, ".mif")==0) return(IMG_FORMAT_FLAT);

  return(IMG_FORMAT_UNKNOWN);
}
/*****************************************************************************/

/*****************************************************************************/
/** Identify the string representation of the IMG file format.
    @return enum imgformat, or 0 (enum IMG_FORMAT_UNKNOWN) if not identified.
    @sa imgFormatFromFName
    @author Vesa Oikonen
 */
imgformat imgFormatIdentify(
  /** IMG format as a string */
  const char *s
) {
  if(s==NULL || strlen(s)<1) return IMG_FORMAT_UNKNOWN;
  /* Try if string can be found directly in the table */
  for(int i=0; i<IMG_FORMAT_LAST; i++) {
    if(strcasecmp(img_format[i], s)==0) return i;
  }
  /* Format string is not following TPC standard, lets try something else */
  if(     strcasecmp(s, "bin")==0)              return IMG_FORMAT_FLAT;
  else if(strcasecmp(s, "v")==0)                return IMG_FORMAT_E7;
  else if(strcasecmp(s, "dcm")==0)              return IMG_FORMAT_DICOM;
  else if(strcasecmp(s, "ana")==0)              return IMG_FORMAT_ANA_L;
  else if(strcasecmp(s, "nii")==0)              return IMG_FORMAT_NIFTI_1S;
  else if(strcasecmp(s, "nifti")==0)            return IMG_FORMAT_NIFTI_1S;

  return IMG_FORMAT_UNKNOWN;
}
/*****************************************************************************/

/*****************************************************************************/
