/** @file elements.c
    @brief Element names and symbols.
    See: https://iupac.org/what-we-do/periodic-table-of-elements
 */
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <time.h>
#include <string.h>
/*****************************************************************************/
#include "tpcisotope.h"
/*****************************************************************************/

/*****************************************************************************/
/** List of element symbols, in the order of Atomic number (Z). */
static char *element_symbol[] = {
"Unknown",
"H",
"He",
"Li",
"Be",
"B",
"C",
"N",
"O",
"F",
"Ne",
"Na",
"Mg",
"Al",
"Si",
"P",
"S",
"Cl",
"Ar",
"K",
"Ca",
"Sc",
"Ti",
"V",
"Cr",
"Mn",
"Fe",
"Co",
"Ni",
"Cu",
"Zn",
"Ga",
"Ge",
"As",
"Se",
"Br",
"Kr",
"Rb",
"Sr",
"Y",
"Zr",
"Nb",
"Mo",
"Tc",
"Ru",
"Rh",
"Pd",
"Ag",
"Cd",
"In",
"Sn",
"Sb",
"Te",
"I",
"Xe",
"Cs",
"Ba",
"La",
"Ce",
"Pr",
"Nd",
"Pm",
"Sm",
"Eu",
"Gd",
"Tb",
"Dy",
"Ho",
"Er",
"Tm",
"Yb",
"Lu",
"Hf",
"Ta",
"W",
"Re",
"Os",
"Ir",
"Pt",
"Au",
"Hg",
"Tl",
"Pb",
"Bi",
"Po",
"At",
"Rn",
"Fr",
"Ra",
"Ac",
"Th",
"Pa",
"U",
"Np",
"Pu",
"Am",
"Cm",
"Bk",
"Cf",
"Es",
"Fm",
"Md",
"No",
"Lr",
"Rf",
"Db",
"Sg",
"Bh",
"Hs",
"Mt",
"Ds",
"Rg",
"Cn",
"Nh",
"Fl",
"Mc",
"Lv",
"Ts",
"Og",
0};
/*****************************************************************************/
/** List of element names, in the order of Atomic number (Z). */
static char *element_name[] = {
"Unknown",
"Hydrogen",
"Helium",
"Lithium",
"Beryllium",
"Boron",
"Carbon",
"Nitrogen",
"Oxygen",
"Fluorine",
"Neon",
"Sodium",
"Magnesium",
"Aluminium",
"Silicon",
"Phosphorus",
"Sulfur",
"Chlorine",
"Argon",
"Potassium",
"Calcium",
"Scandium",
"Titanium",
"Vanadium",
"Chromium",
"Manganese",
"Iron",
"Cobalt",
"Nickel",
"Copper",
"Zinc",
"Gallium",
"Germanium",
"Arsenic",
"Selenium",
"Bromine",
"Krypton",
"Rubidium",
"Strontium",
"Yttrium",
"Zirconium",
"Niobium",
"Molybdenum",
"Technetium",
"Ruthenium",
"Rhodium",
"Palladium",
"Silver",
"Cadmium",
"Indium",
"Tin",
"Antimony",
"Tellurium",
"Iodine",
"Xenon",
"Caesium",
"Barium",
"Lanthanum",
"Cerium",
"Praseodymium",
"Neodymium",
"Promethium",
"Samarium",
"Europium",
"Gadolinium",
"Terbium",
"Dysprosium",
"Holmium",
"Erbium",
"Thulium",
"Ytterbium",
"Lutetium",
"Hafnium",
"Tantalum",
"Tungsten",
"Rhenium",
"Osmium",
"Iridium",
"Platinum",
"Gold",
"Mercury",
"Thallium",
"Lead",
"Bismuth",
"Polonium",
"Astatine",
"Radon",
"Francium",
"Radium",
"Actinium",
"Thorium",
"Protactinium",
"Uranium",
"Neptunium",
"Plutonium",
"Americium",
"Curium",
"Berkelium",
"Californium",
"Einsteinium",
"Fermium",
"Mendelevium",
"Nobelium",
"Lawrencium",
"Rutherfordium",
"Dubnium",
"Seaborgium",
"Bohrium",
"Hassium",
"Meitnerium",
"Darmstadtium",
"Roentgenium",
"Copernicium",
"Nihonium",
"Flerovium",
"Moscovium",
"Livermorium",
"Tennessine",
"Oganesson",
0};
/*****************************************************************************/

/*****************************************************************************/
/** Return pointer to element name.
    @sa elementSymbol, elementIdentify
    @return pointer to the name string.
 */
char *elementName(
  /** Atomic number (Z) of the element. */
  unsigned short int z
) {
  if(z>MAX_ATOMIC_NUMBER) return(element_name[0]);
  return(element_name[z]);
}
/*****************************************************************************/

/*****************************************************************************/
/** Return pointer to element symbol.
    @sa elementName, elementIdentify
    @return pointer to the symbol string.
 */
char *elementSymbol(
  /** Atomic number (Z) of the element. */
  unsigned short int z
) {
  if(z>MAX_ATOMIC_NUMBER) return(element_name[0]);
  return(element_symbol[z]);
}
/*****************************************************************************/

/*****************************************************************************/
/** Identify the given string representation of element name or symbol.
   @sa elementName, elementSymbol, isotopeIdentify
   @return Element atomic number Z, or 0 if not identified.
 */
unsigned short int elementIdentify(
  /** Name or symbol of element, for example 'Carbon' or 'C'. */
  const char *str
) {
  if(str==NULL || strnlen(str, 5)<1) return(0); 

  /* First, search string from the list of element names */
  for(unsigned short int z=1; z<MAX_ATOMIC_NUMBER; z++)
    if(strcasecmp(str, element_name[z])==0) return(z);

  /* Then, search string from the list of element symbols */
  for(unsigned short int z=1; z<MAX_ATOMIC_NUMBER; z++)
    if(strcasecmp(str, element_symbol[z])==0) return(z);

  /* Not yet successful; then try to find element name as part of the string */
  for(unsigned short int z=1; z<MAX_ATOMIC_NUMBER; z++)
    if(strcasestr(str, element_name[z])!=NULL) return(z);

  /* Then try to find two-char element symbol as part of the string */
  for(unsigned short int z=1; z<MAX_ATOMIC_NUMBER; z++)
    if(strlen(element_symbol[z])>1 && strstr(str, element_symbol[z])!=NULL) return(z);
  /* Desperate last step... try to find one-char element symbol as part of the string */
  for(unsigned short int z=1; z<MAX_ATOMIC_NUMBER; z++)
    if(strlen(element_symbol[z])==1 && strstr(str, element_symbol[z])!=NULL) return(z);

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
