/** @file interpolate.c
    @brief Linear interpolation and integration.
    @todo Separate float version of this library, maybe lif.
 */
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <time.h>
#include <string.h>
/*****************************************************************************/
#include "tpcextensions.h"
/*****************************************************************************/
#include "tpcli.h"
/*****************************************************************************/

/*****************************************************************************/
/** @brief Find the initial x step size.
    @author Vesa Oikonen
    @return the first x[i+1]-x[i] that is > 0, or x[last] if not found,
     or 0 if even that was not possible.
 */
double liFirstStepSize(
  /** Array of x (time) values; must be sorted by ascending x, 
      but consecutive points may have the same x. Negative x is accepted.
      Array must not contain NaNs. */
  double *x,
  /** Number of samples in data. */
  const int nr
) {
  int i;
  double s, d;
  if(nr<1 || x==NULL) return 0.0;
  s=x[nr-1];
  for(i=1; i<nr; i++) {
    d=x[i]-x[i-1]; 
    if(d>1.0E-20) {s=d; break;} 
  }
  return s;
}
/*****************************************************************************/

/*****************************************************************************/
/** Create sample times with more frequent sampling in the beginning.
  @return Returns the number of samples.
  @sa liInterpolate, liInterpolateForPET, simC1, simBTAC
  */
unsigned int simSamples(
  /** Initial sampling frequency. */
  double initStep,
  /** Maximal sampling frequency which is never exceeded; enter 0 to set it automatically. */
  double maxStep,
  /** End time; last sample will usually be later. */
  double endTime,
  /** Sampling scheme:
    - Mode 0: 1s, 2s, 3s, 4s, ...
    - Mode 1: 10 x s, 5 x 2s, 2 x 5s, s*=10, ...
    - Mode 2: 20 x s, 10 x 2s, 5 x 5s, s*=10, ...
  */
  int mode,
  /** Allocated array for the samples; NULL if only computing the number of samples. */
  double *x
) {
  unsigned int n=0;
  if(initStep<=0.0 || endTime<=0.0 || endTime<initStep) return(n);
  if(maxStep<initStep) maxStep=endTime;

  double x1=0.0, s=initStep;
  if(mode==0) {
    while(x1<endTime) {
      if(x!=NULL) x[n]=x1;
      x1+=s; n++;
    }
    if(x!=NULL) x[n]=x1;
    n++;
  } else if(mode==1) {
    while(x1<endTime) {
      for(int i=0; i<10 && x1<endTime; i++, n++) {
        if(x!=NULL) x[n]=x1;
        x1+=s;
      }
      s*=2.0; if(s>maxStep) s=maxStep;
      for(int i=0; i<5 && x1<endTime; i++, n++) {
        if(x!=NULL) x[n]=x1;
        x1+=s;
      }
      s*=2.5; if(s>maxStep) s=maxStep;
      for(int i=0; i<2 && x1<endTime; i++, n++) {
        if(x!=NULL) x[n]=x1;
        x1+=s;
      }
      s*=2.0; if(s>maxStep) s=maxStep;
    }
    if(x!=NULL) x[n]=x1;
    n++;
  } else if(mode==2) {
    while(x1<endTime) {
      for(int i=0; i<20 && x1<endTime; i++, n++) {
        if(x!=NULL) x[n]=x1;
        x1+=s;
      }
      s*=2.0; if(s>maxStep) s=maxStep;
      for(int i=0; i<10 && x1<endTime; i++, n++) {
        if(x!=NULL) x[n]=x1;
        x1+=s;
      }
      s*=2.5; if(s>maxStep) s=maxStep;
      for(int i=0; i<5 && x1<endTime; i++, n++) {
        if(x!=NULL) x[n]=x1;
        x1+=s;
      }
      s*=2.0; if(s>maxStep) s=maxStep;
    }
    if(x!=NULL) x[n]=x1;
    n++;
  }
  return(n);
}
/*****************************************************************************/

/*****************************************************************************/
/** @brief Linear interpolation and/or integration with trapezoidal method.
  
    @details 
    PET data must be frequently sampled to get accurate results.
    AUC calculation is started from time 0 or first input sample time, whichever is smaller.
    Very simple data extrapolation can be optionally enabled.
    Results will be wrong if data contains NaNs.
    To get the same result as with interpolate() in the libtpcmodel, set se=3 and ee=1.
    
    @author Vesa Oikonen
    @return 0 if successful, otherwise >0. 
    @sa liInterpolateForPET, tacInterpolate, tacAUC
     
    @todo Compute tyi only if either of integrals is needed.
 */
int liInterpolate(
  /** Array of input data x (time) values; obligatory.
      Data must be sorted by ascending x, but consecutive points may have the same x.
      Negative x is accepted, AUCs are then started from the first x, otherwise from 0. 
      Array must not contain NaNs. */
  double *x,
  /** Array of input data y (concentration) values; obligatory.
      Array must not contain NaNs. */
  double *y,
  /** Number of samples in input data; obligatory. */
  const int nr,
  /** Array of output data x values; obligatory. Must be in ascending order.
      Negative x is accepted. Array must not contain NaNs. */
  double *newx,
  /** Array for interpolated (extrapolated) y values; NULL can be given if not needed. */
  double *newy,
  /** Array for integrals (AUCs); NULL can be given if not needed. */
  double *newyi,
  /** Arrays for 2nd integrals; NULL can be given if not needed. */
  double *newyii,
  /** Nr of samples in output data; obligatory. */
  const int newnr,
  /** Extrapolation setting for the start of data:
      - 0= assuming y=0 when newx<x[0]
      - 1= assuming y=y[0] when 0<=newx<x[0] and y=0 when newx<x[0]<0
      - 2= assuming y=0 when newx<=0 and x[0]>0, and
                    y=newx*y[0]/x[0] when 0<newx<x[0] and y[0]>0
      - 3= same as '2' if 2*x[0]-x[1]<=0, otherwise same as '0'
      - 4= same as '2' if 2*x[0]-x[1]<=0, otherwise assuming 
           y=0 when newx<=2*x[0]-x[1], and 
           y=(newx-(2*x[0]-x[1]))*y[0]/x[0] when (2*x[0]-x[1])<newx<x[0] and y[0]>0.
   */
  const int se,
  /** Extrapolation setting for the end of data:
      - 0= assuming y=0 when newx>x[nr-1]
      - 1= assuming y=y[nr-1] when newx>x[nr-1]
   */
  const int ee,
  /** Verbose level; if zero, then nothing is printed to stderr or stdout */
  const int verbose
) {
  if(verbose>0) printf("%s(x, y, %d, nx, ny, ni, ni2, %d, %d, %d)\n", __func__, nr, newnr, se, ee);
  /* Check for data */
  if(nr<1 || newnr<1) return 1;
  if(x==NULL || y==NULL || newx==NULL) return 1;

  int i, j;
  double ty, tyi, tyii;
  double ox1, ox2, oy1, oy2, oi1, oi2, oii1, oii2, dt, ndt;

  /* Initiate first two input samples */
  if(se==0) {
    if(x[0]<0.0) {ox1=ox2=x[0]; oy1=oy2=y[0];}
    else {ox1=0.0; ox2=x[0]; oy1=0.0; oy2=0.0;}
  } else if(se==1) {
    if(x[0]<0.0) {ox1=ox2=x[0]; oy1=oy2=y[0];}
    else {ox1=0.0; ox2=x[0]; oy1=oy2=y[0];}
  } else {
    ox1=ox2=x[0];
    oy1=oy2=y[0];
  }
  /* Extrapolate the initial phase with triangle... */
  if(ox1>0.0 && se>=2) {
    oy1=0.0;
    if(se==2) { // always
      ox1=0.0; 
    } else if(se==3 && y[0]>0.0) { // if y[0]>0 
      // and initial gap is not too big 
      if(liFirstStepSize(x, nr)-x[0] >= 1.0E-20)
        ox1=0.0;
    } else if(se==4 && y[0]>0.0) { // if y[0]>0 
      // if initial gap is not too big, start triangle from zero 
      if(liFirstStepSize(x, nr)-x[0] >= 1.0E-20)
        ox1=0.0;
      else
        // otherwise start triangle from guessed zero sample 
        ox1=x[0]-liFirstStepSize(x, nr);
    }
  }

  /* Integrals too */
  oi1=oii1=0.0;
  oi2=oi1 + (ox2-ox1)*(oy1+oy2)/2.0;
  oii2=oii1 + (ox2-ox1)*(oi1+oi2)/2.0;
  if(verbose>1) {
    printf("ox1=%g oy1=%g oi1=%g oii1=%g\n", ox1, oy1, oi1, oii1);
    printf("ox2=%g oy2=%g oi2=%g oii2=%g\n", ox2, oy2, oi2, oii2);
  }

  /* Set interpolated data before input data (even imaginary) to zero */
  if(verbose>2) printf("before input start\n");
  j=0; while(j<newnr && newx[j]<ox1) {
    ty=tyi=tyii=0.0; if(verbose>2) printf("  ndt=%g\n", ox1-newx[j]);
    if(verbose>2) printf("  j=%d newx=%g ty=%g tyi=%g tyii=%g\n", j, newx[j], ty, tyi, tyii);
    if(newy!=NULL) newy[j]=ty;
    if(newyi!=NULL) newyi[j]=tyi;
    if(newyii!=NULL) newyii[j]=tyii;
    j++;
  } 

  /* Set interpolated data between ox1 and ox2 */
  if(verbose>2) printf("before input start #2\n");
//  dt=ox2-ox1; if(dt>0.0) while(j<newnr && newx[j]<=ox2) {
  dt=ox2-ox1; if(dt>0.0) while(j<newnr && newx[j]<ox2) {
    ndt=newx[j]-ox1; if(verbose>2) printf("  ndt=%g\n", ndt);
    ty=((oy2-oy1)/dt)*ndt + oy1; if(newy!=NULL) newy[j]=ty;
    tyi=oi1 + 0.5*(ty+oy1)*ndt; if(newyi!=NULL) newyi[j]=tyi;
    if(newyii!=NULL) newyii[j]=tyii= oii1 + 0.5*(tyi+oi1)*ndt;
    if(verbose>2) printf("  j=%d newx=%g ty=%g tyi=%g tyii=%g\n", j, newx[j], ty, tyi, tyii);
    j++;
  }

  /* Go through input data, sample-by-sample */
  if(verbose>2) printf("inside input range\n");
  for(i=0; i<nr && j<newnr; i++) {

    ox1=ox2; oy1=oy2; oi1=oi2; oii1=oii2;
    ox2=x[i]; oy2=y[i];
    oi2=oi1 + (ox2-ox1)*(oy1+oy2)/2.0;
    oii2=oii1 + (ox2-ox1)*(oi1+oi2)/2.0;
    if(verbose>2) {
      printf("ox1=%g oy1=%g oi1=%g oii1=%g\n", ox1, oy1, oi1, oii1);
      printf("ox2=%g oy2=%g oi2=%g oii2=%g\n", ox2, oy2, oi2, oii2);
    }
    
    /* Calculate input sample distance */
    dt=ox2-ox1; if(dt<0.0) return 3; else if(dt==0.0) continue;

    /* Any need for interpolation between ox1 and ox2? */
    while(j<newnr && newx[j]<=ox2) {
      ndt=newx[j]-ox1;
      ty=((oy2-oy1)/dt)*ndt + oy1; if(newy!=NULL) newy[j]=ty;
      tyi=oi1 + 0.5*(ty+oy1)*ndt; if(newyi!=NULL) newyi[j]=tyi;
      if(newyii!=NULL) newyii[j]=tyii= oii1 + 0.5*(tyi+oi1)*ndt;
      if(verbose>4)
        printf("  j=%d newx=%g ndt=%g ty=%g tyi=%g tyii=%g\n", j, newx[j], ndt, ty, tyi, tyii);
      j++;
    }

  } // next input sample

  /* Set interpolated data after input data, assuming zero or steady input */
  if(verbose>2) printf("after input range\n");
  while(j<newnr) {
    ndt=newx[j]-ox2;
    if(ee==0) ty=0.0; else ty=oy2;
    if(ee==0) tyi=oi2; else tyi=oi2 + oy2*ndt;
    tyii=oii2 + 0.5*(tyi+oi2)*ndt;
    if(newy!=NULL) newy[j]=ty;
    if(newyi!=NULL) newyi[j]=tyi;
    if(newyii!=NULL) newyii[j]=tyii;
    if(verbose>4) printf("  j=%d newx=%g ty=%g tyi=%g tyii=%g\n", j, newx[j], ty, tyi, tyii);
    j++;
  }

  if(verbose>3) printf("%s() ready\n", __func__);
  return 0;
}
/*****************************************************************************/

/*****************************************************************************/
/** @brief Linear TAC interpolation and/or integration to PET frames.
  
    @details 
    PET data must be frequently sampled to get accurate results.
    AUC calculation is started from time 0 or first input sample time, whichever is smaller.
    Original data and new x values must be sorted by ascending x.
    Subsequent x values can have equal values, enabling the use of step functions.
    PET frames can overlap, but interpolation may then be slower.
    Negative x (time) values can be processed.
    Results will be wrong if data contains NaNs.
    Very simple data extrapolation can be optionally enabled.
    To get the same result as with interpolate4pet() in the libtpcmodel, set se=3 and ee=1.
    
    @author Vesa Oikonen
    @sa liInterpolate(), liIntegratePET(), tacInterpolateInto, tacInterpolate, tacAUC
    @return 0 if successful, otherwise >0. 
 */
int liInterpolateForPET(
  /** Array of input data x (time) values; obligatory.
      Data must be sorted by ascending x, but consecutive points may have the same x.
      Negative x is accepted, AUCs are then started from the first x, otherwise from 0. */
  double *x,
  /** Array of input data y (concentration) values; obligatory. */
  double *y,
  /** Number of samples in input data; obligatory. */
  const int nr,
  /** Array of PET frame start times for output data; obligatory.
      Must be in ascending order. Negative times are accepted. */
  double *newx1,
  /** Array of PET frame end times for output data; obligatory.
      Must be in ascending order. Negative times are accepted, but frame duration
      (x2-x1) must be >=0. */
  double *newx2,
  /** Array for mean concentration values for each PET frame; NULL can be given if not needed. */
  double *newy,
  /** Array for integrals (AUCs) at frame middle times;
      NULL can be given if not needed. */
  double *newyi,
  /** Arrays for 2nd integrals at frame middle times;
      NULL can be given if not needed. */
  double *newyii,
  /** Nr of samples in output data; obligatory. */
  const int newnr,
  /** Extrapolation setting for the start of data:
      - 0= assuming y=0 when newx<x[0]
      - 1= assuming y=y[0] when 0<=newx<x[0] and y=0 when newx<x[0]<0
      - 2= assuming y=0 when newx<=0 and x[0]>0, and
                    y=newx*y[0]/x[0] when 0<newx<x[0] and y[0]>0
      - 3= same as '2' if 2*x[0]-x[1]<=0, otherwise same as '0'
      - 4= same as '2' if 2*x[0]-x[1]<=0, otherwise assuming 
           y=0 when newx<=2*x[0]-x[1], and 
           y=(newx-(2*x[0]-x[1]))*y[0]/x[0] when (2*x[0]-x[1])<newx<x[0] and y[0]>0.
   */
  const int se,
  /** Extrapolation setting for the end of data:
      - 0= assuming y=0 when newx>x[nr-1]
      - 1= assuming y=y[nr-1] when newx>x[nr-1]
   */
  const int ee,
  /** Verbose level; if zero, then nothing is printed to stderr or stdout. */
  const int verbose
) {
  if(verbose>0) 
    printf("%s(x, y, %d, nx1, nx2, ny, ni, ni2, %d, %d, %d)\n", __func__, nr, newnr, se, ee);

  /* Check for data */
  if(nr<1 || newnr<1) return 1;
  if(x==NULL || y==NULL || newx1==NULL || newx2==NULL) return 1;
  /* Check that input data is not totally outside the input data */
  if(newx2[newnr-1]<=x[0] || newx1[0]>=x[nr-1]) return 3;
  
  /* Nothing to do if output arrays were not given */
  if(newy==NULL && newyi==NULL && newyii==NULL) return 0;

  int ret, fi, zeroframe=0, overlap=0;
  double fdur;
  
  /* Check frame lengths, also for overlap and zero length frames */
  for(fi=0; fi<newnr; fi++) {
    /* Calculate frame length */
    fdur=newx2[fi]-newx1[fi];
    /* If frame length is <0, that is an error */
    if(fdur<0.0) return 4;
    if(fdur==0.0) zeroframe++;
    /* Overlap? */
    if(fi>0 && newx2[fi-1]>newx1[fi]) overlap++;
  }
  if(verbose>2) {
    printf("overlap := %d\n", overlap);
    printf("zeroframe := %d\n", zeroframe);
  }

  /* Interpolate and integrate one frame at a time, if:
     -there are overlapping frames
     -there are frames of zero length
     -only few interpolated frames
     -if frame mean values are not needed
  */
  
  if(overlap>0 || zeroframe>0 || newnr<=3 || newy==NULL )
  {
    double petx[3], pety[3], petyi[3], petyii[3];

    if(verbose>1) printf("frame-by-frame interpolation/integration\n");
    for(fi=0; fi<newnr; fi++) {
      /* Set frame start, middle and end times */
      petx[0]=newx1[fi]; petx[2]=newx2[fi]; petx[1]=0.5*(petx[0]+petx[2]);
      /* Calculate frame length */
      fdur=petx[2]-petx[0];
      /* If frame length is <0, that is an error */
      if(fdur<0.0) return 4;
      /* If frame length is 0, then use direct interpolation */
      if(fdur==0.0) {
        ret=liInterpolate(x, y, nr, petx, pety, petyi, petyii, 1, se, ee, verbose-1);
        if(ret) return 10+ret;
        if(newy!=NULL) newy[fi]=petyi[0];
        if(newyi!=NULL) newyi[fi]=petyi[0];
        if(newyii!=NULL) newyii[fi]=petyii[0];
        continue;
      }
      /* Calculate integrals at frame start, middle, and end */
      ret=liInterpolate(x, y, nr, petx, NULL, petyi, petyii, 3, se, ee, verbose-1);
      if(ret) return 20+ret;
      /* Set output integrals, if required */
      if(newyi!=NULL) newyi[fi]=petyi[1];
      if(newyii!=NULL) newyii[fi]=petyii[1];
      /* Calculate frame mean, if required */
      if(newy!=NULL) newy[fi]=(petyi[2]-petyi[0])/fdur;
    } // next frame

  } else {

    if(verbose>1) printf("all-frames-at-once interpolation/integration\n");
    /* Set temp array */
    double *tp=NULL; tp=malloc(newnr*sizeof(double)); if(tp==NULL) return 7;
    /* Integrate at frame start times */
    ret=liInterpolate(x, y, nr, newx1, NULL, tp, NULL, newnr, se, ee, verbose-1);
    if(ret) {free(tp); return 10+ret;}
    /* Integrate at frame end times */
    ret=liInterpolate(x, y, nr, newx2, NULL, newy, NULL, newnr, se, ee, verbose-1);
    if(ret) {free(tp); return 10+ret;}
    /* Calculate average frame value */
    for(fi=0; fi<newnr; fi++)
      newy[fi]=(newy[fi]-tp[fi])/(newx2[fi]-newx1[fi]);
    /* Calculate integrals */
    if(newyi!=NULL || newyii!=NULL) {
      for(fi=0; fi<newnr; fi++) tp[fi]=0.5*(newx2[fi]+newx1[fi]);
      ret=liInterpolate(x, y, nr, tp, NULL, newyi, newyii, newnr, se, ee, verbose-1);
      if(ret) {free(tp); return 10+ret;}
    }
    free(tp);
  }

  if(verbose>1) printf("%s() ready\n", __func__);
  
  
  return 0;
}
/*****************************************************************************/

/*****************************************************************************/
/** @brief Calculate the y value (concentration) at PET frame middle time based on 
    dot-to-dot linear interpolation, assuming that original y values represent the mean value 
    during the frame, i.e. AUC(frame start - frame end time)/frame duration.
  
    @details
    This function must NOT be used for TAC interpolation, but only for computing PET TAC AUCs. 
    Original data and new x values must be sorted by ascending x.
    Subsequent x values can have equal values, enabling the use of step functions.
    PET frames can overlap, but interpolation may then be slower.
    Negative x (time) values can be processed.
    Results will be wrong if data contains NaNs.
    The beginning and end of true TAC is assumed to follow a line drawn based on the two first/last 
    TAC middle frame y values. 
    The first and last sample are always set to the original values. 
    
    @author Vesa Oikonen
    @sa liInterpolate(), liIntegratePET(), liInterpolateForPET()
    @return 0 if successful, otherwise >0. 
 */
int liFrameMidValue(
  /** Array of PET frame start times; obligatory.
      Data must be sorted by ascending x, but consecutive points may have the same x. 
      Negative x is accepted. */
  double *x1,
  /** Array of PET frame end times; obligatory.
      Data must be sorted by ascending x, but consecutive points may have the same x.
      Negative x is accepted, but frame duration (x2-x1) must be >=0. */
  double *x2,
  /** Array of original TAC y (concentration) values, representing the average concentration
      during the PET frame; obligatory. */
  double *y,
  /** Number of samples in the data; obligatory. */
  const size_t nr,
  /** Array for y (concentration) values at the frame middle time for each PET; obligatory. */
  double *newy,
  /** Verbose level; if zero, then nothing is printed to stderr or stdout. */
  const int verbose
) {
  if(verbose>0) printf("%s(x1, x2, y, %zu, ny, %d)\n", __func__, nr, verbose);

  /* Check for data */
  if(nr<1) return 1;
  if(x1==NULL || x2==NULL || y==NULL || newy==NULL) return 1;
  if(nr<3) {
    if(verbose>2) printf("less than 3 frames; nothing done.\n");
    for(size_t i=0; i<nr; i++) newy[i]=y[i];
    return 0; 
  }

  /* Calculate frame middle times and durations / 2.
     Check that frames durations are >=0.
     Check that frame times are in ascending order.
  */
  double hfdur[nr], x[nr];
  for(size_t i=0; i<nr; i++) {
    x[i]=0.5*(x1[i]+x2[i]);
    hfdur[i]=0.5*(x2[i]-x1[i]); 
    if(hfdur[i]<0.0) {
      if(verbose>0) fprintf(stderr, "negative frame length.\n");
      return 2;
    }
    if(i>0 && x[i]<x[i-1]) {
      if(verbose>0) fprintf(stderr, "frame times not ascending.\n");
      return 3;
    }
  }

  /* Compute data matrix for solving (very simply) the y values at frame middle
     times */
  double A[2][nr], B[nr], xdif;
  A[0][0]=1.0; A[1][0]=0.0; B[0]=y[0];
  for(size_t i=1; i<nr-1; i++) {
    xdif=x[i+1]-x[i];
    if(!(hfdur[i]>1.0E-100) || !(xdif>1.0E-100)) { // catches both zero and NaN
      A[0][i]=0.0; A[1][i]=0.0; B[i]=y[i];
    } else {
      A[0][i]=4.0*(x[i]-x[i-1])/hfdur[i] - (x[i+1]-x[i-1])/xdif;
      A[1][i]=(x[i]-x[i-1])/xdif;
      B[i]=4.0*y[i]*(x[i]-x[i-1])/hfdur[i];
    }
  }
  A[0][nr-1]=1.0; A[1][nr-1]=0.0; B[nr-1]=y[nr-1]; 
  if(verbose>4) {
    printf("A[1]\tA[2]\t\tB\n");
    for(size_t i=0; i<nr; i++) printf("%g\t%g\t\t%g\n", A[0][i], A[1][i], B[i]);
  }

  /* Solve */
  for(size_t i=1; i<nr-1; i++) {
    if(fabs(A[0][i])<1.0E-100) continue;
    A[0][i]-=A[1][i-1]; B[i]-=B[i-1];
    A[1][i]/=A[0][i]; B[i]/=A[0][i]; A[0][i]=1.0; 
  }
  if(verbose>5) {
    printf("A[1]\tA[2]\t\tB\n");
    for(size_t i=0; i<nr; i++) printf("%g\t%g\t\t%g\n", A[0][i], A[1][i], B[i]);
  }
  newy[0]=y[0]; newy[nr-1]=y[nr-1];
  for(size_t i=nr-2; i>0; i--) {
    newy[i]=B[i]-newy[i+1]*A[1][i];
    if(verbose>5) printf("X[%zu]=%g\n", i+1, newy[i]);
  }
  /* Set any NaNs to original values (which of course may be NaN as well) */
  for(size_t i=1; i<nr-1; i++) if(isnan(newy[i])) newy[i]=y[i];

  if(verbose>1) printf("%s() ready\n", __func__);
  return 0;
}
/*****************************************************************************/

/*****************************************************************************/





#if(0)
/*****************************************************************************/

/*****************************************************************************/
/** Calculates integrals of PET data at frame end times.

    Data does not have to be continuous, but it must be increasing in time.
    For faster performance in repetitive calls, allocate memory for integral[]
    even if it is not needed.
    If x1[0] is >0 AND x1[0] is <=(x2[0]-x1[0]),
    then the beginning is interpolated from (0, 0) to (x1[0], y1[0]*x1[0]/x)
    where x is midtime of the first frame.
\return Returns 0 if ok.
*/
int petintegrate(
  /** Array of frame start times */
  double *x1,
  /** Array of frame end times */
  double *x2,
  /** Array of y values (avg during each frame) */
  double *y,
  /** Nr of frames */
  int nr,
  /** Output: integral values at frame end times, or NULL */
  double *newyi,
  /** Output: 2nd integral values at frame end times, or NULL */
  double *newyii
) {
  int i, allocated=0;
  double *ti, x, a;


  if(INTEGR_TEST) printf("in petintegrate()\n");
  /* Check that data is increasing in time and is not (much) overlapping */
  if(nr<1 || x1[0]<0) return 1;
  for(i=0; i<nr; i++) if(x2[i]<x1[i]) return 2;
  for(i=1; i<nr; i++) if(x1[i]<=x1[i-1]) return 3;

  /* Allocate memory for temp data, if necessary */
  if(newyi==NULL) {
    allocated=1;
    ti=(double*)malloc(nr*sizeof(double)); if(ti==NULL) return 4;
  } else
    ti=newyi;

  /* Calculate the integral at the end of first frame */
  ti[0]=(x2[0]-x1[0])*y[0];
  /* If frame does not start at 0 time, add the area in the beginning */
  /* But only if the gap in the beginning is less than the lenght of the 
     first frame */
  if(x1[0]>0) {
    if(x1[0]<=x2[0]-x1[0]) {
       x=(x1[0]+x2[0])/2.0; 
       a=(x1[0]*(y[0]/x)*x1[0])/2.0; 
       ti[0]+=a;
    }
  }

  /* Calculate integrals at the ends of following frames */
  for(i=1; i<nr; i++) {
    /* Add the area of this frame to the previous integral */
    a=(x2[i]-x1[i])*y[i]; ti[i]=ti[i-1]+a;
    /* Check whether frames are contiguous */
    if(x1[i]==x2[i-1]) continue;
    /* When not, calculate the area of an imaginary frame */
    x=(x1[i]+x2[i-1])/2.0;
    a=(x1[i]-x2[i-1])*
      (y[i]-(y[i]-y[i-1])*(x2[i]+x1[i]-2.0*x)/(x2[i]+x1[i]-x2[i-1]-x1[i-1]));
    ti[i]+=a;
  }

  /* Copy integrals to output if required */
  if(allocated) for(i=0; i<nr; i++) newyi[i]=ti[i];

  /* Calculate 2nd integrals if required */
  if(newyii!=NULL) {
    newyii[0]=x2[0]*ti[0]/2.0;
    for(i=1; i<nr; i++)
      newyii[i]=newyii[i-1]+(x2[i]-x2[i-1])*(ti[i-1]+ti[i])/2.0;
  }
  
  /* Free memory */
  if(allocated) free((char*)ti);

  if(INTEGR_TEST) printf("out petintegrate()\n");
  return 0;
}

/* Float version of the same function */
int fpetintegrate(
  /** Array of frame start times */
  float *x1,
  /** Array of frame end times */
  float *x2,
  /** Array of y values (avg during each frame) */
  float *y,
  /** Nr of frames */
  int nr,
  /** Output: integral values at frame end times, or NULL */
  float *newyi,
  /** Output: 2nd integral values at frame end times, or NULL */
  float *newyii
) {
  int i, allocated=0;
  float *ti, x, a;


  if(INTEGR_TEST) printf("in fpetintegrate()\n");
  /* Check that data is increasing in time and is not (much) overlapping */
  if(nr<1 || x1[0]<0) return 1;
  for(i=0; i<nr; i++) if(x2[i]<x1[i]) return 2;
  for(i=1; i<nr; i++) if(x1[i]<=x1[i-1]) return 3;

  /* Allocate memory for temp data, if necessary */
  if(newyi==NULL) {
    allocated=1;
    ti=(float*)malloc(nr*sizeof(float)); if(ti==NULL) return 4;
  } else
    ti=newyi;

  /* Calculate the integral at the end of first frame */
  ti[0]=(x2[0]-x1[0])*y[0];
  /* If frame does not start at 0 time, add the area in the beginning */
  /* But only if the gap in the beginning is less than the lenght of the 
     first frame */
  if(x1[0]>0) {
    if(x1[0]<=x2[0]-x1[0]) {
       x=(x1[0]+x2[0])/2.0; 
       a=(x1[0]*(y[0]/x)*x1[0])/2.0; 
       ti[0]+=a;
    }
  }

  /* Calculate integrals at the ends of following frames */
  for(i=1; i<nr; i++) {
    /* Add the area of this frame to the previous integral */
    a=(x2[i]-x1[i])*y[i]; ti[i]=ti[i-1]+a;
    /* Check whether frames are contiguous */
    if(x1[i]==x2[i-1]) continue;
    /* When not, calculate the area of an imaginary frame */
    x=(x1[i]+x2[i-1])/2.0;
    a=(x1[i]-x2[i-1])*
      (y[i]-(y[i]-y[i-1])*(x2[i]+x1[i]-2.0*x)/(x2[i]+x1[i]-x2[i-1]-x1[i-1]));
    ti[i]+=a;
  }

  /* Copy integrals to output if required */
  if(allocated) for(i=0; i<nr; i++) newyi[i]=ti[i];

  /* Calculate 2nd integrals if required */
  if(newyii!=NULL) {
    newyii[0]=x2[0]*ti[0]/2.0;
    for(i=1; i<nr; i++)
      newyii[i]=newyii[i-1]+(x2[i]-x2[i-1])*(ti[i-1]+ti[i])/2.0;
  }
  
  /* Free memory */
  if(allocated) free((char*)ti);

  if(INTEGR_TEST) printf("out fpetintegrate()\n");
  return 0;
}
/*****************************************************************************/

/*****************************************************************************/
/** Interpolate and integrate TAC to PET frames.

  It is assumed, that original data is not from framed data, but that
  the values represent the actual value at specified time point, which
  allows the integration to be calculated dot-by-dot.

  If NULL is specified for *newy, *newyi and/or *newyii, their values are
  not calculated.

  Original data and new x values must be sorted by ascending x.
  Subsequent x values can have equal values, enabling the use of step functions.
  PET frames can overlap, but interpolation may then be slower.
  Negative x (time) values can be processed.
  If necessary, the data is extrapolated assuming that 1) y[inf]=y[nr-1] and
  2) if x[0]>0 and y[0]>0 and x[0]>x[1]-x[0], then an imaginary line is drawn
  from (0,0) to (x[0],y[0]).
*/
int interpolate4pet(
  /** Times of original data */
  double *x,
  /** Values of original data */
  double *y,
  /** Number of original data values */
  int nr,
  /** PET frame start times; frames may overlap */
  double *newx1,
  /** PET frame end times; frames may overlap */
  double *newx2, 
  /** Mean value during PET frame, or NULL if not needed;
   *  calculation may be faster if newyi is calculated too */
  double *newy,
  /** Integral at frame mid time, or NULL if not needed */
  double *newyi,
  /** 2nd integral at frame mid time, or NULL if not needed */
  double *newyii,
  /** Number of PET frames */
  int newnr
) {
  int ret, fi, overlap=0, zeroframe=0;
  double petx[3], pety[3], petyi[3], petyii[3], fdur;

  if(INTEGR_TEST) printf("in interpolate4pet()\n");

  /* Check for data */
  if(nr<1 || newnr<1) return 1;
  /* Check that programmer understood that outp data must've been allocated */
  if(newy==NULL && newyi==NULL && newyii==NULL) return 2;
  /* Check that input data is not totally outside the input data */
  if(newx2[newnr-1]<=x[0] || newx1[0]>=x[nr-1]) return 3;

  /* Check frame lengths, also for overlap and zero length frames */
  for(fi=0; fi<newnr; fi++) {
    /* Calculate frame length */
    fdur=newx2[fi]-newx1[fi];
    /* If frame length is <0, that is an error */
    if(fdur<0.0) return 4;
    if(fdur==0.0) zeroframe++;
    /* Overlap? */
    if(fi>0 && newx2[fi-1]>newx1[fi]) overlap++;
  }
  if(INTEGR_TEST>1) {
    printf("overlap := %d\n", overlap);
    printf("zeroframe := %d\n", zeroframe);
  }

  /* Interpolate and integrate one frame at a time, if there is:
     -overlap in frames
     -frames of zero length
     -only few interpolated frames
     -if only integrals are needed
     -if neither of integrals is needed, then there is no temp memory to
      do otherwise
  */
  if(overlap>0 || zeroframe>0 || newnr<=3 ||
     newy==NULL || (newyi==NULL && newyii==NULL) )
  {

    if(INTEGR_TEST>1) printf("frame-by-frame interpolation/integration\n");
    for(fi=0; fi<newnr; fi++) {
      /* Set frame start, middle and end times */
      petx[0]=newx1[fi]; petx[2]=newx2[fi]; petx[1]=0.5*(petx[0]+petx[2]);
      /* Calculate frame length */
      fdur=petx[2]-petx[0];
      /* If frame length is <0, that is an error */
      if(fdur<0.0) return 4;
      /* If frame length is 0, then use direct interpolation */
      if(fdur==0.0) {
        ret=interpolate(x, y, nr, petx, pety, petyi, petyii, 1);
        if(ret) return 10+ret;
        if(newy!=NULL) newy[fi]=petyi[0];
        if(newyi!=NULL) newyi[fi]=petyi[0];
        if(newyii!=NULL) newyii[fi]=petyii[0];
        continue;
      }
      /* Calculate integrals at frame start, middle, and end */
      ret=interpolate(x, y, nr, petx, NULL, petyi, petyii, 3);
      if(ret) return 20+ret;
      /* Set output integrals, if required */
      if(newyi!=NULL) newyi[fi]=petyi[1];
      if(newyii!=NULL) newyii[fi]=petyii[1];
      /* Calculate frame mean, if required */
      if(newy!=NULL) newy[fi]=(petyi[2]-petyi[0])/fdur;
    } // next frame

  } else {

    if(INTEGR_TEST>1) printf("all-frames-at-once interpolation/integration\n");
    /* Set temp array */
    double *tp; if(newyii!=NULL) tp=newyii; else tp=newyi;
    /* Integrate at frame start times */
    ret=interpolate(x, y, nr, newx1, NULL, tp, NULL, newnr);
    if(ret) return 10+ret;
    /* Integrate at frame end times */
    ret=interpolate(x, y, nr, newx2, NULL, newy, NULL, newnr);
    if(ret) return 10+ret;
    /* Calculate average frame value */
    for(fi=0; fi<newnr; fi++)
      newy[fi]=(newy[fi]-tp[fi])/(newx2[fi]-newx1[fi]);
    /* Calculate integrals */
    if(newyi!=NULL || newyii!=NULL) {
      for(fi=0; fi<newnr; fi++) newx1[fi]+=0.5*(newx2[fi]-newx1[fi]);
      ret=interpolate(x, y, nr, newx1, NULL, newyi, newyii, newnr);
      if(ret) return 10+ret;
      for(fi=0; fi<newnr; fi++) newx1[fi]-=(newx2[fi]-newx1[fi]);
    }
  }

  if(INTEGR_TEST) printf("out interpolate4pet()\n");
  return 0;
}

/** float version of interpolate4pet() */
int finterpolate4pet(
  /** Times of original data */
  float *x,
  /** Values of original data */
  float *y,
  /** Number of original data values */
  int nr,
  /** PET frame start times; frames may overlap */
  float *newx1,
  /** PET frame end times; frames may overlap */
  float *newx2, 
  /** Mean value during PET frame, or NULL if not needed;
   *  calculation may be faster if newyi is calculated too */
  float *newy,
  /** Integral at frame mid time, or NULL if not needed */
  float *newyi,
  /** 2nd integral at frame mid time, or NULL if not needed */
  float *newyii,
  /** Number of PET frames */
  int newnr
) {
  int ret, fi, overlap=0, zeroframe=0;
  float petx[3], pety[3], petyi[3], petyii[3], fdur;

  if(INTEGR_TEST) printf("in finterpolate4pet()\n");

  /* Check for data */
  if(nr<1 || newnr<1) return 1;
  /* Check that programmer understood that outp data must've been allocated */
  if(newy==NULL && newyi==NULL && newyii==NULL) return 2;
  /* Check that input data is not totally outside the input data */
  if(newx2[newnr-1]<=x[0] || newx1[0]>=x[nr-1]) return 3;

  /* Check frame lengths, also for overlap and zero length frames */
  for(fi=0; fi<newnr; fi++) {
    /* Calculate frame length */
    fdur=newx2[fi]-newx1[fi];
    /* If frame length is <0, that is an error */
    if(fdur<0.0) return 4;
    if(fdur==0.0) zeroframe++;
    /* Overlap? */
    if(fi>0 && newx2[fi-1]>newx1[fi]) overlap++;
  }
  if(INTEGR_TEST>1) {
    printf("overlap := %d\n", overlap);
    printf("zeroframe := %d\n", zeroframe);
  }

  /* Interpolate and integrate one frame at a time, if there is:
     -overlap in frames
     -frames of zero length
     -only few interpolated frames
     -if only integrals are needed
     -if neither of integrals is needed, then there is no temp memory to
      do otherwise
  */
  if(overlap>0 || zeroframe>0 || newnr<=3 ||
     newy==NULL || (newyi==NULL && newyii==NULL) )
  {

    if(INTEGR_TEST>1) printf("frame-by-frame interpolation/integration\n");
    for(fi=0; fi<newnr; fi++) {
      /* Set frame start, middle and end times */
      petx[0]=newx1[fi]; petx[2]=newx2[fi]; petx[1]=0.5*(petx[0]+petx[2]);
      /* Calculate frame length */
      fdur=petx[2]-petx[0];
      /* If frame length is <0, that is an error */
      if(fdur<0.0) return 4;
      /* If frame length is 0, then use direct interpolation */
      if(fdur==0.0) {
        ret=finterpolate(x, y, nr, petx, pety, petyi, petyii, 1);
        if(ret) return 10+ret;
        if(newy!=NULL) newy[fi]=petyi[0];
        if(newyi!=NULL) newyi[fi]=petyi[0];
        if(newyii!=NULL) newyii[fi]=petyii[0];
        continue;
      }
      /* Calculate integrals at frame start, middle, and end */
      ret=finterpolate(x, y, nr, petx, NULL, petyi, petyii, 3);
      if(ret) return 20+ret;
      /* Set output integrals, if required */
      if(newyi!=NULL) newyi[fi]=petyi[1];
      if(newyii!=NULL) newyii[fi]=petyii[1];
      /* Calculate frame mean, if required */
      if(newy!=NULL) newy[fi]=(petyi[2]-petyi[0])/fdur;
    } // next frame

  } else {

    if(INTEGR_TEST>1) printf("all-frames-at-once interpolation/integration\n");
    /* Set temp array */
    float *tp; if(newyii!=NULL) tp=newyii; else tp=newyi;
    /* Integrate at frame start times */
    ret=finterpolate(x, y, nr, newx1, NULL, tp, NULL, newnr);
    if(ret) return 10+ret;
    /* Integrate at frame end times */
    ret=finterpolate(x, y, nr, newx2, NULL, newy, NULL, newnr);
    if(ret) return 10+ret;
    /* Calculate average frame value */
    for(fi=0; fi<newnr; fi++)
      newy[fi]=(newy[fi]-tp[fi])/(newx2[fi]-newx1[fi]);
    /* Calculate integrals */
    if(newyi!=NULL || newyii!=NULL) {
      for(fi=0; fi<newnr; fi++) newx1[fi]+=0.5*(newx2[fi]-newx1[fi]);
      ret=finterpolate(x, y, nr, newx1, NULL, newyi, newyii, newnr);
      if(ret) return 10+ret;
      for(fi=0; fi<newnr; fi++) newx1[fi]-=(newx2[fi]-newx1[fi]);
    }

  }

  if(INTEGR_TEST) printf("out finterpolate4pet()\n");
  return 0;
}
/*****************************************************************************/

/*****************************************************************************/
/** Integrate PET TAC data to frame mid times.
    Any of output arrays may be set to NULL if that is not needed.
    Frames must be in ascending time order. Gaps and small overlap are allowed.
    If x1[0]>0 and x1[0]<=x2[0]-x1[0], then an imaginary line is 
    drawn from (0,0) to (x[0],y[0]).
\return Returns 0 if ok.
*/
int petintegral(
  /** frame start times */
  double *x1,
  /** frame end times */
  double *x2,
  /** avg value during frame */
  double *y,
  /** number of frames */
  int nr,
  /** integrals at frame mid time */
  double *ie,
  /** 2nd integrals at frame mid time */
  double *iie
) {
  int i;
  double x, last_x, last_x2, last_y, last_integral, box_integral, half_integral;
  double gap_integral, integral, integral2, frame_len, xdist, s;

  if(INTEGR_TEST) printf("in petintegral()\n");
  /* Check for data */
  if(nr<1 || nr<1) return(1);
  /* Check that programmer understood that output must've been allocated */
  if(ie==NULL && iie==NULL) return(2);

  /* Initiate values to zero */
  last_x=last_x2=last_y=last_integral=0.0;
  box_integral=integral=integral2=frame_len=0.0;

  for(i=0; i<nr; i++) {
    frame_len=x2[i]-x1[i]; if(frame_len<0.0) return(5);
    x=0.5*(x1[i]+x2[i]); xdist=x-last_x; if(last_x>0.0 && xdist<=0.0) return(6);
    if(x<0) {
      if(ie!=NULL) ie[i]=integral; if(iie!=NULL) iie[i]=integral2;
      continue;
    }
    s=(y[i]-last_y)/xdist; /* slope */
    /*If there is a big gap in the beginning it is eliminated */
    if(i==0)if(x1[0]>x2[0]-x1[0]){last_x2=last_x=x1[0];}
    /*Integral of a possible gap between frames*/
    gap_integral=(x1[i]-last_x2)*(last_y+s*((last_x2+x1[i])/2.0-last_x)); 
    /*Integral from the beginning of the frame to the middle of the frame */
    half_integral=(x-x1[i])*(last_y+s*((x1[i]+x)/2.0-last_x));
    integral=box_integral+gap_integral+half_integral; 
    /* half_integral is not added to box because it is more accurate to 
       increment integrals of full frames */
    box_integral+=gap_integral+frame_len*y[i];
    integral2+=xdist*(integral+last_integral)*0.5;
    if(ie!=NULL) ie[i]=integral; if(iie!=NULL) iie[i]=integral2;
    last_x=x; last_x2=x2[i];
    last_y=y[i]; last_integral=integral;
  }

  if(INTEGR_TEST) printf("out petintegral()\n");
  return(0);
}

/** float version of petintegral() */
int fpetintegral(
  /** frame start times */
  float *x1,
  /** frame end times */
  float *x2,
  /** avg value during frame */
  float *y,
  /** number of frames */
  int nr,
  /** integrals at frame mid time */
  float *ie,
  /** 2nd integrals at frame mid time */
  float *iie
) {
  int i;
  float x, last_x, last_x2, last_y, last_integral, box_integral, half_integral;
  float gap_integral, integral, integral2, frame_len, xdist, s;

  if(INTEGR_TEST) printf("in fpetintegral()\n");
  /* Check for data */
  if(nr<1 || nr<1) return(1);
  /* Check that programmer understood that output must've been allocated */
  if(ie==NULL && iie==NULL) return(2);

  /* Initiate values to zero */
  last_x=last_x2=last_y=last_integral=0.0;
  box_integral=gap_integral=integral=integral2=frame_len=0.0;

  for(i=0; i<nr; i++) {
    frame_len=x2[i]-x1[i]; if(frame_len<0.0) return(5);
    x=0.5*(x1[i]+x2[i]); xdist=x-last_x; if(last_x>0.0 && xdist<=0.0) return(6);
    if(x<0) {
      if(ie!=NULL) ie[i]=integral; if(iie!=NULL) iie[i]=integral2;
      continue;
    }
    s=(y[i]-last_y)/xdist; /* slope */
    /*If there is a big gap in the beginning it is eliminated */
    if(i==0)if(x1[0]>x2[0]-x1[0]){last_x2=last_x=x1[0];}
    /*Integral of a possible gap between frames*/
    gap_integral=(x1[i]-last_x2)*(last_y+s*((last_x2+x1[i])/2.0-last_x));
    /*Integral from the beginning of the frame i to the middle of the frame */
    half_integral=(x-x1[i])*(last_y+s*((x1[i]+x)/2.0-last_x));
    integral=box_integral+gap_integral+half_integral; 
    /* half_integral is not added to box because it is more accurate to 
       sum integrals of full frames */
    box_integral+=gap_integral+frame_len*y[i];
    integral2+=xdist*(integral+last_integral)*0.5;
    if(ie!=NULL) ie[i]=integral; if(iie!=NULL) iie[i]=integral2;
    last_x=x; last_x2=x2[i];
    last_y=y[i]; last_integral=integral;
  }

  if(INTEGR_TEST) printf("out fpetintegral()\n");
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Integrate PET TAC data to frame end times.
    Any of output arrays may be set to NULL if that is not needed.
    Frames must be in ascending time order. Gaps and small overlap are allowed.
    If x1[0]>0 and x1[0]<=x2[0]-x1[0], then an imaginary line is 
    drawn from (0,0) to (x[0],y[0]).
\return Returns 0 if ok.
*/
int petintegrate2fe(
  /** frame start times */
  double *x1,
  /** frame end times */
  double *x2,
  /** avg value during frame */
  double *y,
  /** number of frames */
  int nr,
  /** values at frame end time */
  double *e,
  /** integrals at frame end time */
  double *ie,
  /** 2nd integrals at frame end time */
  double *iie
) {
  int i;
  double x, last_x, last_x2, last_y, last_integral;
  double value, integral, integral2, frame_len, xdist, s;

  if(INTEGR_TEST) printf("in petintegrate2fe()\n");
  /* Check for data */
  if(nr<1 || nr<1) return(1);
  /* Check that programmer understood that output must've been allocated */
  if(e==NULL && ie==NULL && iie==NULL) return(2);

  /* Initiate values to zero */
  last_x=last_x2=last_y=last_integral=value=integral=integral2=frame_len=s=0.0;

  for(i=0; i<nr; i++) {
    frame_len=x2[i]-x1[i]; if(frame_len<0.0) return(5);
    x=0.5*(x1[i]+x2[i]); xdist=x-last_x; if(last_x>0.0 && xdist<=0.0) return(6);
    if(x<0) {
      if(e!=NULL) e[i]=value; if(ie!=NULL) ie[i]=integral;
      if(iie!=NULL) iie[i]=integral2;
      continue;
    }
    s=(y[i]-last_y)/xdist; /* slope between x[i-1] and x[i] */
    /* If there is a big gap in the beginning, it is eliminated */
    if(i==0 && x1[0]>x2[0]-x1[0]) {last_x2=last_x=x1[0];}
    integral+=(x1[i]-last_x2)*(last_y+s*((last_x2+x1[i])/2.0-last_x)); /*gap*/
    integral+=frame_len*y[i];
    integral2+=(x2[i]-last_x2)*(integral+last_integral)*0.5;
    if(e!=NULL && i>0) {
      value=last_y+s*(last_x2-last_x); /* value at previous frame end */
      e[i-1]=value;
    }
    if(ie!=NULL) ie[i]=integral; if(iie!=NULL) iie[i]=integral2;
    last_x=x; last_x2=x2[i]; last_y=y[i]; last_integral=integral;
  }
  if(e!=NULL) {
    value=last_y+s*(last_x2-last_x); /* Value for the last frame */
    e[i-1]=value;
  }

  if(INTEGR_TEST) printf("out petintegrate2fe()\n");
  return(0);
}

/** float version of petintegrate2fe() */
int fpetintegrate2fe(
  /** frame start times */
  float *x1,
  /** frame end times */
  float *x2,
  /** avg value during frame */
  float *y,
  /** number of frames */
  int nr,
  /** values at frame end time */
  float *e,
  /** integrals at frame end time */
  float *ie,
  /** 2nd integrals at frame end time */
  float *iie
) {
  int i;
  float x, last_x, last_x2, last_y, last_integral;
  float value, integral, integral2, frame_len, xdist, s;

  if(INTEGR_TEST) printf("in fpetintegrate2fe()\n");
  /* Check for data */
  if(nr<1 || nr<1) return(1);
  /* Check that programmer understood that output must've been allocated */
  if(e==NULL && ie==NULL && iie==NULL) return(2);

  /* Initiate values to zero */
  last_x=last_x2=last_y=last_integral=value=integral=integral2=frame_len=s=0.0;

  for(i=0; i<nr; i++) {
    frame_len=x2[i]-x1[i]; if(frame_len<0.0) return(5);
    x=0.5*(x1[i]+x2[i]); xdist=x-last_x; if(last_x>0.0 && xdist<=0.0) return(6);
    if(x<0) {
      if(e!=NULL) e[i]=value; if(ie!=NULL) ie[i]=integral;
      if(iie!=NULL) iie[i]=integral2;
      continue;
    }
    s=(y[i]-last_y)/xdist; /* slope between x[i-1] and x[i] */
    /* If there is a big gap in the beginning, it is eliminated */
    if(i==0 && x1[0]>x2[0]-x1[0]) {last_x2=last_x=x1[0];}
    integral+=(x1[i]-last_x2)*(last_y+s*((last_x2+x1[i])/2.0-last_x)); /*gap*/
    integral+=frame_len*y[i];
    integral2+=(x2[i]-last_x2)*(integral+last_integral)*0.5;
    if(e!=NULL && i>0) {
      value=last_y+s*(last_x2-last_x); /* value at previous frame end */
      e[i-1]=value;
    }
    if(ie!=NULL) ie[i]=integral; if(iie!=NULL) iie[i]=integral2;
    last_x=x; last_x2=x2[i]; last_y=y[i]; last_integral=integral;
  }
  if(e!=NULL) {
    value=last_y+s*(last_x2-last_x); /* Value for the last frame */
    e[i-1]=value;
  }

  if(INTEGR_TEST) printf("out fpetintegrate2fe()\n");
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
#endif
