/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include "tpcextensions.h"
#include "test_tpcli.h"
/*****************************************************************************/

/*****************************************************************************/
int test_liIntegrate(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }

  liIntegrate(NULL, NULL, 0, NULL, 0, 1);
  if(verbose>1) printf("did not segfault, that is good.\n");

  {
  if(verbose>1) printf("Same test as for integrate() in libtpcmodel:\n");
  int ret, nr=10;
  double xin[]={1, 2, 3, 4, 5, 6, 7, 8, 9, 10};
  double yin[]={1, 1, 1, 1, 1, 1, 1, 1, 1, 1};
  double iout[nr];
  ret=liIntegrate(xin, yin, nr, iout, 3, verbose-10);
  if(ret!=0) return 11;
  if(!doubleMatch(9.5, iout[nr-1], 1.0E-10)) return 12;
  }
  
  {
  if(verbose>1) printf("no extrapolation #1:\n");
  int ret, i, nr=2;
  double xin[]={0, 1};
  double yin[]={1, 2};
  double iout[nr];
  double ci[]={0, 1.5};
  ret=liIntegrate(xin, yin, nr, iout, 0, verbose-10);
  if(ret!=0) return 21;
  for(i=0; i<nr; i++) {
    if(!doubleMatch(ci[i], iout[i], 1.0E-10)) return 22;
  }

  if(verbose>1) printf("no extrapolation #2:\n");
  ret=liIntegrate(xin, yin, nr, iout, 1, verbose-10);
  if(ret!=0) return 23;
  for(i=0; i<nr; i++) {
    if(!doubleMatch(ci[i], iout[i], 1.0E-10)) return 24;
  }

  if(verbose>1) printf("no extrapolation #3:\n");
  ret=liIntegrate(xin, yin, nr, iout, 2, verbose-10);
  if(ret!=0) return 25;
  for(i=0; i<nr; i++) {
    if(!doubleMatch(ci[i], iout[i], 1.0E-10)) return 26;
  }

  if(verbose>1) printf("no extrapolation #4:\n");
  ret=liIntegrate(xin, yin, nr, iout, 3, verbose-10);
  if(ret!=0) return 27;
  for(i=0; i<nr; i++) {
    if(!doubleMatch(ci[i], iout[i], 1.0E-10)) return 28;
  }

  if(verbose>1) printf("no extrapolation #5:\n");
  ret=liIntegrate(xin, yin, nr, iout, 4, verbose-10);
  if(ret!=0) return 29;
  for(i=0; i<nr; i++) {
    if(!doubleMatch(ci[i], iout[i], 1.0E-10)) return 30;
  }

  }

  
  {
  if(verbose>1) printf("negative time #1:\n");
  int ret, i, nr=3;
  double xin[]={-1, 0, 1};
  double yin[]={1, 1, 2};
  double iout[nr];
  double ci[]={0, 1, 2.5};
  ret=liIntegrate(xin, yin, nr, iout, 0, verbose-10);
  if(ret!=0) return 101;
  for(i=0; i<nr; i++) {
    if(!doubleMatch(ci[i], iout[i], 1.0E-10)) return 102;
  }

  if(verbose>1) printf("negative time #2:\n");
  ret=liIntegrate(xin, yin, nr, iout, 1, verbose-10);
  if(ret!=0) return 103;
  for(i=0; i<nr; i++) {
    if(!doubleMatch(ci[i], iout[i], 1.0E-10)) return 104;
  }

  if(verbose>1) printf("negative time #3:\n");
  ret=liIntegrate(xin, yin, nr, iout, 2, verbose-10);
  if(ret!=0) return 105;
  for(i=0; i<nr; i++) {
    if(!doubleMatch(ci[i], iout[i], 1.0E-10)) return 106;
  }

  if(verbose>1) printf("negative time #4:\n");
  ret=liIntegrate(xin, yin, nr, iout, 3, verbose-10);
  if(ret!=0) return 107;
  for(i=0; i<nr; i++) {
    if(!doubleMatch(ci[i], iout[i], 1.0E-10)) return 108;
  }

  if(verbose>1) printf("negative time #5:\n");
  ret=liIntegrate(xin, yin, nr, iout, 4, verbose-10);
  if(ret!=0) return 109;
  for(i=0; i<nr; i++) {
    if(!doubleMatch(ci[i], iout[i], 1.0E-10)) return 110;
  }
  }

  
  {
  if(verbose>1) printf("small gap in time #1:\n");
  int ret, i, nr=2;
  double xin[]={1, 3};
  double yin[]={1, 3};
  double iout[nr];
  double ci[]={0, 4.0};
  ret=liIntegrate(xin, yin, nr, iout, 0, verbose-10);
  if(ret!=0) return 201;
  for(i=0; i<nr; i++) {
    if(!doubleMatch(ci[i], iout[i], 1.0E-10)) return 202;
  }

  if(verbose>1) printf("small gap in time #2:\n");
  ci[0]=1.0; ci[1]=5.0;
  ret=liIntegrate(xin, yin, nr, iout, 1, verbose-10);
  if(ret!=0) return 203;
  for(i=0; i<nr; i++) {
    if(!doubleMatch(ci[i], iout[i], 1.0E-10)) return 204;
  }

  if(verbose>1) printf("small gap in time #3:\n");
  ci[0]=0.5; ci[1]=4.5;
  ret=liIntegrate(xin, yin, nr, iout, 2, verbose-10);
  if(ret!=0) return 205;
  for(i=0; i<nr; i++) {
    if(!doubleMatch(ci[i], iout[i], 1.0E-10)) return 206;
  }

  if(verbose>1) printf("small gap in time #4:\n");
  ret=liIntegrate(xin, yin, nr, iout, 3, verbose-10);
  if(ret!=0) return 207;
  for(i=0; i<nr; i++) {
    if(!doubleMatch(ci[i], iout[i], 1.0E-10)) return 208;
  }

  if(verbose>1) printf("small gap in time #5:\n");
  ret=liIntegrate(xin, yin, nr, iout, 4, verbose-10);
  if(ret!=0) return 209;
  for(i=0; i<nr; i++) {
    if(!doubleMatch(ci[i], iout[i], 1.0E-10)) return 210;
  }
  }


  
  {
  if(verbose>1) printf("large gap in time #1:\n");
  int ret, i, nr=2;
  double xin[]={2, 3};
  double yin[]={2, 3};
  double iout[nr];
  double ci[]={0, 2.5};
  ret=liIntegrate(xin, yin, nr, iout, 0, verbose-10);
  if(ret!=0) return 301;
  for(i=0; i<nr; i++) {
    if(!doubleMatch(ci[i], iout[i], 1.0E-10)) return 302;
  }

  if(verbose>1) printf("large gap in time #2:\n");
  ci[0]=4.0; ci[1]=6.5;
  ret=liIntegrate(xin, yin, nr, iout, 1, verbose-10);
  if(ret!=0) return 303;
  for(i=0; i<nr; i++) {
    if(!doubleMatch(ci[i], iout[i], 1.0E-10)) return 304;
  }

  if(verbose>1) printf("large gap in time #3:\n");
  ci[0]=2.0; ci[1]=4.5;
  ret=liIntegrate(xin, yin, nr, iout, 2, verbose-10);
  if(ret!=0) return 305;
  for(i=0; i<nr; i++) {
    if(!doubleMatch(ci[i], iout[i], 1.0E-10)) return 306;
  }

  if(verbose>1) printf("large gap in time #4:\n");
  ci[0]=0.0; ci[1]=2.5;
  ret=liIntegrate(xin, yin, nr, iout, 3, verbose-10);
  if(ret!=0) return 307;
  for(i=0; i<nr; i++) {
    if(!doubleMatch(ci[i], iout[i], 1.0E-10)) return 308;
  }

  if(verbose>1) printf("large gap in time #5:\n");
  xin[0]=3.0; xin[1]=4.0;
  ci[0]=1.0; ci[1]=3.5;
  ret=liIntegrate(xin, yin, nr, iout, 4, verbose-10);
  if(ret!=0) return 309;
  for(i=0; i<nr; i++) {
    if(!doubleMatch(ci[i], iout[i], 1.0E-10)) return 310;
  }
  }

  {
  if(verbose>1) printf("realistic PET TTACs:\n");
  int ret;
  PAR par; parInit(&par); parExampleTTACs(&par, 0, NULL);
  int n=14, i, j;

  double x[]={0.5,1.5,2.5,3.5,4.5,5.5,7.,9.,12.5,17.5,25.,35.,45.,55.};
  double y[n], ci[n], yi[n];
  for(i=ret=0; i<par.tacNr && ret==0; i++) {
    if(verbose>2) printf("\n --- %s ---\n", par.r[i].name);
    /* compute y values */
    ret=mfEvalY(modelCode(par.r[i].model), modelParNr(par.r[i].model),
                par.r[i].p, n, x, y, verbose-20);
    if(ret) break;
    /* compute correct integrals */
    ret=mfEvalInt(modelCode(par.r[i].model), modelParNr(par.r[i].model),
                  par.r[i].p, n, x, ci, verbose-20);
    if(ret) break;
    /* compute integrals using the function to be tested with se=2 */
    if(verbose>3) printf("se=2\n");
    ret=liIntegrate(x, y, n, yi, 2, verbose-10);
    if(ret) break;
    if(verbose>6) {
      for(j=0; j<n; j++)
        printf("%6.1f  %10.3e  %10.3e  %10.3e\n", x[j], y[j], ci[j], yi[j]);
    }
    for(j=0; j<n; j++) {
      if(doubleMatch(ci[j], yi[j], 5.0)) continue;
      if(doubleMatchRel(ci[j], yi[j], 0.1)) continue;
      ret=999; break;
    }
    if(ret) break;
    /* compute integrals using the function to be tested with se=3 */
    if(verbose>3) printf("se=3\n");
    ret=liIntegrate(x, y, n, yi, 3, verbose-10);
    if(ret) break;
    if(verbose>6) {
      for(j=0; j<n; j++)
        printf("%6.1f  %10.3e  %10.3e  %10.3e\n", x[j], y[j], ci[j], yi[j]);
    }
    for(j=0; j<n; j++) {
      if(doubleMatch(ci[j], yi[j], 5.0)) continue;
      if(doubleMatchRel(ci[j], yi[j], 0.1)) continue;
      ret=999; break;
    }
    if(ret) break;
    /* compute integrals using the function to be tested with se=4 */
    if(verbose>3) printf("se=4\n");
    ret=liIntegrate(x, y, n, yi, 4, verbose-10);
    if(ret) break;
    if(verbose>6) {
      for(j=0; j<n; j++)
        printf("%6.1f  %10.3e  %10.3e  %10.3e\n", x[j], y[j], ci[j], yi[j]);
    }
    for(j=0; j<n; j++) {
      if(doubleMatch(ci[j], yi[j], 5.0)) continue;
      if(doubleMatchRel(ci[j], yi[j], 0.1)) continue;
      ret=999; break;
    }
    if(ret) break;
  }
  if(ret) {parFree(&par); return(1001);}


  if(verbose>1) printf("the same with more frequent samples:\n");
  double x2[]={0.001,0.05,0.1,0.2,0.3,0.5,0.75,1.0,1.25,1.50,1.75,2.0,2.5,3.0};
  for(i=ret=0; i<par.tacNr && ret==0; i++) {
    if(verbose>2) printf("\n --- %s ---\n", par.r[i].name);
    /* compute y values */
    ret=mfEvalY(modelCode(par.r[i].model), modelParNr(par.r[i].model),
                par.r[i].p, n, x2, y, verbose-20);
    if(ret) break;
    /* compute correct integrals */
    ret=mfEvalInt(modelCode(par.r[i].model), modelParNr(par.r[i].model),
                  par.r[i].p, n, x2, ci, verbose-20);
    if(ret) break;
    /* compute integrals using the function to be tested with se=2 */
    if(verbose>3) printf("se=2\n");
    ret=liIntegrate(x2, y, n, yi, 2, verbose-10);
    if(ret) break;
    if(verbose>8) {
      for(j=0; j<n; j++)
        printf("%7.3f  %10.3e  %10.3e  %10.3e\n", x2[j], y[j], ci[j], yi[j]);
    }
    for(j=0; j<n; j++) {
      if(doubleMatch(ci[j], yi[j], 1.0E-03)) continue;
      if(doubleMatchRel(ci[j], yi[j], 0.02)) continue;
      if(verbose>1) printf("percentual difference (%d) := %g\n", 
                           1+j, 100.*(yi[j]-ci[j])/ci[j]);
      ret=999; break;
    }
    if(ret) break;
    /* compute integrals using the function to be tested with se=3 */
    if(verbose>3) printf("se=3\n");
    ret=liIntegrate(x2, y, n, yi, 3, verbose-10);
    if(ret) break;
    if(verbose>8) {
      for(j=0; j<n; j++)
        printf("%7.3f  %10.3e  %10.3e  %10.3e\n", x2[j], y[j], ci[j], yi[j]);
    }
    for(j=0; j<n; j++) {
      if(doubleMatch(ci[j], yi[j], 1.0E-03)) continue;
      if(doubleMatchRel(ci[j], yi[j], 0.02)) continue;
      if(verbose>1) printf("percentual difference (%d) := %g\n", 
                           1+j, 100.*(yi[j]-ci[j])/ci[j]);
      ret=999; break;
    }
    if(ret) break;
    /* compute integrals using the function to be tested with se=4 */
    if(verbose>3) printf("se=4\n");
    ret=liIntegrate(x2, y, n, yi, 4, verbose-10);
    if(ret) break;
    if(verbose>8) {
      for(j=0; j<n; j++)
        printf("%7.3f  %10.3e  %10.3e  %10.3e\n", x2[j], y[j], ci[j], yi[j]);
    }
    for(j=0; j<n; j++) {
      if(doubleMatch(ci[j], yi[j], 1.0E-03)) continue;
      if(doubleMatchRel(ci[j], yi[j], 0.02)) continue;
      if(verbose>1) printf("percentual difference (%d) := %g\n", 
                           1+j, 100.*(yi[j]-ci[j])/ci[j]);
      ret=999; break;
    }
    if(ret) break;
  }
  if(ret) {parFree(&par); return(1002);}

   
  parFree(&par);
  }  

  
  fflush(stderr);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
int test_liIntegratePET(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }

  liIntegratePET(NULL, NULL, NULL, 0, NULL, NULL, 1);
  if(verbose>1) printf("did not segfault, that is good.\n");



  {
  if(verbose>1) printf("\n\nrealistic PET TTACs with small delay:\n");
  int ret, i, j;
  PAR par; parInit(&par); parExampleTTACs(&par, 0, NULL);
  for(i=0; i<par.tacNr; i++) // set increasing delay to fengm2
    par.r[i].p[modelParNr(par.r[i].model)-1]=0.05+0.05*i;
  int n=15;
  double x[]={0.25,0.75,1.5,2.5,3.5,4.5,5.5,7.,9.0,12.5,17.5,25.,35.,45.,55.};
  double x1[]={0.0,0.5, 1.0,2.0,3.0,4.0,5.0,6.,8.0,10.0,15.0,20.,30.,40.,50.};
  double x2[]={0.5,1.0, 2.0,3.0,4.0,5.0,6.0,8.,10.,15.0,20.0,30.,40.,50.,60.};
  double y[n], ci[n], cii[n], yi[n], yii[n];
  for(i=ret=0; i<par.tacNr && ret==0; i++) {
    if(verbose>2) printf("\n --- %s ---\n", par.r[i].name);
    /* compute y values */
    ret=mfEvalFrameY(modelCode(par.r[i].model), modelParNr(par.r[i].model),
                par.r[i].p, n, x1, x2, y, verbose-20);
    if(ret) break;
    /* compute correct integrals */
    ret=mfEvalInt(modelCode(par.r[i].model), modelParNr(par.r[i].model),
                  par.r[i].p, n, x, ci, verbose-20);
    if(ret) break;
    ret=mfEval2ndInt(modelCode(par.r[i].model), modelParNr(par.r[i].model),
                     par.r[i].p, n, x, cii, verbose-20);
    if(ret) break;
    /* compute integrals using the function to be tested */
    ret=liIntegratePET(x1, x2, y, n, yi, yii, verbose-10);
    if(ret) break;
    if(verbose>1) {
      for(j=0; j<n; j++) printf("%6.2f  %10.3e  %10.3e %10.3e  %10.3e %10.3e\n", 
        x[j], y[j], ci[j], yi[j], cii[j], yii[j]);
    }
    for(j=0; j<n; j++) {
      if(doubleMatch(ci[j], yi[j], 10.0)) continue;
      if(doubleMatchRel(ci[j], yi[j], 3.0E-02)) continue;
      if(verbose>1) printf("percentual i difference (%d) := %g\n", 
                           1+j, 100.*(yi[j]-ci[j])/ci[j]);
      ret=999; break;
    }
    if(ret) break;
    for(j=0; j<n; j++) {
      if(doubleMatch(cii[j], yii[j], 5.0)) continue;
      if(doubleMatchRel(cii[j], yii[j], 3.0E-02)) continue;
      if(verbose>1) printf("percentual ii difference (%d) := %g\n", 
                           1+j, 100.*(yii[j]-cii[j])/cii[j]);
      ret=999; break;
    }
    if(ret) break;
  }
  if(ret) {parFree(&par); return(10);}
   
  parFree(&par);
  }  



  {
  if(verbose>1) printf("\n\nrealistic PET TTACs with short frames:\n");
  int ret, i, j;
  PAR par; parInit(&par); parExampleTTACs(&par, 0, NULL);
  int n=6001;
  double x[n], x1[n], x2[n], fdur=0.01;
  double y[n], ci[n], cii[n], yi[n], yii[n];
  i=0; x1[i]=0.0; x2[i]=x1[i]+fdur; x[i]=0.5*(x1[i]+x2[i]);
  for(i=1; i<n; i++) {
    x1[i]=x2[i-1]; x2[i]=x1[i]+fdur; x[i]=0.5*(x1[i]+x2[i]);
  }
  for(i=ret=0; i<par.tacNr && ret==0; i++) {
    if(verbose>2) printf("\n --- %s ---\n", par.r[i].name);
    /* compute y values */
    ret=mfEvalFrameY(modelCode(par.r[i].model), modelParNr(par.r[i].model),
                par.r[i].p, n, x1, x2, y, verbose-20);
    if(ret) break;
    /* compute correct integrals */
    ret=mfEvalInt(modelCode(par.r[i].model), modelParNr(par.r[i].model),
                  par.r[i].p, n, x, ci, verbose-20);
    if(ret) break;
    ret=mfEval2ndInt(modelCode(par.r[i].model), modelParNr(par.r[i].model),
                     par.r[i].p, n, x, cii, verbose-20);
    if(ret) break;
    /* compute integrals using the function to be tested */
    ret=liIntegratePET(x1, x2, y, n, yi, yii, verbose-10);
    if(ret) break;
    if(verbose>100) {
      for(j=0; j<n; j++) printf("%6.2f  %10.3e  %10.3e %10.3e  %10.3e %10.3e\n", 
        x[j], y[j], ci[j], yi[j], cii[j], yii[j]);
    }
    for(j=0; j<n; j++) {
      if(doubleMatch(ci[j], yi[j], 0.01)) continue;
      if(doubleMatchRel(ci[j], yi[j], 1.0E-04)) continue;
      if(verbose>1) {
        printf("%6.2f  %10.3e  %10.3e %10.3e\n", 
               x[j], y[j], ci[j], yi[j]);
        printf("percentual i difference (%d) := %g\n", 
               1+j, 100.*(yi[j]-ci[j])/ci[j]);
      }
      ret=999; break;
    }
    if(ret) break;
    for(j=0; j<n; j++) {
      if(doubleMatch(cii[j], yii[j], 0.001)) continue;
      if(doubleMatchRel(cii[j], yii[j], 1.0E-04)) continue;
      if(verbose>1) {
        printf("%6.2f  %10.3e  %10.3e %10.3e\n", 
               x[j], y[j], cii[j], yii[j]);
        printf("percentual ii difference (%d) := %g\n", 
               1+j, 100.*(yii[j]-cii[j])/cii[j]);
      }
      ret=999; break;
    }
    if(ret) break;
  }
  if(ret) {parFree(&par); return(20);}
   
  parFree(&par);
  }  


  {
  if(verbose>1) printf("\none frame, starting from zero:\n");
  int ret, i, n=1;
  double x1[]={0}, x2[]={2};
  double y[]={1};
  double yi[n], yii[n];

  ret=liIntegratePET(x1, x2, y, n, yi, yii, verbose-10);
  if(ret!=0) return 101;
  if(verbose>3) for(i=0; i<n; i++) printf("  i=%d: x1=%g x2=%g y=%g -> %g %g\n", 
                  i, x1[i], x2[i], y[i], yi[i], yii[i]);
  for(i=0; i<n; i++) {
    if(!doubleMatch(1.0, yi[i], 1.0E-10)) return 102;
    if(!doubleMatch(0.5, yii[i], 1.0E-10)) return 103;
  }
  }  

  {
  if(verbose>1) printf("\nOne frame with initial gap:\n");
  int ret, i, n=1;
  double x1[]={6}, x2[]={8};
  double y[]={1};
  double yi[n], yii[n];

  ret=liIntegratePET(x1, x2, y, n, yi, yii, verbose-10);
  if(ret!=0) return 101;
  if(verbose>3) for(i=0; i<n; i++) printf("  i=%d: x1=%g x2=%g y=%g -> %g %g\n", 
                  i, x1[i], x2[i], y[i], yi[i], yii[i]);
  for(i=0; i<n; i++) {
    if(!doubleMatch(1.0, yi[i], 1.0E-10)) return 112;
    if(!doubleMatch(0.5, yii[i], 1.0E-10)) return 113;
  }
  }  


  {
  if(verbose>1) printf("\n Two frames, gap between frames:\n");
  int ret, i, n=2;
  double x1[]={0, 4}, x2[]={2, 12};
  double y[]={1, 3};
  double yi[n], yii[n];
  double cyi[n], cyii[n];

  if(verbose>2) printf("... should give the same result as with the gap filled\n");
  int tn=3;
  double tx1[]={0, 2, 4}, tx2[]={2, 4, 12};
  double ty[]={1, 2.6, 3};
  double tyi[tn], tyii[tn];
  ret=liIntegratePET(tx1, tx2, ty, tn, tyi, tyii, verbose-10);
  if(ret!=0) return 201;
  cyi[0]=tyi[0]; cyii[0]=tyii[0];
  cyi[1]=tyi[2]; cyii[1]=tyii[2];

  ret=liIntegratePET(x1, x2, y, n, yi, yii, verbose-10);
  if(ret!=0) return 202;
  if(verbose>3) for(i=0; i<n; i++) 
    printf("  i=%d: x1=%g x2=%g y=%g -> %g %g (%g %g)\n", 
                  i, x1[i], x2[i], y[i], yi[i], yii[i], cyi[i], cyii[i]);
  for(i=0; i<n; i++) {
    if(!doubleMatch(cyi[i],  yi[i],  1.0E-10)) return 203;
    if(!doubleMatch(cyii[i], yii[i], 1.0E-10)) return 204;
  }
  }


  fflush(stderr);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
int test_liIntegrateHalfFrame(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }

  int ret;

  {
  if(verbose>1) printf("testing with empty data\n");
  ret=liFrameMidValue(NULL, NULL, NULL, 0, NULL, verbose-1);
  if(verbose>1) printf("did not segfault, that is good.\n");
  if(ret==0) return 1;
  ret=liIntegrateHalfFrame(NULL, NULL, NULL, 100, NULL, verbose-1);
  if(verbose>1) printf("did not segfault, that is good.\n");
  if(ret==0) return 2;

  double a[10], b[10], c[10], d[10];
  ret=liIntegrateHalfFrame(a, b, c, 10, NULL, verbose-1);
  if(verbose>1) printf("did not segfault, that is good.\n");
  if(ret==0) return 3;
  ret=liIntegrateHalfFrame(a, b, c, 0, d, verbose-1);
  if(verbose>1) printf("did not segfault, that is good.\n");
  if(ret==0) return 4;
  ret=liIntegrateHalfFrame(a, b, NULL, 10, d, verbose-1);
  if(verbose>1) printf("did not segfault, that is good.\n");
  if(ret==0) return 5;
  ret=liIntegrateHalfFrame(a, NULL, c, 0, d, verbose-1);
  if(verbose>1) printf("did not segfault, that is good.\n");
  if(ret==0) return 6;
  ret=liIntegrateHalfFrame(NULL, b, c, 0, d, verbose-1);
  if(verbose>1) printf("did not segfault, that is good.\n");
  if(ret==0) return 7;
  }

  {
  if(verbose>1) printf("testing with one sample\n");
  double x1[1], x2[1], y[1], hfi[1];
  x1[0]=0.0; x2[0]=2.0; y[0]=100.0;
  ret=liIntegrateHalfFrame(x1, x2, y, 1, hfi, verbose-1);
  if(ret!=0 || !doubleMatch(100.0, hfi[0], 1.0E-20)) return 11;

  x1[0]=0.0; x2[0]=0.0; y[0]=100.0;
  ret=liIntegrateHalfFrame(x1, x2, y, 1, hfi, verbose-1);
  if(ret!=0 || !doubleMatch(0.0, hfi[0], 1.0E-20)) return 12;
  }


  {
  if(verbose>1) printf("testing with two samples\n");
  size_t n=2;
  double x1[n], x2[n], y[n], hfi[n], ci[n];
  x1[0]=0.0; x2[0]=2.0;  y[0]=100.0;  ci[0]=550./6.;
  x1[1]=4.0; x2[1]=10.0; y[1]=200.0;  ci[1]=525.;
  ret=liIntegrateHalfFrame(x1, x2, y, n, hfi, verbose-1);
  if(ret!=0) return 21;
  for(size_t i=0; i<n; i++) if(!doubleMatch(ci[i], hfi[i], 1.0E-20)) return 22;

  x1[0]=0.0; x2[0]=6.0;  y[0]=100.0;  ci[0]=200.;
  x1[1]=5.0; x2[1]=10.0; y[1]=200.0;  ci[1]=7750./18.;
  ret=liIntegrateHalfFrame(x1, x2, y, n, hfi, verbose-1);
  if(ret!=0) return 23;
  for(size_t i=0; i<n; i++) if(!doubleMatch(ci[i], hfi[i], 1.0E-20)) return 24;
  }


  {
  if(verbose>1) printf("testing with 6 samples against known results\n");
  size_t i, n=6;
  double x1[n], x2[n], y[n], hfi[n], icorr[n];
  i=0; x1[i]=0.0;   x2[i]=10.0;
  i++; x1[i]=10.0;  x2[i]=30.0;
  i++; x1[i]=30.0;  x2[i]=60.0;
  i++; x1[i]=60.0;  x2[i]=100.0;
  i++; x1[i]=100.0; x2[i]=150.0;
  i++; x1[i]=150.0; x2[i]=250.0;
  i=0; y[i]=5.0;          icorr[i]=12.5;
  i++; y[i]=25.5;         icorr[i]=150.0;
  i++; y[i]=85.85714286;  icorr[i]=1140.0;
  i++; y[i]=76.19047619;  icorr[i]=1714.285714;
  i++; y[i]=27.5;         icorr[i]=916.6666667;
  i++; y[i]=10.0;         icorr[i]=666.6666667;
  ret=liIntegrateHalfFrame(x1, x2, y, n, hfi, verbose-10);
  if(ret!=0) return 31;
  for(size_t i=0; i<n; i++) if(!doubleMatch(icorr[i], hfi[i], 1.0E-06)) {
    printf("y[%zu]: %E vs %E\n", i+1, icorr[i], hfi[i]);
    return 32;
  }

  if(verbose>1) printf("negative frame duration should lead to error\n");
  i=n-1; x1[i]=350.0; x2[i]=250.0;
  ret=liIntegrateHalfFrame(x1, x2, y, n, hfi, verbose-2);
  if(ret==0) return 41;

  if(verbose>1) printf("non-ascending x values should lead to error\n");
  i=n-1; x1[i]=50.0; x2[i]=60.0;
  ret=liIntegrateHalfFrame(x1, x2, y, n, hfi, verbose-2);
  if(ret==0) return 51;

  if(verbose>1) printf("testing with one NaN y value\n");
  i=0; x1[i]=0.0;   x2[i]=10.0;
  i++; x1[i]=10.0;  x2[i]=30.0;
  i++; x1[i]=30.0;  x2[i]=60.0;
  i++; x1[i]=60.0;  x2[i]=100.0;
  i++; x1[i]=100.0; x2[i]=150.0;
  i++; x1[i]=150.0; x2[i]=250.0;
  i=0; y[i]=5.0;          icorr[i]=12.5;
  i++; y[i]=20.0;         icorr[i]=150.0;
  i++; y[i]=100.0;        icorr[i]=1140.0;
  i++; y[i]=nan("");      icorr[i]=nan("");
  i++; y[i]=20.0;         icorr[i]=nan("");
  i++; y[i]=10.0;         icorr[i]=666.6666667;
  ret=liIntegrateHalfFrame(x1, x2, y, n, hfi, verbose-10);
  if(ret!=0) return 52;
  /* half-integrals should be calculated using original y values as mid point 
     values, but with NaN at missing point and the sample after that */
  for(size_t i=0; i<n; i++) if(!doubleMatch(icorr[i], hfi[i], 1.0E-06)) {
    printf("y[%zu]: %E vs %E\n", i+1, icorr[i], hfi[i]);
    return 53;
  }

  }


  {
  if(verbose>1) printf("testing with 4 samples against known results\n");
  size_t i, n=4;
  double x1[n], x2[n], y[n], hfi[n], icorr[n];
  i=0; x1[i]=0.0;   x2[i]=10.0;
  i++; x1[i]=10.0;  x2[i]=30.0;
  i++; x1[i]=30.0;  x2[i]=50.0;
  i++; x1[i]=50.0;  x2[i]=100.0;
  i=0; y[i]=5.0;          icorr[i]=12.5;
  i++; y[i]=16.25;        icorr[i]=150.0;
  i++; y[i]=10.89285714;  icorr[i]=125.0;
  i++; y[i]=5.0;          icorr[i]=169.6428571;
  ret=liIntegrateHalfFrame(x1, x2, y, n, hfi, verbose-10);
  if(ret!=0) return 61;
  for(size_t i=0; i<n; i++) if(!doubleMatch(icorr[i], hfi[i], 1.0E-06)) {
    printf("y[%zu]: %E vs %E\n", i+1, icorr[i], hfi[i]);
    return 62;
  }

  if(verbose>1) printf("testing with negative frame times\n");
  for(size_t i=0; i<n; i++) {x1[i]-=80.0; x2[i]-=80.0;}
  ret=liIntegrateHalfFrame(x1, x2, y, n, hfi, verbose-10);
  if(ret!=0) return 71;
  for(size_t i=0; i<n; i++) if(!doubleMatch(icorr[i], hfi[i], 1.0E-06)) {
    printf("y[%zu]: %E vs %E\n", i+1, icorr[i], hfi[i]);
    return 72;
  }

  if(verbose>1) printf("testing with all-zero frame times\n");
  for(size_t i=0; i<n; i++) {x1[i]=0.0; x2[i]=0.0;}
  ret=liIntegrateHalfFrame(x1, x2, y, n, hfi, verbose-10);
  if(ret!=0) return 81;
  for(size_t i=0; i<n; i++) if(!doubleMatch(0.0, hfi[i], 1.0E-06)) {
    printf("y[%zu]: %E vs %E\n", i+1, 0.0, hfi[i]);
    return 82;
  }

  if(verbose>1) printf("testing with missing (NaN) frame times\n");
  for(size_t i=0; i<n; i++) {x1[i]=x2[i]=nan("");}
  ret=liIntegrateHalfFrame(x1, x2, y, n, hfi, verbose-10);
  if(ret!=0) return 91;
  for(size_t i=0; i<n; i++) if(!isnan(hfi[i])) {
    printf("y[%zu]: NaN vs %E\n", i+1, hfi[i]);
    return 92;
  }
  }


  {
  if(verbose>1) printf("testing with pseudoframe TAC (all frame lengths 0)\n");
  size_t i, n=6;
  double x1[n], x2[n], y[n], hfi[n], icorr[n];
  i=0; x1[i]=x2[i]=10.0;  y[i]=2.0;   icorr[i]=0.0;
  i++; x1[i]=x2[i]=30.0;  y[i]=200.0; icorr[i]=0.0;
  i++; x1[i]=x2[i]=50.0;  y[i]=180.0; icorr[i]=0.0;
  i++; x1[i]=x2[i]=100.0; y[i]=130.0; icorr[i]=0.0;
  i++; x1[i]=x2[i]=200.0; y[i]=90.0;  icorr[i]=0.0;
  i++; x1[i]=x2[i]=500.0; y[i]=60.0;  icorr[i]=0.0;
  ret=liIntegrateHalfFrame(x1, x2, y, n, hfi, verbose-10);
  if(ret!=0) return 101;
  /* half-integrals should be 0 */
  for(size_t i=0; i<n; i++) if(!doubleMatch(icorr[i], hfi[i], 1.0E-06)) {
    printf("y[%zu]: %E vs %E\n", i+1, icorr[i], hfi[i]);
    return 102;
  }

  if(verbose>1) printf("testing with one NaN y value\n");
  i=3; y[i]=nan(""); icorr[i]=icorr[i+1]=nan("");
  ret=liIntegrateHalfFrame(x1, x2, y, n, hfi, verbose-10);
  if(ret!=0) return 103;
  /* half-integrals are still zeroes, except for the sample and sample following
     it, where result is also NaN */
  for(size_t i=0; i<n; i++) if(!doubleMatch(icorr[i], hfi[i], 1.0E-06)) {
    printf("y[%zu]: %E vs %E\n", i+1, icorr[i], hfi[i]);
    return 104;
  }
  }

  
  fflush(stderr);
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
