/** @file rndpoint.c
 *  @brief Random point for optimization routines.
 */
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <time.h>
#include <string.h>
/*****************************************************************************/
#include "tpcextensions.h"
#include "tpcrand.h"
/*****************************************************************************/
#include "tpcnlopt.h"
/*****************************************************************************/

/*****************************************************************************/
/** Create random parameters between specified limits. 
    @pre Uses rand(), therefore set seed for a new series of pseudo-random numbers;
    to produce truly random numbers (not just pseudo-random), do srand(time(0)) before calling
    this function. If no seed is set, then value 1 is used as default seed by rand().
    @sa drand, drandRange, drandGaussian, drandExponential, mertwiRandomBetween
    @author Vesa Oikonen
    @return Returns non-zero in case of an error.
 */
int nloptRandomPoint(
  /** Pointer to parameter list to be filled. */
  double *p,
  /** Pointer to parameter lower limits. */
  double *low,
  /** Pointer to parameter upper limits. */
  double *up,
  /** List length. */
  unsigned int n,
  /** Pointer to initiated and seeded Mersenne Twister MT19937 data structure; 
      enter NULL to use drand() instead.
      @sa mertwiInit, mertwiInitWithSeed64  */
  MERTWI *mt
) {
  if(p==NULL || low==NULL || up==NULL) return(1);
  if(n==0) return(0);

  if(mt==NULL) {
    for(unsigned int i=0; i<n; i++) {
      double dif=up[i]-low[i];
      if(dif<=0.0) p[i]=low[i]; else p[i]=low[i]+drand()*dif;
    }
  } else {
    for(unsigned int i=0; i<n; i++) {
      double dif=up[i]-low[i];
      if(dif<=0.0) p[i]=low[i]; else p[i]=low[i]+mertwiRandomDouble1(mt)*dif;
    }
  }

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Create random parameters with Gaussian distribution. 
    @pre Uses rand(), therefore set seed for a new series of pseudo-random numbers;
    to produce truly random numbers (not just pseudo-random), do srand(time(0)) before calling
    this function. If no seed is set, then value 1 is used as default seed by rand().
    @sa drand, nloptRandomPoint, drandRange, drandGaussian, drandExponential, mertwiRandomGaussian
    @author Vesa Oikonen
    @return Returns non-zero in case of an error.
 */
int nloptGaussianPoint(
  /** Pointer to parameter list to be filled, allocated for length n. */
  double *p,
  /** Pointer to list of mean values for each parameter. */
  double *mean,
  /** Pointer to list of standard deviations for each parameter. */
  double *sd,
  /** Pointer to parameter lower limits; generated pseudo-random numbers falling below the lower
      limit will be set to the limit; NULL if not needed. */
  double *low,
  /** Pointer to parameter upper limits; generated pseudo-random numbers falling above the upper
      limit will be set to the limit; NULL if not needed. */
  double *up,
  /** List length. */
  unsigned int n,
  /** Pointer to initiated and seeded Mersenne Twister MT19937 data structure; 
      enter NULL to use drandGaussian() instead.
      @sa mertwiInit, mertwiInitWithSeed64  */
  MERTWI *mt
) {
  if(p==NULL || mean==NULL || sd==NULL) return(1);
  if(n==0) return(0);

  for(unsigned int i=0; i<n; i++) {
    if(mt==NULL) p[i]=mean[i]+sd[i]*drandGaussian();
    else p[i]=mean[i]+sd[i]*mertwiRandomGaussian(mt);
    if(low!=NULL && p[i]<low[i]) p[i]=low[i];
    if(up!=NULL  && p[i]>up[i])  p[i]=up[i];
  }

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
