/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include "tpcextensions.h"
#include "test_tpcnlopt.h"
/*****************************************************************************/

/*****************************************************************************/
int test_nloptInit(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }

  NLOPT nlo;
  nloptInit((NLOPT*)NULL);
  if(verbose>1) printf("did not segfault, that is good.\n");
  nloptInit(&nlo);
  if(verbose>1) printf("did not segfault, that is good.\n");

  statusSet(status, __func__, __FILE__, __LINE__, 0);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
int test_nloptFree(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }

  nloptFree((NLOPT*)NULL);
  if(verbose>1) printf("did not segfault, that is good.\n");

  NLOPT nlo; nloptInit(&nlo);
  nloptFree(&nlo);
  if(verbose>1) printf("did not segfault, that is good.\n");

  statusSet(status, __func__, __FILE__, __LINE__, 0);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
int test_nloptAllocate(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }

  int ret;

  ret=nloptAllocate((NLOPT*)NULL, 1000);
  if(ret==TPCERROR_OK) return 1;
  if(verbose>1) printf("did not segfault, that is good.\n");

  NLOPT nlo; nloptInit(&nlo);
  ret=nloptAllocate(&nlo, 10);
  if(ret!=TPCERROR_OK) {nloptFree(&nlo); return 2;}
  nloptFree(&nlo);
  if(verbose>1) printf("did not segfault, that is good.\n");

  statusSet(status, __func__, __FILE__, __LINE__, 0);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
int test_nloptDuplicate(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }

  int ret;

  ret=nloptDuplicate((NLOPT*)NULL, NULL);
  if(ret==TPCERROR_OK) return 1;
  if(verbose>1) printf("did not segfault, that is good.\n");

  NLOPT nlo1; nloptInit(&nlo1);
  ret=nloptDuplicate(&nlo1, NULL);
  if(ret==TPCERROR_OK) return 2;
  if(verbose>1) printf("did not segfault, that is good.\n");

  NLOPT nlo2; nloptInit(&nlo2);
  ret=nloptDuplicate(&nlo1, &nlo2);
  if(ret!=TPCERROR_OK) return 3;
  if(verbose>2) printf("did not segfault, that is good.\n");

  ret=nloptAllocate(&nlo1, 10);
  if(ret!=TPCERROR_OK) {nloptFree(&nlo1); return 10;}
  ret=nloptDuplicate(&nlo1, &nlo2);
  if(ret!=TPCERROR_OK) {nloptFree(&nlo1); nloptFree(&nlo2); return 11;}
  if(nlo2.totalNr!=10) {nloptFree(&nlo1); nloptFree(&nlo2); return 12;}

  nloptFree(&nlo1); nloptFree(&nlo2);
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
int test_nloptAddP(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }

  if(verbose>1) printf("\nTesting with NULL input\n");
  if(nloptAddP(NULL, NULL, nan(""))==TPCERROR_OK) return(1);
  if(verbose>1) printf("did not segfault, that is good.\n");

  NLOPT nlo1; nloptInit(&nlo1);
  if(nloptAddP(&nlo1, NULL, nan(""))==TPCERROR_OK) return(2);
  if(verbose>1) printf("did not segfault, that is good.\n");

  int parNr=5; double p[parNr];
  if(nloptAddP(&nlo1, p, nan(""))==TPCERROR_OK) return(3);
  if(verbose>1) printf("did not segfault, that is good.\n");

  if(nloptAllocate(&nlo1, parNr)!=TPCERROR_OK) {nloptFree(&nlo1); return(100);}
  for(unsigned int i=0; i<nlo1.totalNr; i++) {
    nlo1.xlower[i]=0.0; nlo1.xupper[i]=1.0+i; nlo1.xfull[i]=0.5*(nlo1.xlower[i]+nlo1.xupper[i]);
  }
  nlo1.usePList=1;
  if(verbose>2) nloptWrite(&nlo1, stdout);
  
  for(unsigned int i=0; i<nlo1.totalNr; i++) p[i]=1.1*(double)i;
  nlo1.funCalls++;
  if(nloptAddP(&nlo1, p, 9.9)!=TPCERROR_OK) {nloptFree(&nlo1); return(111);}

  for(unsigned int i=0; i<nlo1.totalNr; i++) p[i]=1.2*(double)i;
  nlo1.funCalls++;
  if(nloptAddP(&nlo1, p, 6.6)!=TPCERROR_OK) {nloptFree(&nlo1); return(112);}

  for(unsigned int i=0; i<nlo1.totalNr; i++) p[i]=1.3*(double)i;
  nlo1.funCalls++;
  if(nloptAddP(&nlo1, p, 3.3)!=TPCERROR_OK) {nloptFree(&nlo1); return(113);}

  nloptFree(&nlo1);
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
int test_nloptSortP(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }

  if(verbose>1) printf("\nTesting with NULL input\n");
  if(nloptSortP(NULL)==TPCERROR_OK) return(1);
  if(verbose>1) printf("did not segfault, that is good.\n");

  NLOPT nlo1; nloptInit(&nlo1);
  if(nloptSortP(&nlo1)==TPCERROR_OK) return(2);
  if(verbose>1) printf("did not segfault, that is good.\n");

  int parNr=5; double p[parNr];
  if(nloptSortP(&nlo1)==TPCERROR_OK) return(3);
  if(verbose>1) printf("did not segfault, that is good.\n");

  if(nloptAllocate(&nlo1, parNr)!=TPCERROR_OK) {nloptFree(&nlo1); return(100);}
  for(unsigned int i=0; i<nlo1.totalNr; i++) {
    nlo1.xlower[i]=0.0; nlo1.xupper[i]=1.0+i; nlo1.xfull[i]=0.5*(nlo1.xlower[i]+nlo1.xupper[i]);
  }
  nlo1.usePList=1;
  if(verbose>2) nloptWrite(&nlo1, stdout); 
  for(unsigned int i=0; i<nlo1.totalNr; i++) p[i]=1.1*(double)i;
  nlo1.funCalls++;
  if(nloptAddP(&nlo1, p, 9.9)!=TPCERROR_OK) {nloptFree(&nlo1); return(101);}
  for(unsigned int i=0; i<nlo1.totalNr; i++) p[i]=1.2*(double)i;
  nlo1.funCalls++;
  if(nloptAddP(&nlo1, p, 6.6)!=TPCERROR_OK) {nloptFree(&nlo1); return(102);}
  for(unsigned int i=0; i<nlo1.totalNr; i++) p[i]=1.3*(double)i;
  nlo1.funCalls++;
  if(nloptAddP(&nlo1, p, 3.3)!=TPCERROR_OK) {nloptFree(&nlo1); return(103);}

  if(verbose>1) printf("\nSorting\n");
  if(nloptSortP(&nlo1)!=TPCERROR_OK) {nloptFree(&nlo1); return(110);}
  if(!doubleMatch(0.0, nlo1.plist[0], 1.0E-20)) {nloptFree(&nlo1); return(111);}
  if(!doubleMatch(1.3, nlo1.plist[1], 1.0E-20)) {nloptFree(&nlo1); return(112);}
  if(!doubleMatch(3.3, nlo1.plist[parNr], 1.0E-20)) {nloptFree(&nlo1); return(113);}

  nloptFree(&nlo1);

  statusSet(status, __func__, __FILE__, __LINE__, 0);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
int test_nloptMeanP(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }

  if(verbose>1) printf("\nTesting with NULL input\n");
  if(nloptMeanP(NULL, 0, NULL, NULL)==TPCERROR_OK) return(1);
  if(verbose>1) printf("did not segfault, that is good.\n");

  NLOPT nlo1; nloptInit(&nlo1);
  if(nloptMeanP(&nlo1, 0, NULL, NULL)==TPCERROR_OK) return(2);
  if(verbose>1) printf("did not segfault, that is good.\n");

  int parNr=5; double p[parNr];
  if(nloptMeanP(&nlo1, 200, p, NULL)==TPCERROR_OK) return(3);
  if(verbose>1) printf("did not segfault, that is good.\n");

  if(nloptAllocate(&nlo1, parNr)!=TPCERROR_OK) {nloptFree(&nlo1); return(100);}
  for(unsigned int i=0; i<nlo1.totalNr; i++) {
    nlo1.xlower[i]=0.0; nlo1.xupper[i]=1.0+i; nlo1.xfull[i]=0.5*(nlo1.xlower[i]+nlo1.xupper[i]);
  }
  nlo1.usePList=1;
  for(unsigned int i=0; i<nlo1.totalNr; i++) p[i]=1.1*(double)i;
  nlo1.funCalls++;
  if(nloptAddP(&nlo1, p, 9.9)!=TPCERROR_OK) {nloptFree(&nlo1); return(101);}
  for(unsigned int i=0; i<nlo1.totalNr; i++) p[i]=1.2*(double)i;
  nlo1.funCalls++;
  if(nloptAddP(&nlo1, p, 6.6)!=TPCERROR_OK) {nloptFree(&nlo1); return(102);}
  for(unsigned int i=0; i<nlo1.totalNr; i++) p[i]=1.3*(double)i;
  nlo1.funCalls++;
  if(nloptAddP(&nlo1, p, 3.3)!=TPCERROR_OK) {nloptFree(&nlo1); return(103);}
  if(verbose>2) nloptWrite(&nlo1, stdout); 

  if(verbose>1) printf("\nMeans of all\n");
  if(nloptMeanP(&nlo1, 0, p, NULL)!=TPCERROR_OK) return(111);
  if(!doubleMatch(0.0, p[0], 1.0E-20)) {nloptFree(&nlo1); return(112);}
  if(!doubleMatch(1.2, p[1], 1.0E-20)) {nloptFree(&nlo1); return(113);}
  if(!doubleMatch(2.4, p[2], 1.0E-20)) {nloptFree(&nlo1); return(114);}
  if(!doubleMatch(3.6, p[3], 1.0E-20)) {nloptFree(&nlo1); return(115);}
  if(!doubleMatch(4.8, p[4], 1.0E-20)) {nloptFree(&nlo1); return(116);}

  if(verbose>1) printf("\nMeans of more than available\n");
  if(nloptMeanP(&nlo1, 200, p, NULL)!=TPCERROR_OK) return(117);
  if(!doubleMatch(3.6, p[3], 1.0E-20)) {nloptFree(&nlo1); return(118);}

  if(verbose>1) printf("\n'Means' and SD of one\n");
  double sd[parNr];
  if(nloptMeanP(&nlo1, 1, p, sd)!=TPCERROR_OK) return(120);
  if(!doubleMatch(0.0, p[0], 1.0E-20)) {nloptFree(&nlo1); return(121);}
  if(!doubleMatch(1.1, p[1], 1.0E-20)) {nloptFree(&nlo1); return(122);}
  if(!doubleMatch(2.2, p[2], 1.0E-20)) {nloptFree(&nlo1); return(123);}
  if(!doubleMatch(3.3, p[3], 1.0E-10)) {nloptFree(&nlo1); return(124);}
  if(!doubleMatch(4.4, p[4], 1.0E-10)) {nloptFree(&nlo1); return(125);}
  if(!doubleMatch(0.0, sd[0], 1.0E-20)) {nloptFree(&nlo1); return(126);}
  if(!doubleMatch(0.0, sd[1], 1.0E-20)) {nloptFree(&nlo1); return(127);}
  if(!doubleMatch(0.0, sd[2], 1.0E-20)) {nloptFree(&nlo1); return(128);}
  if(!doubleMatch(0.0, sd[4], 1.0E-20)) {nloptFree(&nlo1); return(129);}


  if(verbose>1) printf("\nSorting\n");
  if(nloptSortP(&nlo1)!=TPCERROR_OK) {nloptFree(&nlo1); return(130);}

  if(verbose>1) printf("\nMeans and SDs of two first\n");
  if(nloptMeanP(&nlo1, 2, p, sd)!=TPCERROR_OK) return(131);
  if(!doubleMatch(0.0, p[0], 1.0E-20)) {nloptFree(&nlo1); return(132);}
  if(!doubleMatch(1.25, p[1], 1.0E-20)) {nloptFree(&nlo1); return(133);}
  if(!doubleMatch(2.5, p[2], 1.0E-20)) {nloptFree(&nlo1); return(134);}
  if(!doubleMatch(3.75, p[3], 1.0E-20)) {nloptFree(&nlo1); return(135);}
  if(!doubleMatch(5.0, p[4], 1.0E-20)) {nloptFree(&nlo1); return(136);}
  if(!doubleMatch(0.0, sd[0], 1.0E-20)) {nloptFree(&nlo1); return(137);}
  if(!doubleMatch(0.282843, sd[4], 1.0E-05)) {nloptFree(&nlo1); return(138);}

  nloptFree(&nlo1);

  statusSet(status, __func__, __FILE__, __LINE__, 0);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
int test_nloptLimitFixedNr(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }

  NLOPT nlo; nloptInit(&nlo);
  unsigned int i, n;

  if(verbose>1) printf("\nTesting with NULL input\n");
  n=nloptLimitFixedNr(NULL);
  if(n>0) return(1);
  n=nloptLimitFixedNr(&nlo);
  if(n>0) return(2);

  if(verbose>1) printf("\nTesting with reasonable input\n");
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(nloptAllocate(&nlo, 9)!=TPCERROR_OK) return(10);
  for(i=0; i<nlo.totalNr; i++) {
    nlo.xlower[i]=0.0; nlo.xupper[i]=1.0+i;
  }

  n=nloptLimitFixedNr(&nlo);
  if(n>0) {nloptFree(&nlo); return(11);}

  nlo.xupper[2]=nlo.xlower[2]; if(verbose>5) nloptWrite(&nlo, stdout);
  n=nloptLimitFixedNr(&nlo);
  if(n!=(unsigned int)1) {nloptFree(&nlo); return(12);}

  if(verbose>1) printf("\nTesting with NaNs\n");
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  nlo.xupper[3]=nlo.xlower[3]=nan("");
  n=nloptLimitFixedNr(&nlo);
  if(n!=(unsigned int)1) {nloptFree(&nlo); return(22);}

  nloptFree(&nlo);
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
int test_nloptFixedNr(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }

  NLOPT nlo; nloptInit(&nlo);
  unsigned int i, n;

  if(verbose>1) printf("\nTesting with NULL input\n");
  n=nloptFixedNr(NULL);
  if(n>0) return(1);
  n=nloptFixedNr(&nlo);
  if(n>0) return(2);

  if(verbose>1) printf("\nTesting with reasonable input\n");
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(nloptAllocate(&nlo, 9)!=TPCERROR_OK) return(10);
  for(i=0; i<nlo.totalNr; i++) {
    nlo.xlower[i]=0.0; nlo.xupper[i]=1.0+i;
    nlo.xdelta[i]=0.0987*(nlo.xupper[i]-nlo.xlower[i]);
  }

  n=nloptFixedNr(&nlo);
  if(n>0) {nloptFree(&nlo); return(11);}

  nlo.xdelta[1]=0.0; if(verbose>5) nloptWrite(&nlo, stdout);
  n=nloptFixedNr(&nlo);
  if(n!=(unsigned int)1) {nloptFree(&nlo); return(12);}
  nlo.xupper[2]=nlo.xlower[2]; if(verbose>5) nloptWrite(&nlo, stdout);
  n=nloptFixedNr(&nlo);
  if(n!=(unsigned int)2) {nloptFree(&nlo); return(13);}
  nlo.xdelta[2]=0.0; if(verbose>5) nloptWrite(&nlo, stdout);
  n=nloptFixedNr(&nlo);
  if(n!=(unsigned int)2) {nloptFree(&nlo); return(14);}

  if(verbose>1) printf("\nTesting with NaNs\n");
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  nlo.xdelta[1]=nan(""); if(verbose>5) nloptWrite(&nlo, stdout);
  n=nloptFixedNr(&nlo);
  if(n!=(unsigned int)1) {nloptFree(&nlo); return(21);}
  nlo.xupper[2]=nlo.xlower[2]=nan("");
  n=nloptFixedNr(&nlo);
  if(n!=(unsigned int)1) {nloptFree(&nlo); return(22);}
  nlo.xdelta[2]=nan(""); if(verbose>5) nloptWrite(&nlo, stdout);
  n=nloptFixedNr(&nlo);
  if(n!=(unsigned int)0) {nloptFree(&nlo); return(23);}

  nloptFree(&nlo);
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
int test_nloptRemoveEmpties(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }

  NLOPT nlo; nloptInit(&nlo);
  unsigned int i;

  if(verbose>1) printf("\nTesting with NULL input\n");
  nloptRemoveEmpties(NULL);
  if(verbose>1) printf("  -> good, did not crash\n");
  if(verbose>1) printf("\nTesting with empty input\n");
  nloptRemoveEmpties(&nlo);
  if(verbose>1) printf("  -> good, did not crash\n");

  if(verbose>1) printf("\nTesting with reasonable input\n");
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(nloptAllocate(&nlo, 9)!=TPCERROR_OK) return(10);
  for(i=0; i<nlo.totalNr; i++) {
    nlo.xlower[i]=0.0; nlo.xupper[i]=1.0+i;
    nlo.xfull[i]=0.45*(nlo.xupper[i]+nlo.xlower[i]);
    nlo.xdelta[i]=0.0987*(nlo.xupper[i]-nlo.xlower[i]);
  }

  if(verbose>5) nloptWrite(&nlo, stdout);
  nloptRemoveEmpties(&nlo);
  if(nlo.totalNr!=9) {nloptFree(&nlo); return(11);}

  statusSet(status, __func__, __FILE__, __LINE__, 0);
  nlo.xfull[0]=nan("");
  if(verbose>5) nloptWrite(&nlo, stdout);
  nloptRemoveEmpties(&nlo);
  if(nlo.totalNr!=8) {nloptFree(&nlo); return(12);}

  statusSet(status, __func__, __FILE__, __LINE__, 0);
  nlo.xdelta[0]=nan("");
  if(verbose>5) nloptWrite(&nlo, stdout);
  nloptRemoveEmpties(&nlo);
  if(nlo.totalNr!=7) {nloptFree(&nlo); return(13);}

  nloptFree(&nlo);
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** De Jong's second function or Rosenbrock's valley or Banana function.
    One minimum f(xi)=0. With n=2 minimum is at (1,1) and with n=3 at(1,1,1).
    With 4<=n<=7 there is one global minimum at (1,1,...,1) and local minimum somewhere 
    near (-1,1,...1). For larger n the method breaks down.
    If used to test local optimization, note that initial guess must be quite close to the minimum.
 */
double nloptDejong2(
  /** Number of parameters. */
  int n, 
  /** Pointer to the parameter array of length n. */
  double *x, 
  /** Pointer to additional data to the function; not used here. */
  void *func_data
) {
  if(func_data==NULL) {/* just to prevent compiler warning */}
  double d1, d2, f=0.0;
  for(int i=0; i<n-1; i++) {
    d1=fma(-x[i], x[i], x[i+1]); // d1=x[i+1]-x[i]*x[i];
    d2=1.0-x[i];
    f+=100.0*d1*d1 + d2*d2; 
  }
#if(0)
  printf("  p[]: %g", x[0]);
  for(int i=1; i<n; i++) printf(", %g", x[i]);
  printf(" => %g\n", f);
#endif
  return f;
}
/*****************************************************************************/

/*****************************************************************************/
/** Generalized Rastrigin function,
    f(x) = A*n + SUM(i=1..n; (xi^2 - A*cos(2*PI*xi))), where A=10 and xi=[-5.12,5.12].
    It has global minimum value f(x)=0.0 at xi=0.0, and large nr of local minima around it.
    If used to test local optimization, note that initial guess must be quite close to the minimum.
 */
double nloptRastrigin(
  /** Number of parameters. */
  int n,
  /** Pointer to the parameter array of length n. */
  double *x, 
  /** Pointer to additional data to the function; not used here. */
  void *func_data
) {
  if(func_data==NULL) {} // just to prevent compiler warning
  if(n<1 || n>50) return(nan(""));
  for(int i=0; i<n; i++) if(x[i]<-5.12 || x[i]>5.12) return(nan(""));

  double f=0.0;
  for(int i=0; i<n; i++)
    f+=fma(x[i], x[i], -10.0*cos(2.0*M_PI*x[i]));
  f+=10.0*n;
  return f;
}
/*****************************************************************************/

/*****************************************************************************/
/** Generalized Rastrigin function,
    modified so that 1.0 is added to the returned value.
 */
double nloptRastrigin1(
  /** Number of parameters. */
  int n,
  /** Pointer to the parameter array of length n. */
  double *x, 
  /** Pointer to additional data to the function; not used here. */
  void *func_data
) {
  if(func_data==NULL) {} // just to prevent compiler warning
  if(n<1 || n>50) return(nan(""));
  for(int i=0; i<n; i++) if(x[i]<-5.12 || x[i]>5.12) return(nan(""));

  double f=0.0;
  for(int i=0; i<n; i++)
    f+=fma(x[i], x[i], -10.0*cos(2.0*M_PI*x[i]));
  f+=10.0*n + 1.0;
  return f;
}
/*****************************************************************************/

/*****************************************************************************/
/** Simple f(x)=(x-50.5)^2 function, with zero minimum at x=50.5. */
double nloptX2(
  /** Number of parameters. */
  int n,
  /** Pointer to the parameter array of length n. */
  double *x, 
  /** Pointer to additional data to the function; not used here. */
  void *func_data
) {
  if(func_data==NULL) {} // just to prevent compiler warning
  if(n!=1 || x==NULL) return(nan(""));
  double v=x[0]-50.5;
  return(v*v);
}
/*****************************************************************************/

/*****************************************************************************/
/** Bi-exponential function f(x)=p1*exp(-p2*x)+p3*exp(-p4*x). */
double nloptBiexp(
  /** Number of parameters; four is needed, any extra parameters are ignored. */
  int parNr,
  /** Pointer to the parameter array of length n. */
  double *p, 
  /** Pointer to additional data to the function; mus be stored in NLOPT_DATA structure. */
  void *func_data
) {
  NLOPT_DATA *d=(NLOPT_DATA*)func_data;

  if(d->verbose>0) {fprintf(d->fp, "%s(%d, p[], data)\n", __func__, parNr); fflush(d->fp);}
  if(parNr<4 || p==NULL || func_data==NULL) return(nan(""));
  if(d->n<1 || d->x==NULL || d->y==NULL || d->sy==NULL) return(nan(""));
  if(d->verbose>1) {
    fprintf(d->fp, "p[]: %g", p[0]); for(int i=1; i<4; i++) fprintf(d->fp, " %g", p[i]);
    fprintf(d->fp, "\n"); fflush(d->fp);
  }

  /* Simulate the values with given parameters */
  for(unsigned i=0; i<d->n; i++) {
    d->sy[i] = p[0]*exp(-p[1]*d->x[i]) + p[2]*exp(-p[3]*d->x[i]);
  }

  /* Calculate the weighted SS */
  double wss=0.0;
  for(unsigned i=0; i<d->n; i++) {
    double v=d->sy[i] - d->y[i];
    wss+=d->w[i]*v*v;
  }
  if(d->verbose>1) {fprintf(d->fp, "  --> %e\n", wss); fflush(d->fp);}

  return(wss);
}
/*****************************************************************************/

/*****************************************************************************/
