/** @file parexample.c
 *  @brief Create PAR data for use in testing TPCCLIB.
 */
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <time.h>
#include <string.h>
/*****************************************************************************/
#include "tpcpar.h"
/*****************************************************************************/

/*****************************************************************************/
/** Create PAR data containing function parameters for typical-looking 
    PET TTACs for use in algorithm and software tests.
    @details Creates three fengm2 functions with zero delay.
    @return enum tpcerror (TPCERROR_OK when successful).
    @author Vesa Oikonen
    @sa parInit, parExamplePerfectBolus
 */
int parExampleTTACs(
  /** Pointer to initiated PAR struct, which will be allocated and filled here;
      any previous content will be deleted. */
  PAR *d,
  /** TTAC type; currently unused */
  int type,
  /** Pointer to status data; enter NULL if not needed */
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  if(verbose>0) printf("%s(par, %d)\n", __func__, type);

  /* Check that required data exists */
  if(d==NULL) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_DATA);
    return TPCERROR_NO_DATA;
  }
  statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);

  int ret, pi, ri, parNr;
  
  /* Allocate memory */
  parNr=modelParNr(modelCodeIndex("fengm2"));
  ret=parAllocate(d, parNr, 3); if(ret!=TPCERROR_OK) return(ret);
  d->tacNr=3; d->parNr=5;
  d->format=PAR_FORMAT_CSV_INT;
  /* Set header fields */
  iftPut(&d->h, "studynr", "ttac1313", 1, NULL);
  iftPut(&d->h, "analysis_time", "2016-07-01 15:05:43", 1, NULL);
  iftPut(&d->h, "program", "libtpcpar (c) 2016", 1, NULL);
  iftPut(&d->h, "datafile","ttac1313.tac",1,NULL);
  iftPut(&d->h, "weighting", "no", 1, NULL);
  iftPut(&d->h, "unit", unitName(UNIT_KBQ_PER_ML), 1, NULL);
  iftPut(&d->h, "timeunit", unitName(UNIT_MIN), 1, NULL);
  /* Set parameter names */
  pi=0; strcpy(d->n[pi].name, "A1"); d->n[pi].unit=UNIT_KBQ_PER_ML;
  pi++; strcpy(d->n[pi].name, "L1"); d->n[pi].unit=UNIT_PER_MIN;
  pi++; strcpy(d->n[pi].name, "A2"); d->n[pi].unit=UNIT_KBQ_PER_ML;
  pi++; strcpy(d->n[pi].name, "L2"); d->n[pi].unit=UNIT_PER_MIN;
  pi++; strcpy(d->n[pi].name, "A3"); d->n[pi].unit=UNIT_KBQ_PER_ML;
  pi++; strcpy(d->n[pi].name, "L4"); d->n[pi].unit=UNIT_PER_MIN;
  pi++; strcpy(d->n[pi].name, "tDelay"); d->n[pi].unit=UNIT_MIN;
  /* Set regional fit options */
  for(ri=0; ri<d->tacNr; ri++) {
    d->r[ri].model=modelCodeIndex("fengm2");
    d->r[ri].fitNr=parNr;
    d->r[ri].dataNr=100;
    d->r[ri].start=0.0;
    d->r[ri].end=60.0;
    d->r[ri].wss=0.0E+000;
  }
  /* Set parameter values and region names */
  ri=0; 
  strcpy(d->r[ri].name, "fast");
  pi=0; d->r[ri].p[pi]=200.;
  pi++; d->r[ri].p[pi]=-0.8;
  pi++; d->r[ri].p[pi]=50.;
  pi++; d->r[ri].p[pi]=-0.2;
  pi++; d->r[ri].p[pi]=20.;
  pi++; d->r[ri].p[pi]=-0.005;
  pi++; d->r[ri].p[pi]=0.0;
  ri++; 
  strcpy(d->r[ri].name, "moder");
  pi=0; d->r[ri].p[pi]=50.;
  pi++; d->r[ri].p[pi]=-0.2;
  pi++; d->r[ri].p[pi]=50.;
  pi++; d->r[ri].p[pi]=-0.02;
  pi++; d->r[ri].p[pi]=3.;
  pi++; d->r[ri].p[pi]=-0.001;
  pi++; d->r[ri].p[pi]=0.0;
  ri++; 
  strcpy(d->r[ri].name, "slow");
  pi=0; d->r[ri].p[pi]=5.;
  pi++; d->r[ri].p[pi]=-0.05;
  pi++; d->r[ri].p[pi]=70.;
  pi++; d->r[ri].p[pi]=-0.001;
  pi++; d->r[ri].p[pi]=30.;
  pi++; d->r[ri].p[pi]=-0.0005;
  pi++; d->r[ri].p[pi]=0.0;

  statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  return TPCERROR_OK;
}
/*****************************************************************************/

/*****************************************************************************/
/** Create PAR data containing function parameters for perfect bolus TAC,
    starting from zero and approaching zero after the peak.
    @note For use in algorithm and software tests.
    @details Creates Gamma variate function (gammav) function with zero delay
     and with p[1]=1. Maximum is at 1.2 s and max value is 100 kBq/mL.
     Integral from 0 to infinity is 326.19382 (p[0]*p[2]^2).
    @return enum tpcerror (TPCERROR_OK when successful).
    @author Vesa Oikonen
    @sa parInit, parExampleTTACs, simBTAC
 */
int parExamplePerfectBolus(
  /** Pointer to initiated PAR struct, which will be allocated and filled here;
      any previous content will be deleted. */
  PAR *d,
  /** Pointer to status data; enter NULL if not needed */
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  if(verbose>0) printf("%s(par)\n", __func__);

  /* Check that required data exists */
  if(d==NULL) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_DATA);
    return TPCERROR_NO_DATA;
  }
  statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);

  int ret, pi, parNr;
  
  /* Allocate memory */
  parNr=modelParNr(modelCodeIndex("gammav"));
  ret=parAllocate(d, parNr, 1); if(ret!=TPCERROR_OK) return(ret);
  d->tacNr=1; d->parNr=4;
  d->format=PAR_FORMAT_CSV_INT;
  /* Set header fields */
  iftPut(&d->h, "studynr", "ptac1401", 1, NULL);
  iftPut(&d->h, "analysis_time", "2016-07-05 15:40:03", 1, NULL);
  iftPut(&d->h, "program", "libtpcpar (c) 2016", 1, NULL);
  iftPut(&d->h, "datafile","ptac1401.tac",1,NULL);
  iftPut(&d->h, "weighting", "no", 1, NULL);
  iftPut(&d->h, "unit", unitName(UNIT_KBQ_PER_ML), 1, NULL);
  iftPut(&d->h, "timeunit", unitName(UNIT_SEC), 1, NULL);
  /* Set parameter names */
  pi=0; strcpy(d->n[pi].name, "A"); d->n[pi].unit=UNIT_KBQ_PER_ML;
  pi++; strcpy(d->n[pi].name, "B"); d->n[pi].unit=UNIT_UNITLESS;
  pi++; strcpy(d->n[pi].name, "C"); d->n[pi].unit=UNIT_PER_SEC;
  pi++; strcpy(d->n[pi].name, "tDelay"); d->n[pi].unit=UNIT_SEC;
  /* Set region names */
  sprintf(d->r[0].name, "bolus");
  /* Set regional fit options */
  d->r[0].model=modelCodeIndex("gammav");
  d->r[0].fitNr=parNr;
  d->r[0].dataNr=100;
  d->r[0].start=0.0;
  d->r[0].end=60.0;
  d->r[0].wss=0.0E+000;
  /* Set parameter values and region names */
  pi=0; d->r[0].p[pi]=226.523;
  pi++; d->r[0].p[pi]=1.0;
  pi++; d->r[0].p[pi]=1.2;
  pi++; d->r[0].p[pi]=0.0;

  statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  return TPCERROR_OK;
}
/*****************************************************************************/

/*****************************************************************************/
