/** @file parift.c
 *  @brief IFT I/O functions for TPC parameter files.
 */
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <time.h>
#include <string.h>
/*****************************************************************************/
#include "tpcextensions.h"
#include "tpcift.h"
/*****************************************************************************/
#include "tpcpar.h"
/*****************************************************************************/

/*****************************************************************************/
/** Copy PAR data into IFT struct.
 *  @return enum tpcerror (TPCERROR_OK when successful).
 *  @author Vesa Oikonen
 */
int parToIFT(
  /** Pointer to source PAR struct, contents of which are to be copied */
  PAR *par,
  /** Pointer to initiated target IFT struct; any previous contents are 
   *  deleted */
  IFT *ift,
  /** Pointer to status data; enter NULL if not needed */
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  if(ift==NULL) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL);
    return TPCERROR_FAIL;
  }
  if(verbose>0) printf("%s():\n", __func__);
  if(par==NULL || par->tacNr<1 || par->parNr<1) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_DATA);
    return TPCERROR_NO_DATA;
  }

  /* Set decimal and item separators */
  int tointl=0;
  if(par->format==PAR_FORMAT_CSV_INT || par->format==PAR_FORMAT_TSV_INT)
    tointl=1;

  int ret;
  char tmp[128];

  /* Set content id */
  ret=iftPut(ift, "content", "parameters", 1, NULL);
  if(ret!=TPCERROR_OK) {
    statusSet(status, __func__, __FILE__, __LINE__, ret);
    return ret;
  }

  /* Copy PAR headers */
  sprintf(tmp, "%d", par->parNr);
  ret=iftPut(ift, "par_nr", tmp, 1, NULL);
  for(int i=0; i<par->h.keyNr && ret==TPCERROR_OK; i++) {
    /* not if value is missing */
    if(par->h.item[i].value==NULL || strlen(par->h.item[i].value)<1) continue;
    /* not if key is missing */
    if(par->h.item[i].key==NULL || strlen(par->h.item[i].key)<1) continue;
    /* not if this will be written later */
    if(strcasecmp(par->h.item[i].key, "model")==0) continue;
    if(strcasecmp(par->h.item[i].key, "dataNr")==0) continue;
    if(strcasecmp(par->h.item[i].key, "fitNr")==0) continue;
    if(strcasecmp(par->h.item[i].key, "fit_start")==0) continue;
    if(strcasecmp(par->h.item[i].key, "fit_end")==0) continue;
    /* ok to copy */
    ret=iftPut(ift, par->h.item[i].key, par->h.item[i].value, 1, NULL);
  }
  if(ret!=TPCERROR_OK) {
    statusSet(status, __func__, __FILE__, __LINE__, ret);
    return ret;
  }

  if(parIsModel(par)==1) {
    sprintf(tmp, "%d", par->r[0].model);
    iftPut(ift, "model", tmp, 0, NULL);
  }
  if(parIsDataNr(par)==1) {
    sprintf(tmp, "%d", par->r[0].dataNr);
    iftPut(ift, "data_nr", tmp, 0, NULL);
  }
  if(parIsFitNr(par)==1) {
    sprintf(tmp, "%d", par->r[0].fitNr);
    iftPut(ift, "fit_nr", tmp, 0, NULL);
  }
  if(parIsFitRange(par)==1) {
    sprintf(tmp, "%g min", par->r[0].start);
    if(tointl) strReplaceChar(tmp, '.', ',');
    iftPut(ift, "fit_start", tmp, 0, NULL);
    sprintf(tmp, "%g min", par->r[0].end);
    if(tointl) strReplaceChar(tmp, '.', ',');
    iftPut(ift, "fit_end", tmp, 0, NULL);
  }

  /* Copy parameter table contents into IFT */
  int ri, pi;
  for(ri=0; ri<par->tacNr; ri++) {
    sprintf(tmp, "%d", 1+ri);
    iftPut(ift, "parameter-set", tmp, 0, NULL);
    iftPut(ift, "tac_name", par->r[ri].name, 0, NULL);
    if(parIsModel(par)==2) {
      sprintf(tmp, "%d", par->r[ri].model);
      iftPut(ift, "model", tmp, 0, NULL);
    }
    if(parIsDataNr(par)==2) {
      sprintf(tmp, "%d", par->r[ri].dataNr);
      iftPut(ift, "dataNr", tmp, 0, NULL);
    }
    if(parIsFitNr(par)==2) {
      sprintf(tmp, "%d", par->r[ri].fitNr);
      iftPut(ift, "fitNr", tmp, 0, NULL);
    }
    if(parIsFitRange(par)==2) {
      sprintf(tmp, "%g min", par->r[ri].start);
      if(tointl) strReplaceChar(tmp, '.', ',');
      iftPut(ift, "fit_start", tmp, 0, NULL);
      sprintf(tmp, "%g min", par->r[ri].end);
      if(tointl) strReplaceChar(tmp, '.', ',');
      iftPut(ift, "fit_end", tmp, 0, NULL);
    }
    if(!isnan(par->r[ri].wss)) {
      sprintf(tmp, "%g", par->r[ri].wss);
      if(tointl) strReplaceChar(tmp, '.', ',');
      iftPut(ift, "wss", tmp, 0, NULL);
    }
    for(pi=0; pi<par->parNr; pi++) if(!isnan(par->r[ri].p[pi])) {
      if(par->n[pi].unit==UNIT_UNKNOWN) sprintf(tmp, "%g", par->r[ri].p[pi]);
      else sprintf(tmp, "%g %s", par->r[ri].p[pi], unitName(par->n[pi].unit));
      if(tointl) strReplaceChar(tmp, '.', ',');
      iftPut(ift, par->n[pi].name, tmp, 0, NULL);
      if(!isnan(par->r[ri].sd[pi])) {
        sprintf(tmp, "%g", par->r[ri].sd[pi]);
        if(tointl) strReplaceChar(tmp, '.', ',');
        iftPut(ift, "SD", tmp, 0, NULL);
      }
      if(!isnan(par->r[ri].cl1[pi])) {
        sprintf(tmp, "%g", par->r[ri].cl1[pi]);
        if(tointl) strReplaceChar(tmp, '.', ',');
        iftPut(ift, "95%CL1", tmp, 0, NULL);
      }
      if(!isnan(par->r[ri].cl2[pi])) {
        sprintf(tmp, "%g", par->r[ri].cl2[pi]);
        if(tointl) strReplaceChar(tmp, '.', ',');
        iftPut(ift, "95%CL2", tmp, 0, NULL);
      }
    } // next parameter
  } // next tac
  
  /* Quit */
  statusSet(status, __func__, __FILE__, __LINE__, ret);
  return(ret);
}
/*****************************************************************************/

/*****************************************************************************/
/** Write PAR data into specified file pointer in IFT format.
 *  @return enum tpcerror (TPCERROR_OK when successful).
 *  @author Vesa Oikonen
 */
int parWriteIFT(
  /** Pointer to source PAR struct, contents of which are to be written */
  PAR *par,
  /** Output file pointer */
  FILE *fp,
  /** Pointer to status data; enter NULL if not needed */
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  if(fp==NULL) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_CANNOT_WRITE);
    return TPCERROR_CANNOT_WRITE;
  }
  if(verbose>0) printf("%s()\n", __func__);
  if(par==NULL || par->tacNr<1 || par->parNr<1) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_DATA);
    return TPCERROR_NO_DATA;
  }

  int ret;
  IFT ift;
  iftInit(&ift);
  ret=parToIFT(par, &ift, status);
  if(ret!=TPCERROR_OK) {iftFree(&ift); return ret;}
  ret=iftWrite(&ift, fp, status);
  iftFree(&ift);
  if(ret!=TPCERROR_OK) return ret;

  /* Quit */
  statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  return(TPCERROR_OK);
}
/*****************************************************************************/

/*****************************************************************************/
/** Copy parameters data in IFT into PAR struct. IFT contents must be in
 *  specific format and order.
 *  @return enum tpcerror (TPCERROR_OK when successful).
 *  @author Vesa Oikonen
 */
int parFromIFT(
  /** Pointer to initiated target PAR struct; any previous contents are 
   *  deleted */
  PAR *par,
  /** Pointer to source IFT struct, contents of which are to be copied */
  IFT *ift,
  /** Pointer to status data; enter NULL if not needed */
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  if(par==NULL) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL);
    return TPCERROR_FAIL;
  }
  if(verbose>0) printf("%s()\n", __func__);
  if(ift==NULL || ift->keyNr<1) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_DATA);
    return TPCERROR_NO_DATA;
  }

  /* Verify that IFT contains results */
  if(strcasecmp(ift->item[0].key, "content") ||
     strcasecmp(ift->item[0].value, "parameters")
  ) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_UNSUPPORTED);
    return TPCERROR_UNSUPPORTED;
  }

  int ret, parNr, tacNr;
  /* Read the nr of parameters */
  if(iftGetInt(ift, iftFindKey(ift, "par_nr", 0), &parNr)!=0 || parNr<1) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_UNSUPPORTED);
    return TPCERROR_UNSUPPORTED;
  }
  if(verbose>5) printf("parNr := %d\n", parNr);  
  /* Read the nr of parameter sets (tacs) */
  tacNr=iftFindNrOfKeys(ift, "parameter-set");
  if(tacNr<1) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_UNSUPPORTED);
    return TPCERROR_UNSUPPORTED;
  }
  if(verbose>5) printf("tacNr := %d\n", tacNr);  

  /* Allocate memory for PAR */
  parFree(par);
  ret=parAllocate(par, parNr, tacNr);
  if(ret!=TPCERROR_OK) {
    statusSet(status, __func__, __FILE__, __LINE__, ret);
    return ret;
  }
  par->parNr=parNr; par->tacNr=tacNr;

  /* Read header contents */
  for(int i=1, ret=0; i<ift->keyNr && ret==TPCERROR_OK; i++) {
    /* not if value is missing */
    if(ift->item[i].value==NULL || strlen(ift->item[i].value)<1) continue;
    /* not if key is missing */
    if(ift->item[i].key==NULL || strlen(ift->item[i].key)<1) continue;
    /* end of header? */
    if(strcasecmp(ift->item[i].key, "parameter-set")==0) break;
    /* not needed */
    if(strcasecmp(ift->item[i].key, "par_nr")==0) continue;
    if(strcasecmp(ift->item[i].key, "format")==0) continue;
    if(strcasecmp(ift->item[i].key, "tac_nr")==0) continue;
    /* ok to copy */
    ret=iftPut(&par->h, ift->item[i].key, ift->item[i].value, 1, NULL);
  }
  if(ret!=TPCERROR_OK) {
    statusSet(status, __func__, __FILE__, __LINE__, ret);
    return ret;
  }

  /* Read parameter values */
  int i, pi=0, ri=0; ret=0;
  char tmp[128];
  i=iftFindKey(ift, "parameter-set", 0);
  for(i++; i<ift->keyNr && ret==TPCERROR_OK; i++) {
    if(verbose>60)
      printf("i=%d key='%s' value='%s'\n",
             i, ift->item[i].key, ift->item[i].value);
    /* new parameter-set? */
    if(strcasecmp(ift->item[i].key, "parameter-set")==0) {
      ri++; pi=0; if(ri<par->tacNr) continue; else break;}
    /* parameter set info? */
    if(strcasecmp(ift->item[i].key, "tac_name")==0) {
      strncpy(par->r[ri].name, ift->item[i].value, MAX_TACNAME_LEN);
      par->r[ri].name[MAX_TACNAME_LEN]='\0';
      continue;
    }
    if(strcasecmp(ift->item[i].key, "wss")==0) {
      if(iftGetDouble(ift, i, &par->r[ri].wss)) {ret++; break;}
      continue;
    }
    if(strcasecmp(ift->item[i].key, "model")==0) {
      if(iftGetUInt(ift, i, &par->r[ri].model)) {ret++; break;}
      continue;
    }
    if(strcasecmp(ift->item[i].key, "data_nr")==0) {
      if(iftGetInt(ift, i, &par->r[ri].dataNr)) {ret++; break;}
      continue;
    }
    if(strcasecmp(ift->item[i].key, "fit_nr")==0) {
      if(iftGetInt(ift, i, &par->r[ri].fitNr)) {ret++; break;}
      continue;
    }
    if(strcasecmp(ift->item[i].key, "fit_start")==0) {
      if(iftGetDouble(ift, i, &par->r[ri].start)) {ret++; break;}
      continue;
    }
    if(strcasecmp(ift->item[i].key, "fit_end")==0) {
      if(iftGetDouble(ift, i, &par->r[ri].end)) {ret++; break;}
      continue;
    }
    /* SD or CL ? */
    if(strcasecmp(ift->item[i].key, "SD")==0) {
      if(iftGetDouble(ift, i, &par->r[ri].sd[pi])) {ret++; break;}
      continue;
    }
    if(strcasecmp(ift->item[i].key, "95%CL1")==0) {
      if(iftGetDouble(ift, i, &par->r[ri].cl1[pi])) {ret++; break;}
      continue;
    }
    if(strcasecmp(ift->item[i].key, "95%CL2")==0) {
      if(iftGetDouble(ift, i, &par->r[ri].cl2[pi])) {ret++; break;}
      continue;
    }
    /* Ok this should be parameter */
    if(pi>=par->parNr) {ret++; break;}
    strncpy(par->n[pi].name, ift->item[i].key, MAX_PARNAME_LEN);
    par->n[pi].name[MAX_PARNAME_LEN]='\0';
    if(iftGetDouble(ift, i, &par->r[ri].p[pi])) {ret++; break;}
    /* next token, if one exists, should be the unit */
    if(strTokenNr(ift->item[i].value, " \t")==2) {
      if(strTokenNCpy(ift->item[i].value, " \t", 2, tmp, 64)>0)
        par->n[pi].unit=unitIdentify(tmp);
    }
    pi++;
  }
  if(ret!=TPCERROR_OK) {
    statusSet(status, __func__, __FILE__, __LINE__, ret);
    return ret;
  }

  /* Set format in struct */
  par->format=PAR_FORMAT_IFT;

  /* Quit */
  statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  return(TPCERROR_OK);
}
/*****************************************************************************/

/*****************************************************************************/
