/** @file pario.c
 *  @brief I/O functions for TPC parameter files.
 *  @copyright (c) Turku PET Centre
 *  @test Test reading old .res and .fit files.
 */
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <time.h>
#include <string.h>
/*****************************************************************************/
#include "tpcextensions.h"
#include "tpccsv.h"
#include "tpcift.h"
/*****************************************************************************/
#include "tpcpar.h"
/*****************************************************************************/

/*****************************************************************************/
/** Text representations of PAR file format.
   @sa tpcpar.h */
static const char *par_format[] = {
  "Unknown",     ///< PAR_FORMAT_UNKNOWN
  "RES",         ///< PAR_FORMAT_RES
  "FIT",         ///< PAR_FORMAT_FIT
  "IFT",         ///< PAR_FORMAT_IFT
  "CSV-INT",     ///< PAR_FORMAT_CSV_INT
  "CSV-UK",      ///< PAR_FORMAT_CSV_UK
  "TSV-INT",     ///< PAR_FORMAT_TSV_INT
  "TSV-UK",      ///< PAR_FORMAT_TSV_UK
  "XML",         ///< PAR_FORMAT_XML
  "HTML",        ///< PAR_FORMAT_HTML
0};
/*****************************************************************************/
/** Default file name extensions of PAR file formats. */
static const char *par_fn_ext[] = {
  ".txt",    // PAR_FORMAT_UNKNOWN
  ".res",    // PAR_FORMAT_RES
  ".fit",    // PAR_FORMAT_FIT
  ".ift",    // PAR_FORMAT_IFT
  ".csv",    // PAR_FORMAT_CSV_INT
  ".csv",    // PAR_FORMAT_CSV_UK
  ".tsv",    // PAR_FORMAT_TSV_INT
  ".tsv",    // PAR_FORMAT_TSV_UK
  ".xml",    // PAR_FORMAT_XML
  ".html",   // PAR_FORMAT_HTML
0};
/*****************************************************************************/

/*****************************************************************************/
/** Return pointer to PAR file format description with the format code.
    @return pointer to the PAR file format string.
    @sa parformat, parFormatIdentify, parFormatFromExtension
    @author Vesa Oikonen
 */
char *parFormattxt(
  /** PAR format code. */
  parformat c
) {
  if(c<PAR_FORMAT_UNKNOWN || c>=PAR_FORMAT_LAST) return NULL;
  return (char*)par_format[c];
}
/*****************************************************************************/

/*****************************************************************************/
/** Identify the string representations of the PAR file format.
    @return enum parformat, or 0 (enum PAR_FORMAT_UNKNOWN) if not identified.
    @author Vesa Oikonen
    @sa parFormattxt, parformat, parFormatFromExtension, parRead
 */
int parFormatIdentify(
  /** PAR format as a string. */
  const char *s
) {
  if(s==NULL || strlen(s)<1) return PAR_FORMAT_UNKNOWN;
  /* Try if string can be found directly in the table */
  for(int i=0; i<PAR_FORMAT_LAST; i++) {
    if(strcasecmp(par_format[i], s)==0) return i;
  }
  /* Format string is not following TPC standard, lets try something else */
  if(     strcasecmp(s, "dat")==0)              return PAR_FORMAT_TSV_UK;
  else if(strcasecmp(s, "txt")==0)              return PAR_FORMAT_TSV_UK;
  else if(strcasecmp(s, "par")==0)              return PAR_FORMAT_CSV_UK;
  else if(strcasecmp(s, "lim")==0)              return PAR_FORMAT_TSV_UK;
  else if(strcasecmp(s, "htm")==0)              return PAR_FORMAT_HTML;
  else if(strcasecmp(s, "csv")==0)              return PAR_FORMAT_CSV_UK;
  else if(strcasecmp(s, "tsv")==0)              return PAR_FORMAT_TSV_UK;

  return PAR_FORMAT_UNKNOWN;
}
/*****************************************************************************/

/*****************************************************************************/
/** Identify the PAR file format from the extension of file name.
    @return enum parformat, or 0 (enum PAR_FORMAT_UNKNOWN) if not identified.
    @author Vesa Oikonen
    @sa parFormattxt, parformat, parFormatIdentify, parWrite
 */
int parFormatFromExtension(
  /** Pointer to file name with or without path; mere extension is not accepted here.
      For example /path/filename.res or filename.fit.bak are accepted. */
  const char *s
) {
  if(s==NULL || strlen(s)<1) return PAR_FORMAT_UNKNOWN;
  /* Get pointer to the extensions */
  char *cptr=filenameGetExtensions(s);
  if(cptr==NULL) return PAR_FORMAT_UNKNOWN;
  /* Try to find some known extensions */
  if(strcasestr(cptr, ".res")) return PAR_FORMAT_RES;
  if(strcasestr(cptr, ".fit")) return PAR_FORMAT_FIT;
  if(strcasestr(cptr, ".ift")) return PAR_FORMAT_IFT;
  if(strcasestr(cptr, ".csv")) return PAR_FORMAT_CSV_UK;
  if(strcasestr(cptr, ".tsv")) return PAR_FORMAT_TSV_UK;
  if(strcasestr(cptr, ".txt")) return PAR_FORMAT_TSV_UK;
  if(strcasestr(cptr, ".par")) return PAR_FORMAT_TSV_UK;
  if(strcasestr(cptr, ".lim")) return PAR_FORMAT_TSV_UK;
  if(strcasestr(cptr, ".dat")) return PAR_FORMAT_TSV_UK;
  if(strcasestr(cptr, ".xml")) return PAR_FORMAT_XML;
  if(strcasestr(cptr, ".htm")) return PAR_FORMAT_HTML;
  return PAR_FORMAT_UNKNOWN;
}
/*****************************************************************************/

/*****************************************************************************/
/** Return pointer to default PAR file name extension, including the dot, based on the PAR format code.
    @return pointer to the file name extension string.
    @sa parformat, parFormatFromExtension, parFormattxt, parWrite
    @author Vesa Oikonen
 */
char *parDefaultExtension(
  /** PAR format code */
  parformat c
) {
  if(c<PAR_FORMAT_UNKNOWN || c>=PAR_FORMAT_LAST) return NULL;
  return (char*)par_fn_ext[c];
}
/*****************************************************************************/

/*****************************************************************************/
/** Write PAR data into specified file in specified format.
    @return enum tpcerror (TPCERROR_OK when successful).
    @author Vesa Oikonen
    @sa parRead, parFree, parAllocate, parFormatFromExtension
 */
int parWrite(
  /** Pointer to PAR struct, contents of which are to be written. */
  PAR *par,
  /** Output file pointer. */
  FILE *fp,
  /** File format code; enter PAR_FORMAT_UNKNOWN (0) to write data in the format specified inside
      the PAR struct. */
  parformat format,
  /** Write (1) or do not write (0) also extra header fields found in PAR;
      only effective with CSV formats. */
  int extra,
  /** Pointer to status data; enter NULL if not needed. */
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  if(fp==NULL) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_CANNOT_WRITE);
    return TPCERROR_CANNOT_WRITE;
  }
  if(par==NULL || par->tacNr<1 || par->parNr<1) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_DATA);
    return TPCERROR_NO_DATA;
  }
  if(par->tacNr>par->_tacNr || par->parNr>par->_parNr) { // programmers fail
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL);
    return TPCERROR_FAIL;
  }

  /* Determine and verify the write format */
  if(format<=PAR_FORMAT_UNKNOWN || format>=PAR_FORMAT_LAST)
    format=par->format;
  if(format<=PAR_FORMAT_UNKNOWN || format>=PAR_FORMAT_LAST) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_FORMAT);
    return TPCERROR_INVALID_FORMAT;
  }
  if(verbose>0) {
    printf("%s():\n", __func__);
    printf("writing %d parameters from %d TACs in %s format\n",
      par->parNr, par->tacNr, parFormattxt(format));
    fflush(stdout);
  }

  /* Write file */
  int ret;
  switch(format) {
    case PAR_FORMAT_CSV_INT:
      /* __attribute__((fallthrough)); */
    case PAR_FORMAT_CSV_UK:
      /* __attribute__((fallthrough)); */
    case PAR_FORMAT_TSV_INT:
      /* __attribute__((fallthrough)); */
    case PAR_FORMAT_TSV_UK:
      ret=parWriteCSV(par, fp, format, extra, status);
      break;
    case PAR_FORMAT_RES:
      ret=parWriteRES(par, fp, status);
      break;
    case PAR_FORMAT_FIT:
      ret=parWriteFIT(par, fp, status);
      break;
    case PAR_FORMAT_IFT:
      ret=parWriteIFT(par, fp, status);
      break;
    case PAR_FORMAT_XML:
      ret=parWriteXML(par, fp, status);
      break;
    default:
      ret=TPCERROR_INVALID_FORMAT;
  }
  fflush(fp); fflush(stderr);
  
  /* Quit */
  statusSet(status, __func__, __FILE__, __LINE__, ret);
  return(ret);
}
/*****************************************************************************/

/*****************************************************************************/
/** Read parameter file contents into PAR data structure.
    This function reads RES and FIT format, and some CSV formats.
    @author Vesa Oikonen
    @return code tpcerror, TPCERROR_OK (0) when successful.
    @sa parWrite, parInit, parFree
 */
int parRead(
  /** Pointer to initiated PAR struct where parameter data will be written;
      any old content is deleted. Call parInit() before first use. */
  PAR *par,
  /** Pointer to the file name; this string is not modified. */
  const char *fname,
  /** Pointer to status data; enter NULL if not needed. */
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  if(fname==NULL || strlen(fname)<1 || par==NULL) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL);
    return TPCERROR_FAIL;
  }
  if(verbose>1) {printf("%s(%s)\n", __func__, fname); fflush(stdout);}
  int format=PAR_FORMAT_UNKNOWN;

  /* Delete any previous data */
  parFree(par);

  /* Try to read the file as CSV file */
  if(verbose>1) {printf("opening file\n"); fflush(stdout);}
  FILE *fp;
  fp=fopen(fname, "r");
  if(fp==NULL) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_CANNOT_OPEN);
    return TPCERROR_CANNOT_OPEN;
  }
  int ret;
  CSV csv; csvInit(&csv);
  ret=csvRead(&csv, fp, status);
  fclose(fp);
  if(ret!=TPCERROR_OK) {
    statusSet(status, __func__, __FILE__, __LINE__, ret);
    csvFree(&csv); return ret;
  }
  if(verbose>20) {
    printf("\n --- CSV contents ---\n");
    csvWrite(&csv, 0, stdout, status);
  }

  /* Try to identify file format from a magic number in the start */
  if(verbose>1) {printf("checking for magic number\n"); fflush(stdout);}
  if(strncasecmp(csv.c[0].content, "FIT1", 4)==0) {
    format=PAR_FORMAT_FIT;
  } else if(strncasecmp(csv.c[0].content, "<?xml version", 5)==0) {
    format=PAR_FORMAT_XML;
  } else if(strcasecmp(csv.c[0].content, "Parameters")==0) {
    /* now determine the field and decimal separators */
    if(csv.separator==';') format=PAR_FORMAT_CSV_INT;
    else if(csv.separator==',') format=PAR_FORMAT_CSV_UK;
    else if(csv.separator=='\t' || csv.separator==' ') {
      // either UK or INT, we'll decide it later
      format=PAR_FORMAT_TSV_UK;
    }
  }
  if(verbose>3) {printf("format := %s\n", parFormattxt(format)); fflush(stdout);}

  /* If identified already, then read data and return */
  if(format!=PAR_FORMAT_UNKNOWN) {
    if(verbose>1) {printf("format := %s\n", parFormattxt(format)); fflush(stdout);}
    IFT hdr; iftInit(&hdr);
    /* These formats may contain additional header information in comment
       lines; read headers from those into TAC struct for further processing */
    if(format==PAR_FORMAT_CSV_UK || format==PAR_FORMAT_CSV_INT ||
       format==PAR_FORMAT_TSV_UK || format==PAR_FORMAT_TSV_INT
    ) {
      fp=fopen(fname, "r");
      if(fp!=NULL) {
        if(iftRead(&hdr, fp, 1, 2, status)!=TPCERROR_OK) iftFree(&hdr); 
        fclose(fp);
      }
      /* Header may not exist, therefore do not worry about errors here */
    }
    /* FIT (and RES) formats are easier to read partially from IFT struct */
    if(format==PAR_FORMAT_FIT) {
      fp=fopen(fname, "r");
      if(fp==NULL) ret=TPCERROR_CANNOT_OPEN; // these formats must have header
      else {
        ret=iftRead(&hdr, fp, 0, 1, status); 
        fclose(fp);
      }
      if(ret!=TPCERROR_OK) {
        csvFree(&csv); iftFree(&hdr);
        statusSet(status, __func__, __FILE__, __LINE__, ret);
        return ret;
      }
    }
    switch(format) {
      case PAR_FORMAT_FIT:
        ret=parReadFIT(par, &csv, &hdr, status);
        break;
      case PAR_FORMAT_CSV_INT:
        /* __attribute__((fallthrough)); */
      case PAR_FORMAT_CSV_UK:
        /* __attribute__((fallthrough)); */
      case PAR_FORMAT_TSV_INT:
        /* __attribute__((fallthrough)); */
      case PAR_FORMAT_TSV_UK:
        /* __attribute__((fallthrough)); */
      default:
        ret=parReadCSV(par, &csv, &hdr, status);
        break;
    }
    csvFree(&csv); iftFree(&hdr);
    if(ret!=TPCERROR_OK) {
      statusSet(status, __func__, __FILE__, __LINE__, ret);
      parFree(par); return ret;
    }
    /* Set study number, if not yet set, based on file name */
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
    return TPCERROR_OK;
  }

  /* The rest of the formats are a bit more difficult to identify */

  /* Try to read as IFT */
  if(verbose>3) {printf("trying IFT format\n"); fflush(stdout);}
  fp=fopen(fname, "r");
  if(fp!=NULL) {
    IFT hdr; iftInit(&hdr);
    iftRead(&hdr, fp, 1, 1, status); fclose(fp);
    if(hdr.keyNr>0) {
      if(!strcasecmp(hdr.item[0].key, "content") && !strcasecmp(hdr.item[0].value, "parameters"))
        format=PAR_FORMAT_IFT;
    }
    if(format==PAR_FORMAT_IFT) {
      if(verbose>1) printf("format := %s\n", parFormattxt(format));
      ret=parFromIFT(par, &hdr, status);
      csvFree(&csv); iftFree(&hdr);
      if(ret!=TPCERROR_OK) {parFree(par); return ret;}
      /* Set study number, if not yet set, based on file name */
      statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
      return TPCERROR_OK;
    }
    if(verbose>3) printf("not IFT\n");
    iftFree(&hdr);
  }

  /* Try to read as RES file */
  if(verbose>3) {printf("trying RES format\n"); fflush(stdout);}
  /* Header is obligatory, read that to IFT */
  fp=fopen(fname, "r");
  if(fp!=NULL) {
    IFT hdr; iftInit(&hdr);
    ret=iftRead(&hdr, fp, 0, 1, status); fclose(fp);
    if(ret!=TPCERROR_OK) {
      csvFree(&csv); iftFree(&hdr);
      statusSet(status, __func__, __FILE__, __LINE__, ret);
      return ret;
    }
    ret=parReadRES(par, &csv, &hdr, status);
    iftFree(&hdr);
    if(ret==TPCERROR_OK) {
      csvFree(&csv);
      /* Set study number, if not yet set, based on file name */
      statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
      return TPCERROR_OK;
    }
  }

  /* Currently any other format cannot be read */
  if(verbose>3) {printf("currently not supported format\n"); fflush(stdout);}
  csvFree(&csv);
  parFree(par);
  statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_UNSUPPORTED);
  return TPCERROR_UNSUPPORTED;
}
/*****************************************************************************/

/*****************************************************************************/
/** List parameter limits in PAR structure.
    @sa parWriteLimits, parWrite, parFree
 */
void parListLimits(
  /** Pointer to PAR structure, from which the limits are printed. */
  PAR *par,
  /** File pointer for the output; usually stdout. */
  FILE *fp
) {
  //printf("%s()\n", __func__);
  if(par==NULL || par->parNr<1) {fprintf(fp, "No parameter limits.\n"); return;}

  fprintf(fp, "Parameter limits:\n");
  for(int i=0; i<par->parNr; i++) {
    if(strlen(par->n[i].name)>0) fprintf(fp, "%s", par->n[i].name);
    else fprintf(fp, "p%d", 1+i);
    fprintf(fp, "\t%g\t%g\n", par->n[i].lim1, par->n[i].lim2);
  }
  fprintf(fp, "\n");
  return;
}
/*****************************************************************************/

/*****************************************************************************/
/** Write parameter constraints from PAR structure into specified file.
    @return enum tpcerror (TPCERROR_OK when successful).
    @author Vesa Oikonen
    @sa parReadLimits, parWrite, parFree, parListLimits
 */
int parWriteLimits(
  /** Pointer to PAR structure, from which the limits are to be written. */
  PAR *par,
  /** Output file name. */
  const char *fname,
  /** Verbose level; if zero, then nothing is printed to stderr or stdout */
  const int verbose
) {
  if(verbose>0) {printf("%s(%s)\n", __func__, fname); fflush(stdout);}
  if(par==NULL || par->parNr<1) return TPCERROR_NO_DATA;
  if(strlen(fname)<1) return(TPCERROR_INVALID_FILENAME);

  /* Copy limits into IFT struct */
  IFT ift; iftInit(&ift);
  char pname[MAX_PARNAME_LEN+10];
  for(int i=0; i<par->parNr; i++) {
    if(!isnan(par->n[i].lim1)) {
      if(strlen(par->n[i].name)>0) strcpy(pname, par->n[i].name); else sprintf(pname, "p%d", 1+i);
      strcat(pname, "_lower"); iftPutDouble(&ift, pname, par->n[i].lim1, 0, NULL);
    }
    if(!isnan(par->n[i].lim2)) {
      if(strlen(par->n[i].name)>0) strcpy(pname, par->n[i].name); else sprintf(pname, "p%d", 1+i);
      strcat(pname, "_upper"); iftPutDouble(&ift, pname, par->n[i].lim2, 0, NULL);
    }
    if(!isnan(par->n[i].tol)) {
      if(strlen(par->n[i].name)>0) strcpy(pname, par->n[i].name); else sprintf(pname, "p%d", 1+i);
      strcat(pname, "_tol"); iftPutDouble(&ift, pname, par->n[i].tol, 0, NULL);
    }
  }
  if(verbose>1) printf("  keyNr=%d\n", ift.keyNr);
  if(ift.keyNr<1) {iftFree(&ift); return TPCERROR_NO_DATA;}

  /* Open file */
  if(verbose>1) printf("  opening %s\n", fname);
  FILE *fp=fopen(fname, "w");
  if(fp==NULL) {iftFree(&ift); return TPCERROR_CANNOT_WRITE;}

  /* Write IFT */
  if(verbose>1) printf("  writing in %s\n", fname);
  if(iftWrite(&ift, fp, NULL)!=TPCERROR_OK) {
    iftFree(&ift); fclose(fp); return TPCERROR_CANNOT_WRITE;
  }
  iftFree(&ift); fclose(fp);

  return TPCERROR_OK;
}
/*****************************************************************************/

/*****************************************************************************/
/** Read the specified file and try to find constraints and tolerance for parameters listed in PAR structure.
    @return enum tpcerror (TPCERROR_OK when successful).
    @author Vesa Oikonen
    @sa parWriteLimits, parWrite, parFree
 */
int parReadLimits(
  /** Pointer to PAR structure, into which the limits are saved. This must also list the parameter 
      names that are searched for from the file. */
  PAR *par,
  /** Input file name. */
  const char *fname,
  /** Verbose level; if zero, then nothing is printed to stderr or stdout. */
  const int verbose
) {
  if(verbose>0) {printf("%s(%s)\n", __func__, fname); fflush(stdout);}
  if(par==NULL || par->parNr<1) return TPCERROR_NO_DATA;
  if(strlen(fname)<1) return(TPCERROR_INVALID_FILENAME);

  /* Open file */
  if(verbose>1) printf("  opening %s\n", fname);
  FILE *fp=fopen(fname, "r");
  if(fp==NULL) return TPCERROR_CANNOT_READ;

  /* Read file into IFT struct */
  if(verbose>1) printf("  reading into IFT\n");
  IFT ift; iftInit(&ift);
  if(iftRead(&ift, fp, 1, 0, NULL)!=TPCERROR_OK) {fclose(fp); return TPCERROR_CANNOT_READ;}
  fclose(fp);
  if(verbose>1) printf("  keyNr=%d\n", ift.keyNr);

  /* Search for parameter limits in the IFT */
  char pname[MAX_PARNAME_LEN+10];
  int n=0;
  double f;
  for(int i=0; i<par->parNr; i++) {
    if(strlen(par->n[i].name)>0) strcpy(pname, par->n[i].name);
    else sprintf(pname, "p%d", 1+i);
    strcat(pname, "_lower");
    if(iftGetDoubleValue(&ift, pname, 0, &f)>=0 && !isnan(f)) {par->n[i].lim1=f; n++;}
    if(verbose>2) printf("  %s -> %g\n", pname, f);
    pname[strlen(pname)-6]=(char)0; strcat(pname, "_upper");
    if(iftGetDoubleValue(&ift, pname, 0, &f)>=0 && !isnan(f)) {par->n[i].lim2=f; n++;}
    if(verbose>2) printf("  %s -> %g\n", pname, f);
    pname[strlen(pname)-6]=(char)0; strcat(pname, "_tol");
    if(iftGetDoubleValue(&ift, pname, 0, &f)>=0 && !isnan(f)) {par->n[i].tol=f; n++;}
    if(verbose>2) printf("  %s -> %g\n", pname, f);
  }
  iftFree(&ift);
  if(verbose>1) printf("  n=%d\n", n);
  if(n==0) return TPCERROR_NO_DATA;

  return TPCERROR_OK;
}
/*****************************************************************************/

/*****************************************************************************/
