/** @file hcio.c
 *  @brief I/O functions for HRRT and HR+ head curve format.
 */
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include "tpcift.h"
#include "tpcisotope.h"
#include "tpccsv.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <time.h>
#include <string.h>
/*****************************************************************************/
#include "tpctac.h"
/*****************************************************************************/

/*****************************************************************************/
/** Read HRRT head curve (.hc) format from CSV struct into TAC struct.
 *  @return enum tpcerror (TPCERROR_OK when successful).
 *  @author Vesa Oikonen
 */
int tacReadHRRTHC(
  /** Pointer to TAC struct, contents of which are to be written. */
  TAC *tac,
  /** Pointer to CSV from which data is read; if it contains only one column,
   *  then it is assumed to represent the first y column and x column is not
   *  filled. */
  CSV *csv,
  /** Pointer to status data; enter NULL if not needed */
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  if(tac==NULL) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL);
    return TPCERROR_FAIL;
  }
  if(verbose>0) printf("%s()\n", __func__);

  tacFree(tac);

  if(csv==NULL || csv->row_nr<2 || csv->col_nr<4) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_DATA);
    return TPCERROR_NO_DATA;
  }  

  /* Check from the title fields that data indeed is HRRT HC data */
  int ret=0;
  if(strcasecmp(csv->c[0].content, "Singles")!=0) ret++;
  if(strcasecmp(csv->c[1].content, "Randoms")!=0) ret++;
  if(strcasecmp(csv->c[2].content, "Prompts")!=0) ret++;
  if(strncasecmp(csv->c[3].content, "Time(ms)", 4)!=0) ret++;
  if(ret>0) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_FORMAT);
    return TPCERROR_INVALID_FORMAT;
  }

  /* Allocate memory for TAC data */
  ret=tacAllocate(tac, csv->row_nr-1, 3);
  statusSet(status, __func__, __FILE__, __LINE__, ret);
  if(ret!=TPCERROR_OK) return ret;
  tac->tacNr=3; tac->sampleNr=csv->row_nr-1;
  if(verbose>2) printf("  sampleNr=%d\n", tac->sampleNr);

  /* Set time type */
  tac->isframe=0;
  /* Get time unit from the last title field, inside parens */
  char *cptr=strchr(csv->c[3].content, '(');
  if(cptr==NULL) {
    tac->tunit=UNIT_UNKNOWN;
  } else {
    char *tmp=strdup(cptr+1);
    cptr=strrchr(tmp, ')'); if(cptr!=NULL) *cptr='\0';
    tac->tunit=unitIdentify(tmp);
    free(tmp);
  }
  /* Set concentration unit */
  tac->cunit=UNIT_KCPS;

  /* Set TAC names */
  strcpy(tac->c[0].name, "Singles");
  strcpy(tac->c[1].name, "Randoms");
  strcpy(tac->c[2].name, "Prompts");

  /* Copy x and y data from CSV into TAC struct */
  double v; int oknr=0;
  for(int i=4; i<csv->nr; i++) {
    if(csv->c[i].row<1 || csv->c[i].row>tac->sampleNr || csv->c[i].col>3) 
      continue;
    v=atofVerified(csv->c[i].content); if(!isnan(v)) oknr++;
    if(csv->c[i].col==3) tac->x[csv->c[i].row-1]=v;
    else tac->c[csv->c[i].col].y[csv->c[i].row-1]=v;
  }
  if(oknr<4) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_DATA);
    return TPCERROR_NO_DATA;
  }

  /* Set file format */
  tac->format=TAC_FORMAT_HRRT_HC;

  statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  return(TPCERROR_OK);
}
/*****************************************************************************/

/*****************************************************************************/
/** Read HR+ head curve (.r) format from CSV struct into TAC struct.
 *  @return enum tpcerror (TPCERROR_OK when successful).
 *  @author Vesa Oikonen
 */
int tacReadHRPLUSHC(
  /** Pointer to TAC struct, contents of which are to be written. */
  TAC *tac,
  /** Pointer to CSV from which data is read; if it contains only one column,
   *  then it is assumed to represent the first y column and x column is not
   *  filled. */
  CSV *csv,
  /** Pointer to status data; enter NULL if not needed */
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  if(tac==NULL) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL);
    return TPCERROR_FAIL;
  }
  if(verbose>0) printf("%s()\n", __func__);

  tacFree(tac);

  if(csv==NULL || csv->row_nr<2 || csv->col_nr<8) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_DATA);
    return TPCERROR_NO_DATA;
  }  

  /* Check from the title fields that data indeed is HRRT HC data */
  int ret=0;
  if(strcasecmp(csv->c[0].content, "time")!=0) ret++;
  if(strcasecmp(csv->c[1].content, "prompt")!=0) ret++;
  if(strcasecmp(csv->c[2].content, "delayed")!=0) ret++;
  if(strcasecmp(csv->c[3].content, "p_rate")!=0) ret++;
  if(strcasecmp(csv->c[4].content, "d_rate")!=0) ret++;
  if(strcasecmp(csv->c[5].content, "dtime")!=0) ret++;
  if(strcasecmp(csv->c[6].content, "frame")!=0) ret++;
  if(strcasecmp(csv->c[7].content, "singles")!=0) ret++;
  if(ret>0) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_FORMAT);
    return TPCERROR_INVALID_FORMAT;
  }

  /* Allocate memory for TAC data */
  ret=tacAllocate(tac, csv->row_nr-1, 7);
  statusSet(status, __func__, __FILE__, __LINE__, ret);
  if(ret!=TPCERROR_OK) return ret;
  tac->tacNr=7; tac->sampleNr=csv->row_nr-1;
  if(verbose>2) printf("  sampleNr=%d\n", tac->sampleNr);

  /* Set time type */
  tac->isframe=0;
  /* Set time unit */
  tac->tunit=UNIT_MSEC;
  /* Set concentration unit */
  tac->cunit=UNIT_KCPS;

  /* Set TAC names */
  strcpy(tac->c[0].name, "prompt");
  strcpy(tac->c[1].name, "delayed");
  strcpy(tac->c[2].name, "p_rate");
  strcpy(tac->c[3].name, "d_rate");
  strcpy(tac->c[4].name, "dtime");
  strcpy(tac->c[5].name, "frame");
  strcpy(tac->c[6].name, "singles");

  /* Copy x and y data from CSV into TAC struct */
  double v; int oknr=0;
  for(int i=8; i<csv->nr; i++) {
    if(csv->c[i].row<1 || csv->c[i].row>tac->sampleNr || csv->c[i].col>7) 
      continue;
    v=atofVerified(csv->c[i].content); if(!isnan(v)) oknr++;
    if(csv->c[i].col==0) tac->x[csv->c[i].row-1]=v;
    else tac->c[csv->c[i].col-1].y[csv->c[i].row-1]=v;
  }
  if(oknr<8) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_DATA);
    return TPCERROR_NO_DATA;
  }

  /* Set file format */
  tac->format=TAC_FORMAT_HRPLUS_HC;

  statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  return(TPCERROR_OK);
}
/*****************************************************************************/

/*****************************************************************************/
