/** @file simpleio.c
 *  @brief I/O functions for simple ASCII TAC file format.
 */
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include "tpcift.h"
#include "tpcisotope.h"
#include "tpccsv.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <time.h>
#include <string.h>
/*****************************************************************************/
#include "tpctac.h"
/*****************************************************************************/

/*****************************************************************************/
/** Write TAC data into specified file pointer in Simple format.
    @return enum tpcerror (TPCERROR_OK when successful).
    @author Vesa Oikonen
    @sa tacReadSimple, tacWriteCSV
 */
int tacWriteSimple(
  /** Pointer to TAC struct, contents of which are to be written. */
  TAC *tac,
  /** File pointer. */
  FILE *fp,
  /** Write (1) or do not write (0) also extra header fields found in IFT. */
  int extra,
  /** Pointer to status data; enter NULL if not needed. */
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  if(fp==NULL) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_CANNOT_WRITE);
    return TPCERROR_CANNOT_WRITE;
  }
  if(tac==NULL || tac->tacNr<1 || tac->sampleNr<1) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_DATA);
    return TPCERROR_NO_DATA;
  }
  if(verbose>0) {printf("%s(%d)\n", __func__, extra); fflush(stdout);}


  int fi, ri, n, prec=6;
  double v;

  /* Write data */
  for(fi=0; fi<tac->sampleNr; fi++) {
    /* Time (x) */
    if(tac->isframe==0) v=tac->x[fi]; else v=0.5*(tac->x1[fi]+tac->x2[fi]);
    if(isnan(v)) n=fprintf(fp, "."); else n=fprintf(fp, "%.5f", v);
    if(n<1) {
      statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_CANNOT_WRITE);
      return TPCERROR_CANNOT_WRITE;
    }
    /* Concentrations (y values) */
    for(ri=0; ri<tac->tacNr; ri++) {
      if(isnan(tac->c[ri].y[fi])) n=fprintf(fp, " .");
      else fprintf(fp, " %.*e", prec, tac->c[ri].y[fi]);
      if(n<1) {
        statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_CANNOT_WRITE);
        return TPCERROR_CANNOT_WRITE;
      }
    }
    n=fprintf(fp, "\n"); if(n<1) {
      statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_CANNOT_WRITE);
      return TPCERROR_CANNOT_WRITE;
    }
  }
  
  /* Write extra header, if requested */
  if(extra) {
    if(tac->h.keyNr>0) {
      int ret=iftWrite(&tac->h, fp, status);
      if(ret!=TPCERROR_OK) return ret;
    }
    /* Write units */
    if(tac->cunit!=UNIT_UNKNOWN) fprintf(fp, "# unit := %s\n", unitName(tac->cunit));
    if(tac->tunit!=UNIT_UNKNOWN) fprintf(fp, "# timeunit := %s\n", unitName(tac->tunit));
  }
  
  statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  return(TPCERROR_OK);
}
/*****************************************************************************/

/*****************************************************************************/
/** Read simple format from CSV struct into TAC struct.
    @return enum tpcerror (TPCERROR_OK when successful).
    @author Vesa Oikonen
    @sa tacWriteSimple, tacReadCSV, tacReadMat, tacRead
 */
int tacReadSimple(
  /** Pointer to TAC struct, contents of which are to be written. */
  TAC *tac,
  /** Pointer to CSV from which data is read; if it contains only one column,
      then it is assumed to represent the first y column and x column is not filled. */
  CSV *csv,
  /** Pointer to possible header data, which, if available, is processed and copied to TAC too; 
      enter NULL if not available. */
  IFT *hdr,
  /** Pointer to status data; enter NULL if not needed. */
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  if(tac==NULL) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL);
    return TPCERROR_FAIL;
  }
  tacFree(tac);
  if(csv==NULL || csv->row_nr<1 || csv->col_nr<1) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_DATA);
    return TPCERROR_NO_DATA;
  }  
  if(verbose>0) {printf("%s()\n", __func__); fflush(stdout);}

  /* Check that each row has same number of columns */
  if(verbose>1) {printf("  verifying CSV regularity\n"); fflush(stdout);}
  if(!csvIsRegular(csv)) {
    if(verbose>2) {
      printf("\n <-- CSV contents (irregular) ---\n");
      csvWrite(csv, 0, stdout, NULL);
      printf("\n ------------------------------->\n");
      printf("col_nr := %d\n", csv->col_nr);
      printf("row_nr := %d\n", csv->row_nr);
      if(verbose>3) {
        for(int i=0; i<csv->row_nr; i++)
          printf("columns_on_row[%d] := %d\n", 1+i, csvRowLength(csv, i));
        fflush(stdout);
      }
    }
    /* try to make it regular by removing empty columns from the right */
    if(verbose>1) {printf("  trying to make it regular\n"); fflush(stdout);}
    if(csvTrimRight(csv) || !csvIsRegular(csv)) {
      if(verbose>1) {
        printf("\n <-- CSV contents (still irregular) ---\n");
        csvWrite(csv, 0, stdout, NULL);
        printf("\n ------------------------------->\n");
        printf("col_nr := %d\n", csv->col_nr);
        printf("row_nr := %d\n", csv->row_nr);
        if(verbose>5) {
          for(int i=0; i<csv->row_nr; i++)
            printf("columns_on_row[%d] := %d\n", 1+i, csvRowLength(csv, i));
          fflush(stdout);
        }
      }
      statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_FORMAT);
      return TPCERROR_INVALID_FORMAT;
    }
  }

  /* Check if first line contains fields starting with letter, suggesting title line */
  int withTitle=0;
  for(int i=0; i<csv->col_nr; i++) {
    char *c=csvCell(csv, 0, i);
    if(c!=NULL && isalpha(c[0])) withTitle++;
  }
  if(verbose>1 && withTitle>0) {printf("CSV seems to have title line.\n"); fflush(stdout);}
  if(withTitle>0) withTitle=1;

  int ret, n, m;

  /* Allocate memory for TAC data */
  if(verbose>2) {printf("  allocating memory for TAC data\n"); fflush(stdout);}
  n=csv->col_nr; if(n>1) n--;
  m=csv->row_nr-withTitle;
  ret=tacAllocate(tac, m, n);
  statusSet(status, __func__, __FILE__, __LINE__, ret); if(ret!=TPCERROR_OK) return ret;
  tac->tacNr=n; tac->sampleNr=m;
  tac->isframe=0;
  if(verbose>2) {
    printf("  tacNr := %d\n", tac->tacNr);
    printf("  sampleNr := %d\n", tac->sampleNr);
    fflush(stdout);
  }


  /* Copy title line contents as TAC names */
  if(withTitle) {
    if(verbose>2) {printf("  copying TAC names from the title columns\n"); fflush(stdout);}
    char *c;
    for(int ri=0; ri<tac->tacNr; ri++) {
      c=csvCell(csv, 0, 1+ri);
      if(c!=NULL) strlcpy(tac->c[ri].name, c, MAX_TACNAME_LEN+1);
    }
  }

  /* Copy data from CSV into TAC struct */
  if(verbose>2) {printf("  copying data from CSV to TAC\n"); fflush(stdout);}
  int fi, ri, oknr=0;
  char *c;
  for(fi=0; fi<tac->sampleNr; fi++) {
    /* Get the x value from the first column */
    c=csvCell(csv, fi+withTitle, 0);
    tac->x[fi]=atofVerified(c); if(!isnan(tac->x[fi])) oknr++;
    /* Get the y values from the next columns */
    for(ri=0; ri<tac->tacNr; ri++) {
      c=csvCell(csv, fi+withTitle, 1+ri);
      tac->c[ri].y[fi]=atofVerified(c); if(!isnan(tac->c[ri].y[fi])) oknr++;
    }
  }
  if(oknr<1) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_FORMAT);
    return TPCERROR_INVALID_FORMAT;
  }

  /* Copy header to TAC struct */
  iftDuplicate(hdr, &tac->h);

  /* Try to read units from header */
  tacGetHeaderUnit(tac, NULL);
  tacGetHeaderTimeunit(tac, NULL);

  tac->format=TAC_FORMAT_SIMPLE;
  if(tac->weighting==WEIGHTING_UNKNOWN) tac->weighting=WEIGHTING_OFF;

  statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  return(TPCERROR_OK);
}
/*****************************************************************************/

/*****************************************************************************/
