/** @file tac.c
 *  @brief TAC structure processing.
 */
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include "tpcift.h"
#include "tpcisotope.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <time.h>
#include <string.h>
/*****************************************************************************/
#include "tpctac.h"
/*****************************************************************************/

/*****************************************************************************/
/** Initiate the TAC structure before any use.
    @sa tacFree, tacFreeExceptHeader, tacAllocate, tacDuplicate
    @author Vesa Oikonen
 */
void tacInit(
  /** Pointer to TAC */
  TAC *tac
) {
  if(tac==NULL) return;
  tac->sampleNr=tac->_sampleNr=0;
  tac->tacNr=tac->_tacNr=0;
  tac->format=TAC_FORMAT_UNKNOWN;
  tac->isframe=0;
  tac->x=tac->x1=tac->x2=NULL;
  tac->cunit=tac->tunit=UNIT_UNKNOWN;
  tac->w=NULL;
  tac->weighting=WEIGHTING_UNKNOWN;
  tac->c=NULL;
  tac->_data=NULL;
  iftInit(&tac->h);
}
/*****************************************************************************/

/*****************************************************************************/
/** Initiate the TACC structure before any use.
    @sa tacInit, taccFreeExceptHeader, tacInit
    @author Vesa Oikonen
 */
void taccInit(
  /** Pointer to TACC */
  TACC *tacc
) {
  if(tacc==NULL) return;
  tacc->type=TACTYPE_UNKNOWN;
  tacc->size=0.0;
  tacc->sunit=UNIT_UNKNOWN;
  tacc->y=NULL;
  tacc->sw=tacc->sw2=0;
  tacc->name[0]='\0';
}
/*****************************************************************************/

/*****************************************************************************/
/** Free memory allocated for TACC data. All contents are destroyed.
    @pre Before first use initialize the TACC structure with taccInit().
    @sa tacFree, taccInit, tacInit, tacAllocate
    @author Vesa Oikonen
 */
void taccFree(
  /** Pointer to TACC */
  TACC *tacc
) {
  // TACC does not actually contain mallocated or callocated memory,
  // therefore we can just clear the values
  taccInit(tacc);
}
/*****************************************************************************/

/*****************************************************************************/
/** Free memory allocated for TAC data. All contents are destroyed,
    except the header data in IFT structure.
    @pre Before first use initialize the TAC structure with tacInit().
    @sa tacFree, tacInit, tacAllocate, tacDuplicate
 */
void tacFreeExceptHeader(
  /** Pointer to TAC */
  TAC *tac
) {
  if(tac==NULL) return;
  free(tac->_data);
  // Free allocated TACC contents; actually not necessary, but lets do it
  // for consistency */ 
  for(int i=0; i<tac->_tacNr; i++) taccFree(&tac->c[i]); 
  // Free the TACC list
  free(tac->c);
  // then set everything to zero or NULL again
  tacInit(tac);
}
/*****************************************************************************/

/*****************************************************************************/
/** Free memory allocated for TAC data. All contents are destroyed, including any header data. 
    @pre Before first use initialize the TAC structure with tacInit().
    @sa tacFreeExceptHeader, tacInit, tacAllocate, tacDuplicate, tacDeleteTACC
    @author Vesa Oikonen
 */
void tacFree(
  /** Pointer to TAC */
  TAC *tac
) {
  if(tac==NULL) return;
  free(tac->_data);
  // Free allocated TACC contents; actually not necessary, but lets do it
  // for consistency */ 
  for(int i=0; i<tac->_tacNr; i++) taccFree(&tac->c[i]); 
  // Free the TACC list
  free(tac->c);
  // Free the header data
  iftFree(&tac->h);
  // then set everything to zero or NULL again
  tacInit(tac);
}
/*****************************************************************************/

/*****************************************************************************/
/** Allocate memory for TAC data (and set data pointers inside the structure).
    Any previous contents are deleted. Contents are set to NaN.
    @sa tacAllocateMore, tacDuplicate, tacInit, tacFree
    @return Returns TPCERROR status.
 */
int tacAllocate(
  /** Pointer to initiated TAC structure data; any old contents are deleted.
      tacNr and sampleNr inside the structure are set to or kept at zero. */
  TAC *tac,
  /** Nr of samples to allocate. */
  int sampleNr,
  /** Nr of concentration arrays (regional TACs) to allocate. */
  int tacNr
) {
  if(tac==NULL) return TPCERROR_FAIL;
  /* Delete any previous contents */
  tacFree(tac);
  /* If no memory is requested, then just return */
  if(sampleNr<1 && tacNr<1) return TPCERROR_OK;
  if(sampleNr<1 || tacNr<1) return TPCERROR_FAIL;

  /* Allocate memory for TACC data */
  tac->c=(TACC*)malloc(tacNr*sizeof(TACC));
  if(tac->c==NULL) return TPCERROR_OUT_OF_MEMORY;
  for(int i=0; i<tacNr; i++) taccInit(&tac->c[i]);
  tac->_tacNr=tacNr;

  /* Allocate memory for all double arrays */
  int n; n=sampleNr*(4+tacNr);
  tac->_data=(double*)calloc(n, sizeof(double));
  if(tac->_data==NULL) return TPCERROR_OUT_OF_MEMORY;
  for(int i=0; i<n; i++) tac->_data[i]=nan("");
  tac->_sampleNr=sampleNr;

  /* Set pointers for curve data */
  double *d=tac->_data;
  tac->x=d; d+=sampleNr;
  tac->x1=d; d+=sampleNr;
  tac->x2=d; d+=sampleNr;
  tac->w=d; d+=sampleNr;
  for(int i=0; i<tacNr; i++) {tac->c[i].y=d; d+=sampleNr;}

  return TPCERROR_OK;
}
/*****************************************************************************/

/*****************************************************************************/
/** Allocate memory for more TACs in TAC data (and set data pointers inside the structure).

    Previous contents are not changed. New contents are set to NaN.
    @sa tacAllocate, tacAllocateMoreSamples, tacCopyTacc
    @return Returns TPCERROR status.
 */
int tacAllocateMore(
  /** Pointer to initiated and previously allocated TAC structure data; any old contents are 
      preserved, including tacNr. */
  TAC *tac,
  /** Nr of additional concentration arrays (regional TACs) to allocate; if structure contains 
      unused space for requested TACs already, then nothing is done. */
  int tacNr
) {
  if(tac==NULL) return TPCERROR_FAIL;
  if(tac->_sampleNr<1) return TPCERROR_FAIL;
  /* Check if there is enough space already */
  int newNr, addNr;
  newNr=tac->tacNr+tacNr;
  addNr=newNr-tac->_tacNr;
  //printf("newNr := %d\naddNr := %d\n", newNr, addNr);
  if(addNr<=0) return TPCERROR_OK;

  /* Reallocate memory for TACC data */
  TACC *taccPtr;
  taccPtr=realloc(tac->c, sizeof(TACC)*newNr);
  if(taccPtr==NULL) return TPCERROR_OUT_OF_MEMORY;
  tac->c=taccPtr;
  /* Reallocate memory for double arrays */
  double *dPtr;
  dPtr=realloc(tac->_data, sizeof(double)*(4+newNr)*tac->_sampleNr);
  if(dPtr==NULL) return TPCERROR_OUT_OF_MEMORY;
  tac->_data=dPtr;
  /* If both ok, then update the nr of allocated TACs and initiate new TTACs */ 
  for(int i=tac->_tacNr; i<newNr; i++) taccInit(&tac->c[i]);
  tac->_tacNr=newNr;
  
  /* Reset pointers for curve data */
  double *d=tac->_data;
  tac->x=d; d+=tac->_sampleNr;
  tac->x1=d; d+=tac->_sampleNr;
  tac->x2=d; d+=tac->_sampleNr;
  tac->w=d; d+=tac->_sampleNr;
  for(int i=0; i<tac->_tacNr; i++) {tac->c[i].y=d; d+=tac->_sampleNr;}

  /* Set new y values to NaN */
  for(int i=tac->_tacNr-addNr; i<tac->_tacNr; i++)
    for(int j=0; j<tac->_sampleNr; j++) tac->c[i].y[j]=nan("");

  return TPCERROR_OK;
}
/*****************************************************************************/

/*****************************************************************************/
/** Copy TACC contents (y data and its header) from tacc1 to tacc2.

    These may be inside different TAC structures, therefore sampleNr is needed.
    @sa tacCopyTacchdr, tacCopyTaccdata, tacXCopy, tacWCopy, tacDuplicate, tacCopyHdr,
        tacAllocateMore, tacDeleteTACC, tacSwapTACCs
    @return Returns TPCERROR status.
 */
int tacCopyTacc(
  /** Pointer to source TACC structure. */
  TACC *d1,
  /** Pointer to target TACC structure. */
  TACC *d2,
  /** Sample (frame) number. */
  int sampleNr
) {
  int ret;
  
  /* Check that required data exists */
  if(d1==NULL || d2==NULL) return TPCERROR_FAIL;
  /* Copy TACC header */
  ret=tacCopyTacchdr(d1, d2); if(ret!=TPCERROR_OK) return ret;
  /* Copy TACC data */
  ret=tacCopyTaccdata(d1, d2, sampleNr); if(ret!=TPCERROR_OK) return ret;
  return TPCERROR_OK;
}
/*****************************************************************************/

/*****************************************************************************/
/** Copy TACC y data from tacc1 to tacc2.

    These may be inside different TAC structures, therefore sampleNr is needed.
    @sa tacCopyTacchdr, tacXCopy, tacWCopy 
    @return Returns TPCERROR status.
 */
int tacCopyTaccdata(
  /** Pointer to source TACC structure. */
  TACC *d1,
  /** Pointer to target TACC structure. */
  TACC *d2,
  /** Sample (frame) nr. */
  int sampleNr
) {
  /* Check that required data exists */
  if(d1==NULL || d2==NULL) return TPCERROR_FAIL;
  if(sampleNr<1) return TPCERROR_OK;
  /* Copy data array contents */
  for(int i=0; i<sampleNr; i++) d2->y[i]=d1->y[i];
  return TPCERROR_OK;
}
/*****************************************************************************/

/*****************************************************************************/
/** Copy TACC header data from tacc1 to tacc2.
    @sa tacCopyHdr, tacCopyTaccdata, tacCopyTacc
    @return Returns TPCERROR status.
 */
int tacCopyTacchdr(
  /** Pointer to source TACVOI structure. */
  TACC *d1,
  /** Pointer to target TACVOI structure. */
  TACC *d2
) {
  /* Check that required data exists */
  if(d1==NULL || d2==NULL) return TPCERROR_FAIL;
  /* Copy TACC header fields */
  d2->type=d1->type;
  d2->size=d1->size;
  d2->sunit=d1->sunit;
  d2->sw=d1->sw;
  d2->sw2=d1->sw2;
  strcpy(d2->name, d1->name);
  return TPCERROR_OK;
}
/*****************************************************************************/

/*****************************************************************************/
/** @short Copy TAC header data from tac1 to tac2.
   
    Information regarding data size or weighting is not copied. 
    IFT contents are copied if available. 
    Previous IFT contents are deleted. 
    @sa tacCopyTacchdr, tacIsWeighted, tacWCopy
    @return Returns TPCERROR status.
 */
int tacCopyHdr(
  /** Pointer to source TAC structure. */
  TAC *tac1,
  /** Pointer to target TAC structure. */
  TAC *tac2
) {
  /* Check that required data exists */
  if(tac1==NULL || tac2==NULL) return TPCERROR_FAIL;
  /* Copy TAC header fields */
  tac2->format=tac1->format;
  tac2->isframe=tac1->isframe;
  tac2->cunit=tac1->cunit;
  tac2->tunit=tac1->tunit;
  //tac2->weighting=tac1->weighting;
  /* Copy IFT */
  iftFree(&tac2->h); iftDuplicate(&tac1->h, &tac2->h);
  return TPCERROR_OK;
}
/*****************************************************************************/

/*****************************************************************************/
/** Check if any of TAC sets has VOI size (volume).
   @return 1 if size(s) exist, and 0 if not (or in case of an error).
   @sa tacIsWeighted, tacIsX
 */
int tacIsSize(
  /** Pointer to TAC structure. */
  TAC *d
) {
  if(d==NULL || d->tacNr<1) return(0);
  for(int ri=0; ri<d->tacNr; ri++) {
    if(isnan(d->c[ri].size)) continue;
    if(d->c[ri].size<=0.0) continue;
    return(1);
  }
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** @brief Make a duplicate of TAC structure.
    @details In the duplicate, space is allocated only for the tacNr and sampleNr specified in 
     the original TAC. 
    @sa tacExtract, tacAllocate, tacCopyHdr, tacCopyTacc, tacXCopy, tacDeleteTACC
    @return Returns TPCERROR status (0 when successful).
 */
int tacDuplicate(
  /** Pointer to the source TAC. */
  TAC *tac1,
  /** Pointer to the target TAC; must be initiated; any old contents are deleted. */
  TAC *tac2
) {
  if(tac1==NULL || tac2==NULL) return TPCERROR_FAIL;
  if(tac1->sampleNr<1 || tac1->tacNr<1) return TPCERROR_FAIL;

  /* Empty the duplicate */
  tacFree(tac2);

  int ret;

  /* Allocate memory for tac2 */
  ret=tacAllocate(tac2, tac1->sampleNr, tac1->tacNr);
  if(ret!=TPCERROR_OK) return(ret);
  tac2->sampleNr=tac1->sampleNr;
  tac2->tacNr=tac1->tacNr;

  /* Copy the contents */
  ret=tacCopyHdr(tac1, tac2);
  if(ret!=TPCERROR_OK) return(ret);
  for(int i=0, ret=0; i<tac1->tacNr && ret==0; i++)
    ret=tacCopyTacc(&tac1->c[i], &tac2->c[i], tac1->sampleNr);
  if(ret!=0) return TPCERROR_FAIL;
  ret=tacXCopy(tac1, tac2, 0, tac1->sampleNr-1);
  if(ret!=0) return TPCERROR_FAIL;
  ret=tacWCopy(tac1, tac2, 0, tac1->sampleNr-1);
  if(ret!=0) return TPCERROR_FAIL;
  
  return TPCERROR_OK;
}
/*****************************************************************************/

/*****************************************************************************/
/** @brief Extract the specified TAC from existing TAC structure into a new TAC.
    @sa tacDuplicate, tacDeleteTACC, tacCopyTacc, tacExtractSamples
    @return Returns TPCERROR status (0 if ok).
 */
int tacExtract(
  /** Pointer to the source TAC structure. Not modified. */
  TAC *d1,
  /** Pointer to the target TAC structure; any previous contents are deleted. 
     @sa tacInit, tacFree */
  TAC *d2,
  /** Index of the TAC to extract [0..tacNr-1]. */
  const int i
) {
  if(d1==NULL || d2==NULL) return TPCERROR_FAIL;
  if(d1->sampleNr<1 || d1->tacNr<1) return TPCERROR_NO_DATA;
  if(i<0 || i>=d1->tacNr) return TPCERROR_INVALID_VALUE;

  /* Remove any old contents in target structure */
  tacFree(d2);

  /* Allocate space for the new TAC */
  int ret=tacAllocate(d2, d1->sampleNr, 1); if(ret!=TPCERROR_OK) return(ret);
  d2->sampleNr=d1->sampleNr;
  d2->tacNr=1;

  /* Copy contents */
  ret=tacCopyHdr(d1, d2); if(ret!=TPCERROR_OK) {tacFree(d2); return(ret);}
  ret=tacXCopy(d1, d2, 0, d1->sampleNr-1); if(ret!=TPCERROR_OK) {tacFree(d2); return(ret);}
  ret=tacCopyTacc(d1->c+i, d2->c, d1->sampleNr); if(ret!=TPCERROR_OK) {tacFree(d2); return(ret);}
  ret=tacWCopy(d1, d2, 0, d1->sampleNr-1); if(ret!=TPCERROR_OK) {tacFree(d2); return(ret);}

  return(TPCERROR_OK);
}
/*****************************************************************************/

/*****************************************************************************/
/** @brief Allocate memory for more samples in TAC data.
    @details Previous contents are not changed.
    _sampleNr is increased by nr_to_add, but sampleNr is not changed,
    and the contents of the new last frame(s) are empty (NaN). 
    @sa tacAllocateMore, tacAddZeroSample, tacDuplicate
    @return Returns TPCERROR status (0 when successful).
 */
int tacAllocateMoreSamples(
  /** Pointer to initiated and previously allocated TAC structure data; any old 
      contents are preserved, including tacNr and sampleNr. */
  TAC *tac,
  /** Nr of samples to add; if TAC structure contains unused space for requested 
      samples already, then nothing is done. */
  int addNr
) {
  if(addNr<1) return TPCERROR_OK;
  if(tac==NULL) return TPCERROR_FAIL;
  if(tac->_sampleNr<1 || tac->_tacNr<1) return TPCERROR_FAIL;
  
  /* Check if there is enough space already */
  int newNr;
  newNr=tac->sampleNr+addNr;
  addNr=newNr-tac->_sampleNr;
  if(addNr<=0) return TPCERROR_OK;

  /* Make a temporary copy of the original data */
  int ret;
  TAC temp; tacInit(&temp);
  ret=tacDuplicate(tac, &temp); if(ret!=TPCERROR_OK) return(ret);
  
  /* Delete and reallocate the original TAC */
  tacFree(tac);
  ret=tacAllocate(tac, newNr, temp.tacNr);
  if(ret!=TPCERROR_OK) {tacFree(&temp); return(ret);}
  tac->sampleNr=temp.sampleNr;
  tac->tacNr=temp.tacNr;

  /* Copy the contents */
  ret=tacCopyHdr(&temp, tac);
  if(ret!=TPCERROR_OK) {tacFree(&temp); return(ret);}
  for(int i=0, ret=0; i<temp.tacNr && ret==0; i++)
    ret=tacCopyTacc(&temp.c[i], &tac->c[i], temp.sampleNr);
  if(ret!=0) {tacFree(&temp); return TPCERROR_FAIL;}
  ret=tacXCopy(&temp, tac, 0, temp.sampleNr-1);
  if(ret!=0) {tacFree(&temp); return TPCERROR_FAIL;}

  tacFree(&temp); 
  return TPCERROR_OK;
}
/*****************************************************************************/

/*****************************************************************************/
