/** @file tacio.c
 *  @brief TAC file i/o functions.
 *  @todo Support reading static ROI data from Carimas (test/carimas_static.txt).
 */
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include "tpcift.h"
#include "tpcisotope.h"
#include "tpccsv.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <time.h>
#include <string.h>
/*****************************************************************************/
#include "tpctac.h"
/*****************************************************************************/

/*****************************************************************************/
/** Text representations of TAC file format.
    @sa tpctac.h */
static const char *tac_format[] = {
  "Unknown",            // TAC_FORMAT_UNKNOWN
  "Simple",             // TAC_FORMAT_SIMPLE
  "DFT",                // TAC_FORMAT_DFT
  "IFT",                // TAC_FORMAT_IFT
  "NCI",                // TAC_FORMAT_NCI
  "PMOD",               // TAC_FORMAT_PMOD
  "CSV-INT",            // TAC_FORMAT_CSV_INT
  "CSV-UK",             // TAC_FORMAT_CSV_UK
  "TSV-INT",            // TAC_FORMAT_TSV_INT
  "TSV-UK",             // TAC_FORMAT_TSV_UK
  "CPT",                // TAC_FORMAT_CPT
  "IDWC",               // TAC_FORMAT_IDWC
  "IF",                 // TAC_FORMAT_IF
  "XML",                // TAC_FORMAT_XML
  "HTML",               // TAC_FORMAT_HTML
  "SIF",                // TAC_FORMAT_SIF
  "Xeleris",            // TAC_FORMAT_XELERIS
  "Inveon",             // TAC_FORMAT_INVEON
  "Amide",              // TAC_FORMAT_AMIDE
  "Carimas-txt",        // TAC_FORMAT_CARIMAS_TXT
  "QView",              // TAC_FORMAT_QVIEW
  "Mat",                // TAC_FORMAT_MAT
  "HRRT-HC",            // TAC_FORMAT_HRRT_HC
  "HRPLUS-HC",          // TAC_FORMAT_HRPLUS_HC
  "ABSS-Scanditronics", // TAC_FORMAT_ABSS_SCANDITRONICS
  "ABSS-GEMS",          // TAC_FORMAT_ABSS_GEMS
  "ABSS-ALLOGG-OLD",    // TAC_FORMAT_ABSS_ALLOGG_OLD
  "ABSS-ALLOGG",        // TAC_FORMAT_ABSS_ALLOGG
  "Binary",             // TAC_FORMAT_BINARY
0};
/*****************************************************************************/
/** Default file name extensions of TAC file formats. */
static const char *tac_fn_ext[] = {
  ".txt",    // TAC_FORMAT_UNKNOWN
  ".dat",    // TAC_FORMAT_SIMPLE
  ".dft",    // TAC_FORMAT_DFT
  ".ift",    // TAC_FORMAT_IFT
  ".nci",    // TAC_FORMAT_NCI
  ".tac",    // TAC_FORMAT_PMOD
  ".csv",    // TAC_FORMAT_CSV_INT
  ".csv",    // TAC_FORMAT_CSV_UK
  ".tsv",    // TAC_FORMAT_TSV_INT
  ".tsv",    // TAC_FORMAT_TSV_UK
  ".cpt",    // TAC_FORMAT_CPT
  ".idwc",   // TAC_FORMAT_IDWC
  ".if",     // TAC_FORMAT_IF
  ".xml",    // TAC_FORMAT_XML
  ".html",   // TAC_FORMAT_HTML
  ".sif",    // TAC_FORMAT_SIF
  ".xel",    // TAC_FORMAT_XELERIS
  ".csv"     // TAC_FORMAT_INVEON
  ".tsv",    // TAC_FORMAT_AMIDE
  ".txt",    // TAC_FORMAT_CARIMAS_TXT
  ".csv",    // TAC_FORMAT_QVIEW
  ".mat",    // TAC_FORMAT_MAT
  ".hc",     // TAC_FORMAT_HRRT_HC
  ".r",      // TAC_FORMAT_HRPLUS_HC
  ".lis",    // TAC_FORMAT_ABSS_SCANDITRONICS
  ".bld",    // TAC_FORMAT_ABSS_GEMS
  ".alg",    // TAC_FORMAT_ABSS_ALLOGG_OLD
  ".txt",    // TAC_FORMAT_ABSS_ALLOGG
  ".bin",    // TAC_FORMAT_BINARY
0};
/*****************************************************************************/

/*****************************************************************************/
/** Return pointer to TAC file format description with the format code.
    @return pointer to the TAC file format string.
    @sa tpcformat, tacFormatIdentify, tacDefaultExtension
    @author Vesa Oikonen
 */
char *tacFormattxt(
  /** TAC format code. */
  tacformat c
) {
  if(c<TAC_FORMAT_UNKNOWN || c>=TAC_FORMAT_LAST) return NULL;
  return (char*)tac_format[c];
}
/*****************************************************************************/

/*****************************************************************************/
/** Identify the string representation of the TAC file format.
    @return enum tacformat, or 0 (enum TAC_FORMAT_UNKNOWN) if not identified.
    @author Vesa Oikonen
    @sa tpcformat, tacFormattxt, tacDefaultExtension
 */
int tacFormatIdentify(
  /** TAC format as a string. */
  const char *s
) {
  if(s==NULL || strlen(s)<1) return TAC_FORMAT_UNKNOWN;
  /* Try if string can be found directly in the table */
  for(int i=0; i<TAC_FORMAT_LAST; i++) {
    if(strcasecmp(tac_format[i], s)==0) return i;
  }
  /* Format string is not following TPC standard, lets try something else */
  if(     strcasecmp(s, "dat")==0)              return TAC_FORMAT_TSV_UK;
  else if(strcasecmp(s, "txt")==0)              return TAC_FORMAT_TSV_UK;
  else if(strcasecmp(s, "tac")==0)              return TAC_FORMAT_PMOD;
  else if(strcasecmp(s, "bld")==0)              return TAC_FORMAT_PMOD;
  else if(strcasecmp(s, "htm")==0)              return TAC_FORMAT_HTML;
  else if(strcasecmp(s, "csv")==0)              return TAC_FORMAT_CSV_UK;
  else if(strcasecmp(s, "tsv")==0)              return TAC_FORMAT_TSV_UK;
  else if(strcasecmp(s, "lis")==0)              return TAC_FORMAT_ABSS_SCANDITRONICS;
  else if(strcasecmp(s, "alg")==0)              return TAC_FORMAT_ABSS_ALLOGG_OLD;

  return TAC_FORMAT_UNKNOWN;
}
/*****************************************************************************/

/*****************************************************************************/
/** Return pointer to default TAC filename extension, including the dot,
    based on the TAC format code.
    @return pointer to the filename extension string.
    @sa tpcformat, tacFormattxt, tacFormatWriteSupported
    @author Vesa Oikonen
 */
char *tacDefaultExtension(
  /** TAC format code. */
  tacformat c
) {
  if(c<TAC_FORMAT_UNKNOWN || c>=TAC_FORMAT_LAST) return NULL;
  return (char*)tac_fn_ext[c];
}
/*****************************************************************************/

/*****************************************************************************/
/** Determine the format of TAC file.
    @note Not all formats are identified, and identification does not mean
    that reading the format would be supported.
    @return the format number, or TAC_FORMAT_UNKNOWN if not identified or in case of an error.
    @author Vesa Oikonen
    @sa tacDefaultExtension, tacFormatIdentify, tacFormatWriteSupported
 */
int tacFormatDetermine(
  /** Pointer to the file name; this string is not modified. */
  const char *fname,
  /** Pointer to status data; enter NULL if not needed */
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  if(fname==NULL || strlen(fname)<1) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL);
    return TAC_FORMAT_UNKNOWN;
  }
  if(verbose>1) {printf("%s(%s)\n", __func__, fname); fflush(stdout);}
  int format=TAC_FORMAT_UNKNOWN;

  /* Open file; note that 'b' is required for fgetpos() and fsetpos() to work correctly */
  FILE *fp;
  fp=fopen(fname, "rb");
  if(fp==NULL) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_CANNOT_OPEN);
    return format;
  }

  /* Binary data? */
  int ret;
  size_t fsize=asciiFileSize(fp, &ret);
  if(verbose>3) {printf("  fsize := %zu\n", fsize); fflush(stdout);}
  if(fsize<1) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_DATA);
    fclose(fp); return format;
  }
  if(ret==1) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_UNSUPPORTED);
    fclose(fp); return TAC_FORMAT_BINARY;
  }

  /* Read file to a string */
  char *data;
  data=asciiFileRead(fp, NULL, fsize+1); rewind(fp);
  if(data==NULL) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_DATA);
    fclose(fp); return format;
  }
  /* Check if one of Inveon formats */
  if(strncasecmp(data, "#Subject ID,Subject Weight,Subject Sex,Unique Series ID,Series Date,Series Description", 20)==0 ||
     strncasecmp(data, "#Subject ID;Subject Weight;Subject Sex;Unique Series ID;Series Date;Series Description", 20)==0) {
    format=TAC_FORMAT_INVEON;
  }

  /* Read the first non-comment line */
  int i=0, j;
  char *cptr, *line=NULL;
  cptr=data;
  while((line=strTokenDup(cptr, "\n\r", &j))!=NULL) {
    if(!asciiCommentLine(line, NULL) && strlen(line)>1) break;
    free(line); cptr+=j; i++;
  }
  free(data);
  if(line==NULL) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_FORMAT);
    fclose(fp); return format;
  }

  /* Check for identification strings */
  if(verbose>1) {printf("  checking for magic number\n"); fflush(stdout);}
  if(strncasecmp(line, "DFT", 3)==0) {
    format=TAC_FORMAT_DFT;
  } else if(strncasecmp(line, "cpt", 3)==0) {
    format=TAC_FORMAT_NCI;
  } else if(strcmp(line, "directory,file name,num,slice,mean,sd,cov,max,min,pixel,total,group")==0) {
    format=TAC_FORMAT_QVIEW;
  } else if(strstr(line, " - Time")!=NULL && strstr(line, "(upper bound)")!=NULL &&
            strstr(line, "(lower bound)")!=NULL && strstr(line, "(standard deviation)")!=NULL) {
    format=TAC_FORMAT_INVEON;
  }
  free(line);
  if(format!=TAC_FORMAT_UNKNOWN) {
    if(verbose>0) {printf("  identified as %s file\n", tacFormattxt(format)); fflush(stdout);}
    fclose(fp); return format;
  }

  /* Identify certain file name extensions */
  if(verbose>1) {printf("  checking file name extension\n"); fflush(stdout);}
  cptr=strrchr(fname, '.'); if(cptr!=NULL) cptr++;
  if(verbose>1 && cptr!=NULL) printf("  extension := %s\n", cptr);
  if(strcasecmp(cptr, "IDWC")==0 || strcasecmp(cptr, "IDW")==0) {
    format=TAC_FORMAT_IDWC;
  } else if(strcasecmp(cptr, "IF")==0) {
    format=TAC_FORMAT_IF;
  } else if(strcasecmp(cptr, "XML")==0) {
    format=TAC_FORMAT_XML;
  } else if(strcasecmp(cptr, "SIF")==0) {
    format=TAC_FORMAT_SIF;
  } else if(strcasecmp(cptr, "LIS")==0) {
    format=TAC_FORMAT_ABSS_SCANDITRONICS;
  } else if(strcasecmp(cptr, "ALG")==0) {
    format=TAC_FORMAT_ABSS_ALLOGG_OLD;
  } 
  if(format!=TAC_FORMAT_UNKNOWN) {
    if(verbose>0) {printf("  identified as %s file\n", tacFormattxt(format)); fflush(stdout);}
    fclose(fp); return format;
  }

#if(0)
  /* Try to read as CSV (or TSV) file */
  int ret;
  FILE *fp;
  if(fp==NULL) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_CANNOT_OPEN);
    return TPCERROR_CANNOT_OPEN;
  }
  CSV csv; csvInit(&csv);
  ret=csvRead(&csv, fp, status);
  
#endif


  fclose(fp);
  if(verbose>0) {printf("  format not identified\n"); fflush(stdout);}
  return format; 
}
/*****************************************************************************/

/*****************************************************************************/
/** Check whether writing of specified TAC format is supported.
    @return 1 if supported, 0 if not.
    @sa tacWrite, tacFormatDetermine, tacRead
 */
int tacFormatWriteSupported(
  /** File format code, for example TAC->format. */
  tacformat format
) {
  int ret=0;
  switch(format) {
    case TAC_FORMAT_SIMPLE:
      ret=1;
      break;
    case TAC_FORMAT_DFT:
      ret=1;
      break;
    case TAC_FORMAT_PMOD:
      ret=1;
      break;
    case TAC_FORMAT_CSV_INT:
    case TAC_FORMAT_CSV_UK:
    case TAC_FORMAT_TSV_INT:
    case TAC_FORMAT_TSV_UK:
      ret=1;
      break;
    case TAC_FORMAT_SIF:
      ret=1;
      break;
    case TAC_FORMAT_XML:
      ret=1;
      break;
    default:
      ret=0;
  }
  return(ret);
}
/*****************************************************************************/

/*****************************************************************************/
/** Write TAC data into specified file in specified format.
    @pre Before using this, you may want to call tacFormatWriteSupported().
    @return enum tpcerror (TPCERROR_OK when successful).
    @author Vesa Oikonen
    @sa tacRead, tacFree, tacFormatWriteSupported
 */
int tacWrite(
  /** Pointer to TAC structure, contents of which are to be written. */
  TAC *tac,
  /** Output file pointer. */
  FILE *fp,
  /** File format code; enter TAC_FORMAT_UNKNOWN
      to write data in the format specified inside TAC structure. */  
  tacformat format,
  /** Write (1) or do not write (0) also extra header fields found in IFT;
      not effective with SIF. */
  int extra,
  /** Pointer to status data; enter NULL if not needed. */
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  if(fp==NULL) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_CANNOT_WRITE);
    return TPCERROR_CANNOT_WRITE;
  }
  if(tac==NULL || tac->tacNr<1 || tac->sampleNr<1) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_DATA);
    return TPCERROR_NO_DATA;
  }

  /* Determine and verify the write format */
  if(format<=TAC_FORMAT_UNKNOWN || format>=TAC_FORMAT_LAST)
    format=tac->format;
  if(format<=TAC_FORMAT_UNKNOWN || format>=TAC_FORMAT_LAST) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_FORMAT);
    return TPCERROR_INVALID_FORMAT;
  }
  if(verbose>0) {
    printf("%s():\n", __func__);
    printf("writing %d TACs and %d samples in %s format\n",
      tac->tacNr, tac->sampleNr, tacFormattxt(format));
  }

  /* Write file */
  /* Update tacFormatWriteSupported() if you add/remove formats! */
  int ret;
  switch(format) {
    case TAC_FORMAT_SIMPLE:
      ret=tacWriteSimple(tac, fp, extra, status);
      break;
    case TAC_FORMAT_DFT:
      ret=tacWriteDFT(tac, fp, extra, status);
      break;
    case TAC_FORMAT_PMOD:
      ret=tacWritePMOD(tac, fp, extra, status);
      break;
    case TAC_FORMAT_CSV_INT:
    case TAC_FORMAT_CSV_UK:
    case TAC_FORMAT_TSV_INT:
    case TAC_FORMAT_TSV_UK:
      ret=tacWriteCSV(tac, fp, extra, format, status);
      break;
    case TAC_FORMAT_SIF:
      ret=tacWriteSIF(tac, fp, 0, status);
      break;
    case TAC_FORMAT_XML:
      ret=tacWriteXML(tac, fp, status);
      break;
    default:
      ret=TPCERROR_UNSUPPORTED;
  }
  
  /* Quit */
  statusSet(status, __func__, __FILE__, __LINE__, ret);
  return(ret);
}
/*****************************************************************************/

/*****************************************************************************/
/** Read TAC file contents into TAC data structure.

    This function reads simple data format (time + data columns), DFT format, PMOD format, 
    Hammersmith formats (IDWC and IF), and some CSV formats.
    @author Vesa Oikonen
    @return code tpcerror, TPCERROR_OK (0) when successful.
    @sa tacInit, tacWrite, tacFormatDetermine, csvRead, iftRead
 */
int tacRead(
  /** Pointer to initiated TAC struct where TAC data will be written; any old content is deleted. */
  TAC *d,
  /** Pointer to the filename; this string is not modified. */
  const char *fname,
  /** Pointer to status data; enter NULL if not needed. */
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  if(fname==NULL || strlen(fname)<1 || d==NULL) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL);
    return TPCERROR_FAIL;
  }
  if(verbose>1) printf("%s(%s)\n", __func__, fname);
  int format=TAC_FORMAT_UNKNOWN;

  /* Delete any previous data */
  tacFree(d);

  /* Open the file */
  FILE *fp;
  fp=fopen(fname, "r");
  if(fp==NULL) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_CANNOT_OPEN);
    return TPCERROR_CANNOT_OPEN;
  }
  
  /* Verify that file does not contain binary data */
  {
    int n, binpart;
    n=asciiFileSize(fp, &binpart);
    if(verbose>4) printf("file_size := %d\n", n);
    if(n<2 || binpart!=0) {
      if(verbose>0) {
        printf("file_size := %d\n", n);
        printf("binpart := %d\n", binpart);
        fflush(stdout);
      }
      fclose(fp);
      statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_FORMAT);
      return TPCERROR_INVALID_FORMAT;
    }
  }

  /* Try to read the file as CSV file */
  int ret;
  CSV csv; csvInit(&csv);
  rewind(fp);
  ret=csvRead(&csv, fp, status);
  fclose(fp);
  if(ret!=TPCERROR_OK) {csvFree(&csv); return ret;}
  if(verbose>20) {
    printf("\n --- CSV contents ---\n");
    csvWrite(&csv, 0, stdout, status);
    fflush(stdout);
  }

  /* Try to identify file format from a magic number in the start */
  if(verbose>1) printf("checking for magic number\n");
  if(strncasecmp(csv.c[0].content, "DFT", 3)==0) {
    format=TAC_FORMAT_DFT;
  } else if(strncasecmp(csv.c[0].content, "cpt", 3)==0) {
    format=TAC_FORMAT_NCI;
  } else if(csv.nr>8 && strcasecmp(csv.c[0].content, "time")==0 &&
            strcasecmp(csv.c[1].content, "prompt")==0 &&
            strcasecmp(csv.c[2].content, "delayed")==0 &&
            strcasecmp(csv.c[3].content, "p_rate")==0) {
    format=TAC_FORMAT_HRPLUS_HC;
  } else if(strncasecmp(csv.c[0].content, "Time", 4)==0) {
    format=TAC_FORMAT_PMOD;
  } else if(strncasecmp(csv.c[0].content, "Start", 5)==0) {
    format=TAC_FORMAT_PMOD;
  } else if(strncasecmp(csv.c[0].content, "<?xml version", 6)==0) {
    format=TAC_FORMAT_XML;
  } else if(csv.nr>12 && strcasecmp(csv.c[0].content, "directory")==0 &&
            strcasecmp(csv.c[1].content, "file name")==0 &&
            strcasecmp(csv.c[2].content, "num")==0 &&
            strcasecmp(csv.c[3].content, "slice")==0 &&
            strcasecmp(csv.c[4].content, "mean")==0 &&
            strcasecmp(csv.c[5].content, "sd")==0 &&
            strcasecmp(csv.c[6].content, "cov")==0 &&
            strcasecmp(csv.c[7].content, "max")==0 &&
            strcasecmp(csv.c[8].content, "min")==0 &&
            strcasecmp(csv.c[9].content, "pixel")==0 &&
            strcasecmp(csv.c[10].content, "total")==0 &&
            strcasecmp(csv.c[11].content, "group")==0) {
    format=TAC_FORMAT_QVIEW;
  } else if(csv.nr>5 && strstr(csv.c[0].content, " - Time")!=NULL &&
            strstr(csv.c[2].content, "(upper bound)")!=NULL &&
            strstr(csv.c[3].content, "(lower bound)")!=NULL &&
            strstr(csv.c[4].content, "(standard deviation)")!=NULL) {
    format=TAC_FORMAT_INVEON;
  } else if(strcasecmp(csv.c[0].content, "TAC")==0) {
    if(csv.separator=='\t') format=TAC_FORMAT_TSV_UK;
    else if(csv.separator==',') format=TAC_FORMAT_CSV_INT;
    else format=TAC_FORMAT_CSV_UK;
    // CSV subformat may not be strictly correct yet
  } else if(csv.nr>4 && strcasecmp(csv.c[0].content, "Singles")==0 &&
            strcasecmp(csv.c[1].content, "Randoms")==0 &&
            strcasecmp(csv.c[2].content, "Prompts")==0 &&
            strncasecmp(csv.c[3].content, "Time(ms)", 4)==0) {
    format=TAC_FORMAT_HRRT_HC;
  } else if(csvCell(&csv, 1, 0) && strcasecmp(csvCell(&csv, 1, 0), "Num Voxels")==0 ) 
  {
    format=TAC_FORMAT_CARIMAS_TXT;
  }

  /* If not identified, then try certain file name extensions. */
  /* Extension is not always reliable, therefore magic nr is tried first */
  if(format==TAC_FORMAT_UNKNOWN) {
    if(verbose>1) printf("checking file name extension\n");
    char *cptr, *extension;
    cptr=filenameGetExtension(fname);
    if(cptr!=NULL && strlen(cptr)>1) extension=strdup(cptr+1); 
    else extension=strdup("");
    if(verbose>1) printf("extension := '%s'\n", extension);
    cptr=extension;
    if(strcasecmp(cptr, "IDWC")==0 || strcasecmp(cptr, "IDW")==0) {
      format=TAC_FORMAT_IDWC;
    } else if(strcasecmp(cptr, "IF")==0) {
      format=TAC_FORMAT_IF;
    } else if(strcasecmp(cptr, "XML")==0) {
      format=TAC_FORMAT_XML;
    } else if(strcasecmp(cptr, "HTML")==0 || strcasecmp(cptr, "HTM")==0) {
      format=TAC_FORMAT_HTML;
    } else if(strcasecmp(cptr, "SIF")==0) {
      format=TAC_FORMAT_SIF;
    } else if(strcasecmp(cptr, "HC")==0) {
      format=TAC_FORMAT_HRRT_HC;
    } else if(strcasecmp(cptr, "R")==0) {
      format=TAC_FORMAT_HRPLUS_HC;
    } 
    free(extension);
  }

  /* If still unknown, read all data as IFT */
  IFT hdr; iftInit(&hdr);
  if(format==TAC_FORMAT_UNKNOWN) {
    if(verbose>2) {printf("reading all data in IFT structure\n"); fflush(stdout);}
    fp=fopen(fname, "r");
    if(fp!=NULL) {iftRead(&hdr, fp, 0, 1, status); fclose(fp);}
  }
  if(format==TAC_FORMAT_UNKNOWN) {
    /* Try Allogg ABSS */
    if(iftSearchValue(&hdr, "//Heading", 0)>=0 &&
       iftSearchValue(&hdr, "//Data", 0)>=0 &&
       iftFindPair(&hdr, "System ID", "ABSS09282", 0)>=0)
    { 
      format=TAC_FORMAT_ABSS_ALLOGG;
    /* Try Scanditronics and GEMS ABSS */
    } else if(iftSearchValue(&hdr, "AUX", 0)>=0 && iftSearchValue(&hdr, "1st detector pair", 0)>=0) 
    {
      if(iftSearchValue(&hdr, "Scanditronics", 0)>=0)
        format=TAC_FORMAT_ABSS_SCANDITRONICS;
      else
        format=TAC_FORMAT_ABSS_GEMS;
    } else if(iftFindKey(&hdr, "Discriminators", 0)>=0) { // Try old Allogg
      format=TAC_FORMAT_ABSS_ALLOGG_OLD;
    /* Try another Inveon format */
    } else if(iftSearchValue(&hdr, "Subject Weight", 0)>=0 &&
              iftSearchValue(&hdr, "Subject Sex", 0)>=0) {
      format=TAC_FORMAT_INVEON;
    }
  }

  /* Try to identify as Mat file */
  if(format==TAC_FORMAT_UNKNOWN && csvIsRegular(&csv) && csv.row_nr>1 && csv.col_nr>1) {
    if(verbose>2) printf("\nRegular CSV, therefore trying Mat file format\n");
    /* Starting from second row, each first cell should be inside '' */
    int ok=1; char *c; int len;
    for(int r=1; r<csv.row_nr; r++) {
      c=csvCell(&csv, r, 0); if(c==NULL) {ok=0; break;}
      len=strlen(c); if(len<2) {ok=0; break;} // should be at least ''
      if(c[0]!='\'' || c[len-1]!='\'') {ok=0; break;}
    }
    if(ok) format=TAC_FORMAT_MAT;
  }

  /* more formats should be identified here */
  if(format==TAC_FORMAT_UNKNOWN) {
    if(verbose>10) {
      printf("\n FORMAT NOT IDENTIFIED \n");
      printf("\n --- CSV contents ---\n");
      csvWrite(&csv, 0, stdout, status);
      fflush(stdout);
    }
  }

  /* If not identified, then try to read as simple data */
  if(format==TAC_FORMAT_UNKNOWN) format=TAC_FORMAT_SIMPLE;

  if(verbose>1) {printf("  file format: %s\n", tacFormattxt(format)); fflush(stdout);}


  /* Read the supported formats */
  /* Certain formats may contain additional header information in comment
     lines; read comment headers to be added into TAC struct for further 
     processing; ABSS formats use the previously read whole IFT data.
     Header may not exist, therefore do not worry about errors here.
   */
  switch(format) {
    case TAC_FORMAT_SIMPLE:
      iftFree(&hdr);
      fp=fopen(fname, "r");
      if(fp!=NULL) {iftRead(&hdr, fp, 0, 2, status); fclose(fp);}
      ret=tacReadSimple(d, &csv, &hdr, status);
      break;   
    case TAC_FORMAT_DFT:
      iftFree(&hdr);
      fp=fopen(fname, "r");
      if(fp!=NULL) {iftRead(&hdr, fp, 0, 2, status); fclose(fp);}
      ret=tacReadDFT(d, &csv, &hdr, status);
      break;   
    case TAC_FORMAT_PMOD:
      iftFree(&hdr);
      fp=fopen(fname, "r");
      if(fp!=NULL) {iftRead(&hdr, fp, 0, 2, status); fclose(fp);}
      ret=tacReadPMOD(d, &csv, &hdr, status);
      break;   
    case TAC_FORMAT_QVIEW:
      ret=tacReadQView(d, &csv, 0, status);
      break;   
    case TAC_FORMAT_MAT:
      ret=tacReadMat(d, &csv, status);
      break;   
    case TAC_FORMAT_CSV_UK:
    case TAC_FORMAT_TSV_UK:
    case TAC_FORMAT_CSV_INT:
    case TAC_FORMAT_TSV_INT:
      iftFree(&hdr);
      fp=fopen(fname, "r");
      if(fp!=NULL) {iftRead(&hdr, fp, 0, 2, status); fclose(fp);}
      ret=tacReadCSV(d, &csv, &hdr, status);
      break;   
    case TAC_FORMAT_CARIMAS_TXT:
      ret=tacReadCarimasTxt(d, &csv, status);
      break;   
    case TAC_FORMAT_INVEON:
      ret=tacReadInveonCSV(d, &csv, status);
      break;   
    case TAC_FORMAT_SIF:
      iftFree(&hdr);
      fp=fopen(fname, "r");
      if(fp!=NULL) {iftRead(&hdr, fp, 0, 2, status); fclose(fp);}
      ret=tacReadSIF(d, &csv, &hdr, status);
      break;   
    case TAC_FORMAT_HRRT_HC:
      ret=tacReadHRRTHC(d, &csv, status);
      break;   
    case TAC_FORMAT_HRPLUS_HC:
      ret=tacReadHRPLUSHC(d, &csv, status);
      break;   
    case TAC_FORMAT_ABSS_ALLOGG:
      //iftFree(&hdr);
      //fp=fopen(fname, "r");
      //if(fp!=NULL) {iftRead(&hdr, fp, 0, 1, status); fclose(fp);}
      ret=tacReadAllogg(d, &hdr, status);
      break;   
    case TAC_FORMAT_ABSS_ALLOGG_OLD:
      ret=tacReadOldAllogg(d, &hdr, status);
      break;   
    case TAC_FORMAT_ABSS_SCANDITRONICS:
      ret=tacReadScanditronics(d, &hdr, status);
      break;   
    case TAC_FORMAT_ABSS_GEMS:
      ret=tacReadGEMS(d, &hdr, status);
      break;   
    default:
    /* The rest of formats are not supported */
    if(verbose>0) printf("identified as %s file\n", tacFormattxt(format));
    ret=TPCERROR_UNSUPPORTED;
    statusSet(status, __func__, __FILE__, __LINE__, ret);
  }
  csvFree(&csv); iftFree(&hdr);
  if(ret!=TPCERROR_OK) {tacFree(d); return(ret);}
  
  /* Set study number, if not yet set, based on file name */
  if(verbose>1) printf("checking studynr\n");
  if(tacGetHeaderStudynr(&d->h, NULL, status)!=TPCERROR_OK) {
    if(verbose>2) printf("studynr not found in %s\n", fname);
    char studynr[MAX_STUDYNR_LEN+1];
    /* Only valid studynr is accepted */
    ret=studynrFromFilename(fname, studynr, 1);
    if(ret==0) { 
      if(verbose>2) printf("studynr based on filename: %s\n", studynr);
      ret=tacSetHeaderStudynr(&d->h, studynr);
    } else if(verbose>1)
      fprintf(stderr, "Error: cannot get valid studynr from filename.\n");
  }

  /* Set y unit, if not yet set, based on file name */
  if(verbose>1) printf("checking concentration units\n");
  if(d->cunit==UNIT_UNKNOWN) {
    d->cunit=unitIdentifyFilename(fname);
    if(d->cunit!=UNIT_UNKNOWN && verbose>1) {
      printf("concentration units based on filename := %s\n", unitName(d->cunit)); fflush(stdout);
    }
  }
  
  return ret;
}
/*****************************************************************************/

/*****************************************************************************/
