/** @file tacselect.c
 *  @brief Selecting TAC(s) in TAC struct.
 */
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include "tpcift.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <time.h>
#include <string.h>
/*****************************************************************************/
#include "tpctac.h"
/*****************************************************************************/

/*****************************************************************************/
/** Select the TAC(s) that have matching ID name or number.
    @return the number of newly selected VOIs, or <=0 in case of an error.
    @author Vesa Oikonen
    @sa tacSelectedTACs, tacSelectBestReference, roinameMatch, tacIndividualNames
 */
int tacSelectTACs(
  /** Pointer to TAC structure. */
  TAC *d,
  /** Name or TAC number (as string, starting from 1) which is searched; 
      string can be inside quotation marks. Enter NULL to select all TACs, or
      empty string to select TACs with empty name (which really should not exist). */
  const char *region_name,
  /** 1=Non-matching VOIs are deselected, 0=Old selections are preserved. */
  int reset,
  /** Pointer to status data; enter NULL if not needed. */
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  if(verbose>0) printf("%s()\n", __func__);
  /* Check that required data exists */
  if(d==NULL || d->tacNr<1) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_DATA);
    return 0;
  }

  int ri, len, match_nr=0;

  /* Check if NULL was given as tac name; if, then select all */
  if(region_name==NULL) {
    for(ri=0; ri<d->tacNr; ri++) d->c[ri].sw=1;
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
    return(d->tacNr);
  }

  /* Reset all selections if required */
  if(reset!=0) for(ri=0; ri<d->tacNr; ri++) d->c[ri].sw=0;
  
  /* Remove any quotation marks from name */
  len=strlen(region_name);
  char local_name[len+1];
  if(len>1) len=strncpyClean(local_name, region_name, len+1);
  else strcpy(local_name, region_name);

  /* If tac name is empty, then select all tacs with missing names */
  if(len<1) {
    for(ri=0; ri<d->tacNr; ri++) {
      if(strlen(d->c[ri].name)==0) {d->c[ri].sw=1; match_nr++; continue;}
      /* maybe here should also be checked for names consisting of only space characters */
    }
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
    return(match_nr);
  }

  /* Check each TAC name */
  for(ri=0; ri<d->tacNr; ri++) {
    /* does the name match? */
    if(roinameMatch(d->c[ri].name, local_name, status)!=0) {
      d->c[ri].sw=1; match_nr++; continue;
    }
  }
  /* If at least one match was found, then we are ready */
  if(match_nr>0) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
    return(match_nr);
  }

  /* Next try VOI number, if region_name is suitable */
  int n;
  if(sscanf(local_name, "%d", &n)!=1 || n<1 || n>d->tacNr) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_DATA);
    return(0);
  }
  d->c[n-1].sw=1; match_nr=1;
  statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  return(match_nr);
}
/*****************************************************************************/

/*****************************************************************************/
/** Get the number of selected TACs (sw!=0) in TAC structure.
    @return the number of selected TACs.
    @author Vesa Oikonen
    @sa tacSelectTACs, tacSelectBestReference, roinameMatch, tacIndividualNames
 */
int tacSelectedTACs(
  /** Pointer to TAC data. */
  TAC *d
) {
  int ri, n;

  /* Check the input */
  if(d==NULL || d->tacNr<1) return(0);
  /* Check each VOI */
  for(ri=n=0; ri<d->tacNr; ri++) if(d->c[ri].sw!=0) n++;
  return(n);
}
/*****************************************************************************/

/*****************************************************************************/
/** Get the index of first selected TAC (sw!=0) in TAC structure.
    @return the index of first selected TACs, or <0 if not found.
    @sa tacSelectedTACs, tacSelectTACs, tacSelectBestReference
 */
int tacFirstSelected(
  /** Pointer to TAC data. */
  TAC *d
) {
  if(d==NULL || d->tacNr<1) return(-1);
  for(int i=0; i<d->tacNr; i++) if(d->c[i].sw!=0) return(i);
  return(-1);
}
/*****************************************************************************/

/*****************************************************************************/
/** Select the best guess of reference region in case that several TACs were
    matching with tacSelectTACs().
    @return the index of best region, or <0 in case of an error.
    @author Vesa Oikonen
    @sa tacSelectTACs, tacSelectedTACs, tacEnsureNames, roinameMatch, tacIndividualNames
 */
int tacSelectBestReference(
  /** Pointer to TAC structure, after using tacSelectTACs(). */
  TAC *d
) {
  int ri, len, min_len, i;
  
  if(d==NULL || d->tacNr<1) return -1;
  if(tacSelectedTACs(d)<1) return -2;
  for(ri=0, i=-1, min_len=9999; ri<d->tacNr; ri++) {
    if(d->c[ri].sw==0) continue;
    len=strlen(d->c[ri].name);
    if(strcasestr(d->c[ri].name, "ALL")!=NULL) len-=4;
    if(strcasestr(d->c[ri].name, "AVG")!=NULL) len-=4;
    if(strcasestr(d->c[ri].name, "MEAN")!=NULL) len-=5;
    if(len<min_len) {min_len=len; i=ri;}
  }
  if(i<0) return -2; else return i;
}
/*****************************************************************************/

/*****************************************************************************/
#if(0)
/** Test whether tac name matches with a test string. Test string can contain wild cards. 

    If both names contain separators, the corresponding subnames are tested against each other.
    If test_str does not contain separators (' ', '_', or '-') but TAC name does, 
    then names are considered matching if test_str matches with any of subnames.
  
    @return 1, in case of match, or 0 if not matched.
    @author Vesa Oikonen
    @sa roinameMatch(), tacSelectTACs, tacSelectedTACs, tacEnsureNames, tacIndividualNames
    @deprecated Use roinameMatch() instead.
 */
int tacNameMatch(
  /** Pointer to TAC name to be tested. */
  const char *tacname,
  /** Test string. */
  const char *test_str,
  /** Pointer to status data; enter NULL if not needed. */
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  if(verbose>0) printf("%s()\n", __func__);
  statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  /* If both string are empty, then that is a match */
  if((tacname==NULL || strlen(tacname)<1) && (test_str==NULL || strlen(test_str)<1)) {return 1;} 
  /* If just either of those is empty, then that is a no match */
  if(tacname==NULL || strlen(tacname)<1) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_DATA);
    return 0;
  }
  if(test_str==NULL || strlen(test_str)<1) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_DATA);
    return 0;
  }

  /* Split full names into subnames */
  char tacsub[3][MAX_TACNAME_LEN+1];
  char refsub[3][MAX_TACNAME_LEN+1];
  int  ni, tacsubnr=0, refsubnr=0;
  tacsubnr=tacNameSplit(tacname, tacsub[0], tacsub[1], tacsub[2], MAX_TACNAME_LEN);
  refsubnr=tacNameSplit(test_str, refsub[0], refsub[1], refsub[2], MAX_TACNAME_LEN);
  if(verbose>3) {
    printf("tacsubnr := %d\nsubnames :=", tacsubnr);
    for(ni=0; ni<tacsubnr; ni++) printf(" '%s'", tacsub[ni]); 
    printf("\n");
    printf("refsubnr := %d\nsubnames :=", refsubnr);
    for(ni=0; ni<refsubnr; ni++) printf(" '%s'", refsub[ni]); 
    printf("\n");
  }
  if(tacsubnr==0 && refsubnr==0) return(1);
  if(tacsubnr==0 || refsubnr==0) return(0);
  
  /* If more than one subname to test for, then test against corresponding
     subnames with wildcards */
  if(refsubnr>1) {
    if(tacsubnr!=refsubnr) return(0);
    for(ni=0; ni<refsubnr; ni++) {
      if(strlen(refsub[ni])==0 && strlen(tacsub[ni])==0) continue;
      if( strlen(refsub[ni])<1 || fncasematch(tacsub[ni], refsub[ni])==0 ) {
        return(0);}
    }
    return(1);
  }

  /* We have just one subname in the test string */
  /* If only one subname also in TAC name, then test them with wildcards */
  if(tacsubnr==1) {
    if(fncasematch(tacsub[0], refsub[0])==0 ) return(0); else return(1);
  }
  /* If it is found in any of sub names, then we consider it a match */
  for(ni=0; ni<tacsubnr; ni++) if(fncasematch(tacsub[ni], refsub[0])) return(1);
  return(0);
}
#endif
/*****************************************************************************/

/*****************************************************************************/
