/** @file tacunits.c
 *  @brief TAC data unit conversions.
 */
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include "tpcift.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <time.h>
#include <string.h>
/*****************************************************************************/
#include "tpctac.h"
/*****************************************************************************/

/*****************************************************************************/
/** Convert X values (sample times) in TAC struct to the specified unit.
    @sa tacYUnitConvert, unitConversionFactor
    @return Returns TPCERROR_OK (0) when successful.
 */
int tacXUnitConvert(
  /** Pointer to TAC struct */
  TAC *tac,
  /** Enum unit */
  const int u,
  /** Pointer to status data; enter NULL if not needed */
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  if(verbose>0) {printf("%s(%s)\n", __func__, unitName(u)); fflush(stdout);}
  if(tac==NULL || tac->sampleNr<1) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_DATA);
    return TPCERROR_NO_DATA;
  }

  /* If both units are unknown, then return suitable error code */
  if(u==UNIT_UNKNOWN && tac->tunit==UNIT_UNKNOWN) {
    if(verbose>1) {printf("  unknown x units\n"); fflush(stdout);}
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_UNKNOWN_UNIT);
    return TPCERROR_UNKNOWN_UNIT;
  }

  /* Determine the correction factor */
  double cf;
  cf=unitConversionFactor(tac->tunit, u);
  if(isnan(cf)) {
    if(verbose>1) {printf("  cannot make x conversion factor\n"); fflush(stdout);}
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INCOMPATIBLE_UNIT);
    return TPCERROR_INCOMPATIBLE_UNIT;
  }
  
  /* Multiply x values */
  if(verbose>1) {printf("  converting x with factor %g\n", cf); fflush(stdout);}
  for(int i=0; i<tac->sampleNr; i++) {
    tac->x[i]*=cf; tac->x1[i]*=cf; tac->x2[i]*=cf;
  }
  /* Set new unit */
  tac->tunit=u;
  
  statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  return(TPCERROR_OK);
}
/*****************************************************************************/

/*****************************************************************************/
/** Convert Y values (concentrations) in TAC struct to the specified unit.
    @sa tacXUnitConvert, unitConversionFactor
    @return Returns TPCERROR_OK (0) when successful.
 */
int tacYUnitConvert(
  /** Pointer to TAC struct */
  TAC *tac,
  /** Enum unit */
  const int u,
  /** Pointer to status data; enter NULL if not needed */
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  if(verbose>0) printf("%s(%s)\n", __func__, unitName(u));
  if(tac==NULL || tac->sampleNr<1 || tac->tacNr<1) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_DATA);
    return TPCERROR_NO_DATA;
  }
  
  /* If both units are unknown, then return suitable error code */
  if(u==UNIT_UNKNOWN && tac->cunit==UNIT_UNKNOWN) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_UNKNOWN_UNIT);
    return TPCERROR_UNKNOWN_UNIT;
  }

  /* Determine the correction factor */
  double cf;
  cf=unitConversionFactor(tac->cunit, u);
  if(isnan(cf)) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INCOMPATIBLE_UNIT);
    return TPCERROR_INCOMPATIBLE_UNIT;
  }
  
  /* Multiply y values */
  for(int j=0; j<tac->tacNr; j++) {
    for(int i=0; i<tac->sampleNr; i++) {
      tac->c[j].y[i]*=cf;
    }
  }
  /* Set new unit */
  tac->cunit=u;

  statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  return(TPCERROR_OK);
}
/*****************************************************************************/

/*****************************************************************************/
/** Convert Y values (per volume concentrations) in TAC struct to the corresponding
    per mass concentrations.
    @sa tacYUnitMass2Volume, tacXUnitConvert, tacYUnitConvert, unitConversionFactor
    @return Returns TPCERROR_OK (0) when successful.
 */
int tacYUnitVolume2Mass(
  /** Pointer to TAC struct. */
  TAC *tac,
  /** Density (g/mL). */
  const double density,
  /** Pointer to status data; enter NULL if not needed. */
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  if(verbose>0) {printf("%s(%g)\n", __func__, density); fflush(stdout);}
  if(tac==NULL || tac->sampleNr<1 || tac->tacNr<1 || !(density>0.0)) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_DATA);
    return TPCERROR_NO_DATA;
  }

  /* Check that unit contains volume in divider */
  if(!unitDividerHasVolume(tac->cunit)) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INCOMPATIBLE_UNIT);
    return TPCERROR_INCOMPATIBLE_UNIT;
  }

  /* Try to convert the unit */
  int uu=unitDividerMassVolumeConversion(tac->cunit);
  if(uu==UNIT_UNKNOWN) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INCOMPATIBLE_UNIT);
    return TPCERROR_INCOMPATIBLE_UNIT;
  }

  /* Divide y values with the density */
  for(int j=0; j<tac->tacNr; j++) {
    for(int i=0; i<tac->sampleNr; i++) {
      tac->c[j].y[i]/=density;
    }
  }
  /* Set new unit */
  tac->cunit=uu;

  statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  return(TPCERROR_OK);
}
/*****************************************************************************/

/*****************************************************************************/
/** Convert Y values (per mass concentrations) in TAC struct to the corresponding
    per volume concentrations.
    @sa tacYUnitVolume2Mass, tacXUnitConvert, tacYUnitConvert, unitConversionFactor
    @return Returns TPCERROR_OK (0) when successful.
 */
int tacYUnitMass2Volume(
  /** Pointer to TAC struct. */
  TAC *tac,
  /** Density (g/mL). */
  const double density,
  /** Pointer to status data; enter NULL if not needed. */
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  if(verbose>0) {printf("%s(%g)\n", __func__, density); fflush(stdout);}
  if(tac==NULL || tac->sampleNr<1 || tac->tacNr<1 || !(density>0.0)) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_DATA);
    return TPCERROR_NO_DATA;
  }

  /* Check that unit contains mass in divider */
  if(!unitDividerHasMass(tac->cunit)) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INCOMPATIBLE_UNIT);
    return TPCERROR_INCOMPATIBLE_UNIT;
  }

  /* Try to convert the unit */
  int uu=unitDividerMassVolumeConversion(tac->cunit);
  if(uu==UNIT_UNKNOWN) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INCOMPATIBLE_UNIT);
    return TPCERROR_INCOMPATIBLE_UNIT;
  }

  /* Multiply y values with the density */
  for(int j=0; j<tac->tacNr; j++) {
    for(int i=0; i<tac->sampleNr; i++) {
      tac->c[j].y[i]*=density;
    }
  }
  /* Set new unit */
  tac->cunit=uu;

  statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  return(TPCERROR_OK);
}
/*****************************************************************************/

/*****************************************************************************/
