/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include "tpcextensions.h"
#include "test_tpctac.h"
/*****************************************************************************/

/*****************************************************************************/
int create_tac_for_sort(TAC *tac);
/*****************************************************************************/

/*****************************************************************************/
int test_tacVerifyTimeOrder(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }

  if(verbose>1) printf("tested in test_tacSortByTime()\n");

  statusSet(status, __func__, __FILE__, __LINE__, 0);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
int test_tacSortByTime(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }
  
  int ret;
  TAC tac1; tacInit(&tac1);
  TAC tac2; tacInit(&tac2);

  /* Test with empty data */
  ret=tacSortByTime(&tac1, status);
  if(ret==TPCERROR_OK) return 1;

  /* Check with test data (already in order) */  
  if(verbose>1) printf("creating test data\n");
  ret=create_tac_for_sort(&tac1);
  if(ret!=TPCERROR_OK) {tacFree(&tac1); return 2;}
  ret=tacSortByTime(&tac1, status);
  if(ret!=TPCERROR_OK) {tacFree(&tac1); return 3;}
  ret=tacVerifyTimeOrder(&tac1, NULL);
  if(ret!=0) {tacFree(&tac1); return 4;}
  
  /* Create another test data set for comparison */
  if(verbose>1) printf("creating 2nd test data\n");
  ret=create_tac_for_sort(&tac2);
  if(ret!=TPCERROR_OK) {tacFree(&tac1); tacFree(&tac2); return 5;}
  
  /* Data 1 was already in order, nothing should have been changed */
  if(tacCompareTimes(&tac1, &tac2, 1.0E-20, 1.0E-10, status)!=0 ||
     tacCompareConc(&tac1, &tac2, -1, 1.0E-20, 1.0E-10, status)!=0) {
    tacFree(&tac1); tacFree(&tac2); return 6;
  }

  /* Sorting by Frame times must give an error with missing frame times */
  /* Set some sample times to NaN */
  tac1.x[1]=nan(""); tac1.x1[1]=tac1.x2[0]=nan("");
  ret=tacSortByTime(&tac1, status);
  if(ret!=TPCERROR_MISSING_DATA) {
    tacFree(&tac1); tacFree(&tac2); return 11;
  }
  tac1.x[1]=tac2.x[1]; tac1.x1[1]=tac2.x1[1]; tac1.x2[0]=tac2.x2[0];
  
  /* Change the order of sample times */
  tac1.x[0]=tac2.x[2];   tac1.x[2]=tac2.x[0];
  tac1.x1[0]=tac2.x1[2]; tac1.x1[2]=tac2.x1[0];
  tac1.x2[0]=tac2.x2[2]; tac1.x2[2]=tac2.x2[0];
  for(int ri=0; ri<tac1.tacNr; ri++) {
    tac1.c[ri].y[0]=tac2.c[ri].y[2]; tac1.c[ri].y[2]=tac2.c[ri].y[0];
    tac1.w[0]=tac2.w[2]; tac1.w[2]=tac2.w[0];
  }
  ret=tacVerifyTimeOrder(&tac1, NULL);
  if(ret==0) {tacFree(&tac1); tacFree(&tac2); return 12;}

  /* Sort it back */
  tac1.isframe=1; tac2.isframe=1;
  ret=tacSortByTime(&tac1, status);
  if(ret!=TPCERROR_OK) {tacFree(&tac1); tacFree(&tac2); return 21;}
  if(tacCompareTimes(&tac1, &tac2, 1.0E-20, 1.0E-10, status)!=0 ||
     tacCompareConc(&tac1, &tac2, -1, 1.0E-20, 1.0E-10, status)!=0) {
    tacFree(&tac1); tacFree(&tac2); return 22;
  }

  /* the same again with frame mid times */
  tac1.x[0]=tac2.x[2];   tac1.x[2]=tac2.x[0];
  tac1.x1[0]=tac2.x1[2]; tac1.x1[2]=tac2.x1[0];
  tac1.x2[0]=tac2.x2[2]; tac1.x2[2]=tac2.x2[0];
  for(int ri=0; ri<tac1.tacNr; ri++) {
    tac1.c[ri].y[0]=tac2.c[ri].y[2]; tac1.c[ri].y[2]=tac2.c[ri].y[0];
    tac1.w[0]=tac2.w[2]; tac1.w[2]=tac2.w[0];
  }
  /* Sort it back */
  tac1.isframe=0; tac2.isframe=0;
  ret=tacSortByTime(&tac1, status);
  if(ret!=TPCERROR_OK) {tacFree(&tac1); tacFree(&tac2); return 23;}
  if(tacCompareTimes(&tac1, &tac2, 1.0E-20, 1.0E-10, status)!=0 ||
     tacCompareConc(&tac1, &tac2, -1, 1.0E-20, 1.0E-10, status)!=0) {
    tacFree(&tac1); tacFree(&tac2); return 24;
  }
  

  tacFree(&tac1); tacFree(&tac2);
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
int test_tacSortByConc(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }
  
  int ret, i;
  TAC tac1; tacInit(&tac1);
  TAC tac2; tacInit(&tac2);

  /* Test with empty data */
  ret=tacSortByConc(&tac1, 0, status);
  if(ret==TPCERROR_OK) return 1;

  /* Check with test data (sorted by time) */  
  if(verbose>1) printf("creating test data\n");
  ret=create_tac_for_sort(&tac1);
  if(ret!=TPCERROR_OK) {tacFree(&tac1); return 2;}
  i=0;
  ret=tacSortByConc(&tac1, i, status);
  if(ret!=TPCERROR_OK) {tacFree(&tac1); return 3;}
  /* should not be in time order any more */
  ret=tacVerifyTimeOrder(&tac1, NULL);
  if(ret==0) {tacFree(&tac1); return 4;}

  /* Stupid index should lead to an error */
  ret=tacSortByConc(&tac1, -1, status);
  if(ret==TPCERROR_OK) {tacFree(&tac1); return 5;}
  ret=tacSortByConc(&tac1, tac1._tacNr, status);
  if(ret==TPCERROR_OK) {tacFree(&tac1); return 6;}

  /* Create another test data set for comparison */
  if(verbose>1) printf("creating 2nd test data\n");
  ret=create_tac_for_sort(&tac2);
  if(ret!=TPCERROR_OK) {tacFree(&tac1); tacFree(&tac2); return 10;}

  /* Highest y value should be as the first value, and the lowest value
     as the last */
  i=0;
  double ymin, ymax;
  tacYRange(&tac2, i, &ymin, &ymax, NULL, NULL, NULL, NULL);
  if(!doubleMatch(tac1.c[i].y[0], ymax, 1.0E-20) ||
     !doubleMatch(tac1.c[i].y[tac1.sampleNr-1], ymin, 1.0E-20))
  {
    tacFree(&tac1); tacFree(&tac2); return 11;
  }
  
  /* Resorting data by the time should give the original data */
  ret=tacSortByTime(&tac1, status);
  if(ret!=0 ||
     tacCompareTimes(&tac1, &tac2, 1.0E-20, 1.0E-10, status)!=0 ||
     tacCompareConc(&tac1, &tac2, -1, 1.0E-20, 1.0E-10, status)!=0) 
  {
    tacFree(&tac1); tacFree(&tac2); return 12;
  }

  /* This should work also to other indices than the first one */
  i=tac1.tacNr-1;
  ret=tacSortByConc(&tac1, i, status);
  if(ret!=TPCERROR_OK) {tacFree(&tac1); return 21;}
  tacYRange(&tac2, i, &ymin, &ymax, NULL, NULL, NULL, NULL);
  if(!doubleMatch(tac1.c[i].y[0], ymax, 1.0E-20) ||
     !doubleMatch(tac1.c[i].y[tac1.sampleNr-1], ymin, 1.0E-20))
  {
    tacFree(&tac1); tacFree(&tac2); return 22;
  }
  /* Resorting data by the time should give the original data */
  ret=tacSortByTime(&tac1, status);
  if(ret!=0 ||
     tacCompareTimes(&tac1, &tac2, 1.0E-20, 1.0E-10, status)!=0 ||
     tacCompareConc(&tac1, &tac2, -1, 1.0E-20, 1.0E-10, status)!=0) 
  {
    tacFree(&tac1); tacFree(&tac2); return 23;
  }

  tacFree(&tac1); tacFree(&tac2);
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
int test_tacSortByName(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }

  int ret;
  TAC tac1; tacInit(&tac1);
  TAC tac2; tacInit(&tac2);

  /* Test with empty data */
  ret=tacSortByName(NULL, NULL);
  if(verbose>1) printf("good, NULL data did not crash the program.\n");
  ret=tacSortByName(&tac1, status);
  if(verbose>1) printf("good, empty data did not crash the program.\n");

  /* Make test data */
  if(verbose>1) printf("creating test data\n");
  ret=create_tac_for_sort(&tac1);
  if(ret!=TPCERROR_OK) {tacFree(&tac1); return 2;}
  ret=create_tac_for_sort(&tac2);
  if(ret!=TPCERROR_OK) {tacFree(&tac1); tacFree(&tac2); return 3;}

  /* Check with test data (already in order) */  
  ret=tacSortByName(&tac1, status);
  if(ret!=TPCERROR_OK) {tacFree(&tac1); tacFree(&tac2); return 11;}
  /* Data 1 was already in order, nothing should have been changed */
  if(tacCompareNames(&tac1, &tac2, -1, status)!=0 ||
     tacCompareConc(&tac1, &tac2, -1, 1.0E-20, 1.0E-10, status)!=0) {
    tacFree(&tac1); tacFree(&tac2); return 12;
  }
  
  /* Change the order of 2nd and 3rd VOI, and sort back */
  ret=tacCopyTacc(tac2.c+1, tac1.c+2, tac1.sampleNr);
  if(ret==TPCERROR_OK) ret=tacCopyTacc(tac2.c+2, tac1.c+1, tac1.sampleNr);
  if(ret!=TPCERROR_OK) {tacFree(&tac1); tacFree(&tac2); return 21;}
  /* check first that the order was changed */
  if(tacCompareNames(&tac1, &tac2, -1, status)==0) {
    tacFree(&tac1); tacFree(&tac2); return 22;
  }
  ret=tacSortByName(&tac1, status);
  if(ret!=TPCERROR_OK) {tacFree(&tac1); tacFree(&tac2); return 23;}
  /* After sorting everything should be as it was */
  if(tacCompareNames(&tac1, &tac2, -1, status)!=0 ||
     tacCompareConc(&tac1, &tac2, -1, 1.0E-20, 1.0E-10, status)!=0) {
    tacFree(&tac1); tacFree(&tac2); return 23;
  }
  
  tacFree(&tac1); tacFree(&tac2);
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
int test_tacSortByAUC(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }

  int ret;
  TAC tac1; tacInit(&tac1);
  TAC tac2; tacInit(&tac2);

  /* Test with empty data */
  ret=tacSortByAUC(NULL, NULL);
  if(verbose>1) printf("good, NULL data did not crash the program.\n");
  ret=tacSortByAUC(&tac1, status);
  if(verbose>1) printf("good, empty data did not crash the program.\n");

  /* Make test data */
  if(verbose>1) printf("creating test data\n");
  ret=create_tac_for_sort(&tac1);
  if(ret!=TPCERROR_OK) {tacFree(&tac1); return 2;}
  ret=create_tac_for_sort(&tac2);
  if(ret!=TPCERROR_OK) {tacFree(&tac1); tacFree(&tac2); return 3;}

  /* Check with test data (originally in increasing AUC order) */  
  ret=tacSortByAUC(&tac1, status);
  if(ret!=TPCERROR_OK) {tacFree(&tac1); tacFree(&tac2); return 11;}
  if(strcmp(tac1.c[0].name, tac2.c[2].name)!=0) {
    tacFree(&tac1); tacFree(&tac2); return 12;}
  if(strcmp(tac1.c[1].name, tac2.c[1].name)!=0) {
    tacFree(&tac1); tacFree(&tac2); return 13;}
  if(strcmp(tac1.c[2].name, tac2.c[0].name)!=0) {
    tacFree(&tac1); tacFree(&tac2); return 14;}
  if(!doubleMatch(tac1.c[0].y[1], tac2.c[2].y[1], 1.0E-20)) {
    tacFree(&tac1); tacFree(&tac2); return 15;}
  if(!doubleMatch(tac1.c[1].y[1], tac2.c[1].y[1], 1.0E-20)) {
    tacFree(&tac1); tacFree(&tac2); return 16;}
  if(!doubleMatch(tac1.c[2].y[1], tac2.c[0].y[1], 1.0E-20)) {
    tacFree(&tac1); tacFree(&tac2); return 17;}
  
  tacFree(&tac1); tacFree(&tac2);
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
int test_tacMoveTACC(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }

  int ret;

  if(verbose>1) printf("NULL input.\n");
  ret=tacMoveTACC(NULL, 0, 0);
  if(ret==TPCERROR_OK) return 1;
  
  TAC tac; tacInit(&tac);

  if(verbose>1) printf("Trying empty input.\n");
  ret=tacMoveTACC(&tac, 0, 0);
  if(ret==TPCERROR_OK) return 2;

  if(verbose>1) printf("Create test data.\n");
  ret=create_tac_for_sort(&tac); 
  if(ret!=TPCERROR_OK) {
    printf("  create_tac_for_sort() := %d\n", ret);
    tacFree(&tac); return 3;
  }

  if(verbose>1) printf("Trying invalid index.\n");
  ret=tacMoveTACC(&tac, -1, 0);
  if(ret==TPCERROR_OK) {tacFree(&tac); return 11;}
  ret=tacMoveTACC(&tac, 0, -1);
  if(ret==TPCERROR_OK) {tacFree(&tac); return 12;}
  ret=tacMoveTACC(&tac, 0, tac._tacNr);
  if(ret==TPCERROR_OK) {tacFree(&tac); return 13;}

  if(verbose>1) printf("Creating another test data.\n");
  TAC tac2; tacInit(&tac2);
  ret=create_tac_for_sort(&tac2); 
  if(ret!=TPCERROR_OK) {
    printf("  create_tac_for_sort() := %d\n", ret);
    tacFree(&tac); tacFree(&tac2); return 20;
  }

  if(verbose>1) printf("Moving column 1 to column 2.\n");
  ret=tacMoveTACC(&tac, 0, 1);
  if(ret!=TPCERROR_OK) {tacFree(&tac); tacFree(&tac2); return 21;}
  if(strcmp(tac.c[0].name, tac2.c[1].name)!=0) {
    tacFree(&tac); tacFree(&tac2); return 22;}
  if(strcmp(tac.c[1].name, tac2.c[0].name)!=0) {
    tacFree(&tac); tacFree(&tac2); return 23;}
  if(strcmp(tac.c[2].name, tac2.c[2].name)!=0) {
    tacFree(&tac); tacFree(&tac2); return 24;}
  if(!doubleMatch(tac.c[0].y[1], tac2.c[1].y[1], 1.0E-20)) {
    tacFree(&tac); tacFree(&tac2); return 25;}
  if(!doubleMatch(tac.c[1].y[1], tac2.c[0].y[1], 1.0E-20)) {
    tacFree(&tac); tacFree(&tac2); return 26;}
  if(!doubleMatch(tac.c[2].y[1], tac2.c[2].y[1], 1.0E-20)) {
    tacFree(&tac); tacFree(&tac2); return 27;}

  if(verbose>1) printf("Moving column 3 to column 2.\n");
  ret=tacMoveTACC(&tac, 2, 1);
  if(ret!=TPCERROR_OK) {tacFree(&tac); tacFree(&tac2); return 31;}
  if(strcmp(tac.c[0].name, tac2.c[1].name)!=0) {
    tacFree(&tac); tacFree(&tac2); return 32;}
  if(strcmp(tac.c[1].name, tac2.c[2].name)!=0) {
    tacFree(&tac); tacFree(&tac2); return 33;}
  if(strcmp(tac.c[2].name, tac2.c[0].name)!=0) {
    tacFree(&tac); tacFree(&tac2); return 34;}

  if(verbose>1) printf("Moving column 3 to column 1.\n");
  ret=tacMoveTACC(&tac, 2, 0);
  if(ret!=TPCERROR_OK) {tacFree(&tac); tacFree(&tac2); return 41;}
  if(tacCompareNames(&tac, &tac2, -1, NULL)!=0) {
    tacFree(&tac); tacFree(&tac2); return 42;}
  if(tacCompareConc(&tac, &tac2, -1, 1.0E-20, -1.0, NULL)!=0) {
    tacFree(&tac); tacFree(&tac2); return 43;}


  tacFree(&tac); tacFree(&tac2); 
  
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
int test_tacSwapTACCs(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }

  int ret;

  if(verbose>1) printf("NULL input.\n");
  ret=tacSwapTACCs(NULL, 0, 0);
  if(ret==TPCERROR_OK) return 1;
  
  TAC tac; tacInit(&tac);

  if(verbose>1) printf("Trying empty input.\n");
  ret=tacSwapTACCs(&tac, 0, 0);
  if(ret==TPCERROR_OK) return 2;

  if(verbose>1) printf("Create test data.\n");
  ret=create_tac_for_sort(&tac); 
  if(ret!=TPCERROR_OK) {
    printf("  create_tac_for_sort() := %d\n", ret);
    tacFree(&tac); return 3;
  }

  if(verbose>1) printf("Trying invalid index.\n");
  ret=tacSwapTACCs(&tac, -1, 0);
  if(ret==TPCERROR_OK) {tacFree(&tac); return 11;}
  ret=tacSwapTACCs(&tac, 0, -1);
  if(ret==TPCERROR_OK) {tacFree(&tac); return 12;}
  ret=tacSwapTACCs(&tac, 0, tac._tacNr);
  if(ret==TPCERROR_OK) {tacFree(&tac); return 13;}

  if(verbose>1) printf("Creating another test data.\n");
  TAC tac2; tacInit(&tac2);
  ret=create_tac_for_sort(&tac2); 
  if(ret!=TPCERROR_OK) {
    printf("  create_tac_for_sort() := %d\n", ret);
    tacFree(&tac); tacFree(&tac2); return 20;
  }

  if(verbose>1) printf("Swapping columns 1 and 2.\n");
  ret=tacSwapTACCs(&tac, 0, 1);
  if(ret!=TPCERROR_OK) {tacFree(&tac); tacFree(&tac2); return 21;}
  if(strcmp(tac.c[0].name, tac2.c[1].name)!=0) {
    tacFree(&tac); tacFree(&tac2); return 22;}
  if(strcmp(tac.c[1].name, tac2.c[0].name)!=0) {
    tacFree(&tac); tacFree(&tac2); return 23;}
  if(strcmp(tac.c[2].name, tac2.c[2].name)!=0) {
    tacFree(&tac); tacFree(&tac2); return 24;}
  if(!doubleMatch(tac.c[0].y[1], tac2.c[1].y[1], 1.0E-20)) {
    tacFree(&tac); tacFree(&tac2); return 25;}
  if(!doubleMatch(tac.c[1].y[1], tac2.c[0].y[1], 1.0E-20)) {
    tacFree(&tac); tacFree(&tac2); return 26;}
  if(!doubleMatch(tac.c[2].y[1], tac2.c[2].y[1], 1.0E-20)) {
    tacFree(&tac); tacFree(&tac2); return 27;}

  if(verbose>1) printf("Swapping columns 3 and 2.\n");
  ret=tacSwapTACCs(&tac, 2, 1);
  if(ret!=TPCERROR_OK) {tacFree(&tac); tacFree(&tac2); return 31;}
  if(strcmp(tac.c[0].name, tac2.c[1].name)!=0) {
    tacFree(&tac); tacFree(&tac2); return 32;}
  if(strcmp(tac.c[1].name, tac2.c[2].name)!=0) {
    tacFree(&tac); tacFree(&tac2); return 33;}
  if(strcmp(tac.c[2].name, tac2.c[0].name)!=0) {
    tacFree(&tac); tacFree(&tac2); return 34;}

  if(verbose>1) printf("Swapping columns 3 and 1.\n");
  ret=tacSwapTACCs(&tac, 2, 0);
  if(ret!=TPCERROR_OK) {tacFree(&tac); tacFree(&tac2); return 41;}
  if(strcmp(tac.c[0].name, tac2.c[0].name)!=0) {
    tacFree(&tac); tacFree(&tac2); return 42;}
  if(strcmp(tac.c[1].name, tac2.c[2].name)!=0) {
    tacFree(&tac); tacFree(&tac2); return 43;}
  if(strcmp(tac.c[2].name, tac2.c[1].name)!=0) {
    tacFree(&tac); tacFree(&tac2); return 44;}

  if(verbose>1) printf("Swapping again columns 3 and 2.\n");
  ret=tacSwapTACCs(&tac, 1, 2);
  if(ret!=TPCERROR_OK) {tacFree(&tac); tacFree(&tac2); return 51;}
  if(tacCompareNames(&tac, &tac2, -1, NULL)!=0) {
    tacFree(&tac); tacFree(&tac2); return 52;}
  if(tacCompareConc(&tac, &tac2, -1, 1.0E-20, -1.0, NULL)!=0) {
    tacFree(&tac); tacFree(&tac2); return 53;}


  tacFree(&tac); tacFree(&tac2); 
  
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
int test_tacDeleteTACC(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }

  int ret;

  if(verbose>1) printf("NULL input.\n");
  ret=tacDeleteTACC(NULL, 0);
  if(ret==TPCERROR_OK) return 1;
  
  TAC tac; tacInit(&tac);

  if(verbose>1) printf("Trying empty input.\n");
  ret=tacDeleteTACC(&tac, 0);
  if(ret==TPCERROR_OK) return 2;

  if(verbose>1) printf("Create test data.\n");
  ret=create_tac_for_sort(&tac); 
  if(ret!=TPCERROR_OK) {
    printf("  create_tac_for_sort() := %d\n", ret);
    tacFree(&tac); return 3;
  }

  if(verbose>1) printf("Trying invalid index.\n");
  ret=tacDeleteTACC(&tac, -1);
  if(ret==TPCERROR_OK) {tacFree(&tac); return 11;}
  ret=tacDeleteTACC(&tac, tac.tacNr);
  if(ret==TPCERROR_OK) {tacFree(&tac); return 12;}

  if(verbose>1) printf("Creating another test data.\n");
  TAC tac2; tacInit(&tac2);
  ret=create_tac_for_sort(&tac2); 
  if(ret!=TPCERROR_OK) {
    printf("  create_tac_for_sort() := %d\n", ret);
    tacFree(&tac); tacFree(&tac2); return 20;
  }

  if(verbose>1) printf("Deleting first TAC.\n");
  ret=tacDeleteTACC(&tac, 0);
  if(ret!=TPCERROR_OK) {tacFree(&tac); tacFree(&tac2); return 21;}
  if(tac.tacNr!=tac2.tacNr-1) {tacFree(&tac); tacFree(&tac2); return 22;}
  if(strcmp(tac.c[0].name, tac2.c[1].name)!=0) {
    tacFree(&tac); tacFree(&tac2); return 23;}
  if(strcmp(tac.c[1].name, tac2.c[2].name)!=0) {
    tacFree(&tac); tacFree(&tac2); return 24;}
  if(!doubleMatch(tac.c[0].y[1], tac2.c[1].y[1], 1.0E-20)) {
    tacFree(&tac); tacFree(&tac2); return 25;}
  if(!doubleMatch(tac.c[1].y[1], tac2.c[2].y[1], 1.0E-20)) {
    tacFree(&tac); tacFree(&tac2); return 26;}

  if(verbose>1) printf("Deleting the last TAC.\n");
  ret=tacDeleteTACC(&tac, tac.tacNr-1);
  if(ret!=TPCERROR_OK) {tacFree(&tac); tacFree(&tac2); return 31;}
  if(tac.tacNr!=tac2.tacNr-2) {tacFree(&tac); tacFree(&tac2); return 32;}
  if(strcmp(tac.c[0].name, tac2.c[1].name)!=0) {
    tacFree(&tac); tacFree(&tac2); return 33;}
  if(!doubleMatch(tac.c[0].y[1], tac2.c[1].y[1], 1.0E-20)) {
    tacFree(&tac); tacFree(&tac2); return 34;}

  if(verbose>1) printf("Deleting all TACs.\n");
  while(tac.tacNr>0) ret=tacDeleteTACC(&tac, 0);
  if(ret!=TPCERROR_OK) {tacFree(&tac); tacFree(&tac2); return 41;}


  tacFree(&tac); tacFree(&tac2); 
  
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
int test_tacMultipleSamples(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }

  if(verbose>1) printf("NULL input cannot have multiples\n");
  if(tacMultipleSamples(NULL, 0, NULL, verbose-10)!=0) return(1);
  if(verbose>1) printf("but trying to fix NULL input causes error\n");
  if(tacMultipleSamples(NULL, 1, NULL, verbose-10)==0) return(2);
  
  TAC tac1, tac2; tacInit(&tac1); tacInit(&tac2);

  if(verbose>1) printf("\nEmpty input cannot have multiples\n");
  if(tacMultipleSamples(&tac1, 0, NULL, verbose-10)!=0) return(3);
  if(tacMultipleSamples(&tac1, 1, NULL, verbose-10)==0) return(4);
  if(tacMultipleSamples(&tac1, 1, &tac2, verbose-10)==0) return(5);

  if(verbose>1) printf("\nCreating test data.\n");
  if(create_tac_for_sort(&tac1)!=TPCERROR_OK) {
    printf("  create_tac_for_sort() failed\n");
    tacFree(&tac1); return(100);
  }

  if(verbose>1) printf("\nTest data does not have multiples\n");
  if(tacMultipleSamples(&tac1, 0, NULL, verbose-10)!=0) {tacFree(&tac1); return(101);}
  if(verbose>1) printf("\nFixing it makes a copy of data\n");
  if(tacMultipleSamples(&tac1, 1, &tac2, verbose-10)!=0) 
    {tacFree(&tac1); tacFree(&tac2); return(102);}
  if(tacCompareUnit(&tac1, &tac2, NULL) || tacCompareConc(&tac1, &tac2, -1, 1.0E-10, -1., NULL) ||
     tacCompareTimes(&tac1, &tac2, 1.0E-10, -1., NULL) || tacCompareNames(&tac1, &tac2, -1, NULL))
    {tacFree(&tac1); tacFree(&tac2); return(103);}
  if(verbose>1) printf(" ... and fixing without giving output pointer fails\n");
  if(tacMultipleSamples(&tac1, 1, NULL, verbose-10)==0)
    {tacFree(&tac1); tacFree(&tac2); return(104);}
  if(verbose>1) printf(" ... but giving same TAC pointers works\n");
  if(tacMultipleSamples(&tac2, 1, &tac2, verbose-10)!=0)
    {tacFree(&tac1); tacFree(&tac2); return(105);}
  if(tacCompareUnit(&tac1, &tac2, NULL) || tacCompareConc(&tac1, &tac2, -1, 1.0E-10, -1., NULL) ||
     tacCompareTimes(&tac1, &tac2, 1.0E-10, -1., NULL) || tacCompareNames(&tac1, &tac2, -1, NULL))
    {tacFree(&tac1); tacFree(&tac2); return(106);}

  if(verbose>1) printf("\nTest data with three first samples have same times\n");
  tac1.x1[1]=tac1.x1[2]=tac1.x1[0];
  tac1.x2[0]=tac1.x2[1]=tac1.x2[2];
  tac1.x[0]=tac1.x[1]=tac1.x[2]=0.5*(tac1.x1[0]+tac1.x2[2]);
  tac1.isframe=1; tacFree(&tac2);
  if(tacMultipleSamples(&tac1, 0, NULL, verbose-10)!=1) {tacFree(&tac1); return(111);}
  if(tacMultipleSamples(&tac1, 1, NULL, verbose-10)==0) {tacFree(&tac1); return(112);}
  if(verbose>1) printf(" after fix, we should have two samples less\n");
  if(tacMultipleSamples(&tac1, 1, &tac2, verbose-10)!=0 || tac2.sampleNr!=tac1.sampleNr-2) {
    tacFree(&tac1); tacFree(&tac2); return(113);}
  tac1.isframe=0; tacFree(&tac2);
  if(tacMultipleSamples(&tac1, 1, &tac2, verbose-10)!=0 || tac2.sampleNr!=tac1.sampleNr-2) {
    tacFree(&tac1); tacFree(&tac2); return(114);}
  if(verbose>1) printf(" ... with correct contents\n");
  if(!doubleMatch(tac2.x[0], tac1.x[0], 1.0E-20) || !doubleMatch(tac2.x[1], tac1.x[3], 1.0E-20)) {
    tacFree(&tac1); tacFree(&tac2); return(115);}
  for(int ri=0; ri<tac1.tacNr; ri++) {
    double m=0.0;
    for(int fi=0; fi<3; fi++) m+=tac1.c[ri].y[fi];
    m/=(double)3;
    if(!doubleMatch(tac2.c[ri].y[0], m, 1.0E-20) ||
       !doubleMatch(tac2.c[ri].y[1], tac1.c[ri].y[3], 1.0E-20)) {
      tacFree(&tac1); tacFree(&tac2); return(116);}
  }
  tacFree(&tac2);

  if(verbose>1) printf("\nTest data with all the same times\n");
  {
    for(int fi=1; fi<tac1.sampleNr; fi++) tac1.x1[fi]=tac1.x1[0];
    for(int fi=0; fi<tac1.sampleNr-1; fi++) tac1.x2[fi]=tac1.x2[tac1.sampleNr-1];
    for(int fi=0; fi<tac1.sampleNr; fi++) tac1.x[fi]=0.5*(tac1.x1[0]+tac1.x2[tac1.sampleNr-1]);
  }
  tac1.isframe=0;
  if(tacMultipleSamples(&tac1, 0, NULL, verbose-10)!=1) {tacFree(&tac1); return(121);}
  tac1.isframe=1;
  if(tacMultipleSamples(&tac1, 0, NULL, verbose-10)!=1) {tacFree(&tac1); return(122);}
  if(verbose>1) printf(" after fix, we should have one sample left\n");
  if(tacMultipleSamples(&tac1, 1, &tac2, verbose-10)!=0 || tac2.sampleNr!=1) {
    tacFree(&tac1); tacFree(&tac2); return(123);}
  if(verbose>1) printf(" ... with correct contents\n");
  if(!doubleMatch(tac2.x1[0], tac1.x1[0], 1.0E-20) || 
     !doubleMatch(tac2.x2[0], tac1.x2[tac1.sampleNr-1], 1.0E-20)) {
    tacFree(&tac1); tacFree(&tac2); return(124);}
  for(int ri=0; ri<tac1.tacNr; ri++) {
    double m=0.0;
    for(int fi=0; fi<tac1.sampleNr; fi++) m+=tac1.c[ri].y[fi];
    m/=(double)tac1.sampleNr;
    if(!doubleMatch(tac2.c[ri].y[0], m, 1.0E-20)) {
      tacFree(&tac1); tacFree(&tac2); return(125);}
  }
  tacFree(&tac2);

  tacFree(&tac1); tacFree(&tac2);

  statusSet(status, __func__, __FILE__, __LINE__, 0);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Create test TAC data */
int create_tac_for_sort(
  TAC *tac
) {
  int ret, fi, ri;
  
  /* Allocate memory */
  ret=tacAllocate(tac, 5, 3); if(ret!=TPCERROR_OK) return ret;
  tac->tacNr=3; tac->sampleNr=5;
  /* Set TAC information */
  tac->weighting=WEIGHTING_OFF;
  tac->cunit=UNIT_KBQ_PER_ML;
  tac->tunit=UNIT_MIN;
  tac->isframe=1;
  /* Set region names */
  for(ri=0; ri<tac->tacNr; ri++) sprintf(tac->c[ri].name, "roi%d", 1+ri);
  /* Set volumes */
  for(ri=0; ri<tac->tacNr; ri++) tac->c[ri].size=(double)(100*(ri+1));
  /* Set data contents */
  for(fi=0; fi<tac->sampleNr; fi++) {
    tac->x1[fi]=(double)fi; tac->x2[fi]=(double)(fi+1);
    tac->x[fi]=0.5*(tac->x1[fi]+tac->x2[fi]);
    for(ri=0; ri<tac->tacNr; ri++) {
      tac->c[ri].y[fi]=(double)((ri+1)*(fi+ri)-2);
      //printf("%d %d %g\n", fi, ri, tac->c[ri].y[fi]);
    }
  }
  ret=iftPut(&tac->h, "studynr", "test0001", (char)1, NULL);
  ret=iftPut(&tac->h, "scantime", "2013-11-18 14:02:54", (char)1, NULL);
  return TPCERROR_OK;
}
/*****************************************************************************/

/*****************************************************************************/
