/** @file partac.c
 *  @brief Exchanging information between PAR and TAC structures.
 */
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <time.h>
#include <string.h>
/*****************************************************************************/
#include "tpcextensions.h"
#include "tpcmodels.h"
#include "tpcpar.h"
#include "tpctac.h"
/*****************************************************************************/
#include "tpctacmod.h"
/*****************************************************************************/

/*****************************************************************************/
/** @brief Allocate TAC based on data in PAR.
    @details TAC region number, names, and units are copied from given PAR.
    @sa tacAllocate, tacAllocateMore, tacDuplicate
    @return enum tpcerror (TPCERROR_OK when successful).
    @author Vesa Oikonen
 */ 
int tacAllocateWithPAR(
  /** Pointer to initiated TAC structure; any old contents are deleted.
      TAC sampleNr and tacNr will be set, although contents will be empty.
      @pre Structure must be initiated with tacInit. */
  TAC *tac,
  /** Pointer to PAR structure which contains at least the TAC number and names. */
  PAR *par,
  /** Nr of samples (time frames) to allocate. */
  int sampleNr,
  /** Pointer to status data; enter NULL if not needed. */
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  if(verbose>0) printf("%s(tac, par, %d)\n", __func__, sampleNr);

  /* Check the input */
  if(tac==NULL || par==NULL) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL);
    return TPCERROR_FAIL;
  }
  if(sampleNr<1 || par->tacNr<1) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_DATA);
    return TPCERROR_NO_DATA;
  }

  int ret, i;

  /* Allocate */
  ret=tacAllocate(tac, sampleNr, par->tacNr);
  statusSet(status, __func__, __FILE__, __LINE__, ret);
  if(ret!=TPCERROR_OK) return(ret);
  tac->sampleNr=sampleNr; tac->tacNr=par->tacNr;

  /* Copy TAC names */
  for(int i=0; i<par->tacNr; i++) strcpy(tac->c[i].name, par->r[i].name);

  /* Try to find the data units */
  i=iftFindKey(&par->h, "unit", 0);
  if(i<0) i=iftFindKey(&par->h, "calibration_unit", 0);
  if(i>0) tac->cunit=unitIdentify(par->h.item[i].value);
  i=iftFindKey(&par->h, "timeunit", 0);
  if(i<0) i=iftFindKey(&par->h, "time_unit", 0);
  if(i>0) tac->tunit=unitIdentify(par->h.item[i].value);
  /* Try to find study number */
  i=iftFindKey(&par->h, "studynr", 0);
  if(i<0) i=iftFindKey(&par->h, "study", 0);
  if(i>=0) {
    iftPut(&tac->h, par->h.item[i].key, par->h.item[i].value, 1, NULL);
  }

  statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  return(TPCERROR_OK);
}
/*****************************************************************************/

/*****************************************************************************/
/** @brief Allocate PAR based on data in TAC.
    @details Region number, names, etc are copied from given TAC.
    @sa parAllocate, parAllocateMore, parInit, tacToPar
    @return enum tpcerror (TPCERROR_OK when successful).
    @author Vesa Oikonen
 */ 
int parAllocateWithTAC(
  /** Pointer to initiated PAR struct; any old contents are deleted.
      PAR tacNr and parNr will be set, although contents will be empty.
      @pre Struct must be initiated with parInit. */
  PAR *par,
  /** Pointer to TAC struct which contains at least the TAC number and names. */
  TAC *tac,
  /** Nr of parameters to allocate. */
  int parNr,
  /** Pointer to status data; enter NULL if not needed. */
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  if(verbose>0) printf("%s(par, tac, %d)\n", __func__, parNr);

  /* Check the input */
  if(tac==NULL || par==NULL) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL);
    return TPCERROR_FAIL;
  }
  if(parNr<1 || tac->tacNr<1) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_DATA);
    return TPCERROR_NO_DATA;
  }

  int i, ret;

  /* Allocate memory */
  if(verbose>2) printf("allocate memory for PAR\n");
  ret=parAllocate(par, parNr, tac->tacNr);
  if(ret!=TPCERROR_OK) {statusSet(status, __func__, __FILE__, __LINE__, ret); return ret;}
  par->tacNr=tac->tacNr;
  par->parNr=parNr;

  /* Copy TAC names */
  if(verbose>3) printf("copy TAC names\n");
  for(int i=0; i<par->tacNr; i++) strcpy(par->r[i].name, tac->c[i].name);

  /* Copy header information */
  if(verbose>3) printf("copy header\n");

  i=iftFindKey(&tac->h, "studynr", 0);
  if(i<0) i=iftFindKey(&tac->h, "study_number", 0);
  if(i>=0) iftPut(&par->h, tac->h.item[i].key, tac->h.item[i].value, 0, NULL);

  i=iftFindKey(&tac->h, "injection_time", 0);
  if(i<0) i=iftFindKey(&tac->h, "injection time", 0);
  if(i>=0) iftPut(&par->h, tac->h.item[i].key, tac->h.item[i].value, 0, NULL);

  i=iftFindKey(&tac->h, "isotope", 0);
  if(i>=0) iftPut(&par->h, tac->h.item[i].key, tac->h.item[i].value, 0, NULL);

  i=iftFindKey(&tac->h, "radiopharmaceutical", 0);
  if(i>=0) iftPut(&par->h, tac->h.item[i].key, tac->h.item[i].value, 0, NULL);

  i=iftFindKey(&tac->h, "scan_start_time", 0);
  if(i>=0) iftPut(&par->h, tac->h.item[i].key, tac->h.item[i].value, 0, NULL);

  if(tacIsWeighted(tac)) iftPut(&par->h, "weighting", "yes", 0, NULL);
  else iftPut(&par->h, "weighting", "no", 0, NULL);

  if(tac->tunit!=UNIT_UNKNOWN)
    iftPut(&par->h, "timeunit", unitName(tac->tunit), 0, NULL);
  if(tac->cunit!=UNIT_UNKNOWN)
    iftPut(&par->h, "unit", unitName(tac->cunit), 0, NULL);

  statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  return(TPCERROR_OK);
}
/*****************************************************************************/

/*****************************************************************************/
/** @brief Copy the contents of TAC struct into PAR struct.
    @details In addition to data, the header contents such as region names and
     units are copied if possible.
    @sa parInit, parFree, parAllocateWithTAC, parWrite
    @return enum tpcerror (TPCERROR_OK when successful).
    @author Vesa Oikonen
 */ 
int tacToPAR(
  /** Pointer to the TAC struct to be copied. */
  TAC *tac,
  /** Pointer to PAR struct; any old contents are deleted.
      @pre Struct must be initiated with parInit. */
  PAR *par,
  /** Pointer to status data; enter NULL if not needed */
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  if(verbose>0) printf("%s(par, tac)\n", __func__);

  int ret;

  /* Check the input */
  if(tac==NULL || par==NULL) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL);
    return TPCERROR_FAIL;
  }
  if(tac->sampleNr<1 || tac->tacNr<1) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_DATA);
    return TPCERROR_NO_DATA;
  }

  /* Allocate memory and copy most of headers */
  ret=parAllocateWithTAC(par, tac, tac->sampleNr, status);
  if(ret!=TPCERROR_OK) return(ret);

  /* Copy the values */
  for(int i=0; i<tac->tacNr; i++)
    for(int j=0; j<par->parNr; j++)
      par->r[i].p[j]=tac->c[i].y[j];

  statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  return TPCERROR_OK;
}
/*****************************************************************************/

/*****************************************************************************/
