/** @file paradd.c
 *  @brief Insert parameters from one PAR file into another.
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
/*****************************************************************************/
#include "tpcextensions.h"
#include "tpcift.h"
#include "tpcpar.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Insert parameter values from parameter file 2 into another, optionally new,",
  "parameter file 1. If file1 does not exist, it will be created.",
  " ",
  "Usage: @P [options] file1 file2",
  " ",
  "Options:",
  " -par=<list>",
  "     List of parameter numbers or names to be copied; all by default.",
  " -tac=<list>",
  "     List of TAC numbers or names to be copied; all by default.",
  " -ovr",
  "     Existing file1 is overwritten.",
  " --dry",
  "     Nothing is actually copied, application only lists what it would do.",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "Example 1:",
  "Create a new parameter file which contains only parameters 1-3",
  "     @P -ovr -par=1-3 ab123new.res ab123fitk3.res",  
  " ",
  "Example 2:",
  "Create a new parameter file which contains only regions 'caud' and 'putam'",
  "     @P -ovr -tac=caud,putam ab234new.csv ab234srtm.csv",
  " ",
  "Example 3:",
  "Combine parameter BPnd in region 'putam' from given studies into one file",
  "in MS Windows command prompt window, and then convert it to XML format",
  "supported by MS Excel",
  "     del combined.res",
  "     for %g in (s6789*.res) do @P -par=BPnd -tac=putam combined.res %g",  
  "     parformat -f=xml combined.res",
  " ",
  "See also: parformat, parget, parsort, pardiff, parmean, tacadd, rescoll",
  " ",
  "Keywords: parameter, tool, reporting, software testing",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int ai, help=0, version=0, verbose=1;
  int ret, n;
  int drymode=0; // 1=no files are actually edited or created
  int newfile=0; // 0=TAC(s) are added; 1=new file is created
  int overwrite=0; // 0=TAC(s) are added; 1=existing file is overwritten
  char *cptr, parfile1[FILENAME_MAX], parfile2[FILENAME_MAX];
  char *pars=NULL, *tacs=NULL; // remember to free allocated memory before exit
  PAR par1, par2;
  FILE *fp;

  
  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  parInit(&par1); parInit(&par2);
  parfile1[0]=parfile2[0]=(char)0;
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') {
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(!*cptr) continue;
    if(strncasecmp(cptr, "OVR", 2)==0) {
      overwrite=1; continue;
    } else if(strncasecmp(cptr, "DRY", 2)==0) {
      drymode=1; continue;
    } else if(strncasecmp(cptr, "PAR=", 4)==0 && strnlen(cptr, 5)>4) {
      pars=strdup(cptr+4); continue;
    } else if(strncasecmp(cptr, "TAC=", 4)==0 && strnlen(cptr, 5)>4) {
      tacs=strdup(cptr+4); continue;
    }
    fprintf(stderr, "Error: invalid option '%s'\n", argv[ai]);
    return(1);
  } else break; // tac name argument may start with '-'

  TPCSTATUS status; statusInit(&status);
  statusSet(&status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  status.verbose=verbose-1;
  
  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  /* Process other arguments, starting from the first non-option */
  for(; ai<argc; ai++) {
    if(!parfile1[0]) {
      strlcpy(parfile1, argv[ai], FILENAME_MAX); continue;
    } else if(!parfile2[0]) {
      strlcpy(parfile2, argv[ai], FILENAME_MAX); continue;
    }
    fprintf(stderr, "Error: too many arguments: '%s'.\n", argv[ai]);
    return(1);
  }

  /* Check that we got what we need */
  if(!parfile2[0]) {
    tpcPrintUsage(argv[0], info, stderr); 
    free(pars); free(tacs); return(1);
  }
  
  /* In verbose mode print arguments and options */
  if(verbose>1) {
    printf("parfile1 := %s\n", parfile1);
    printf("parfile2 := %s\n", parfile2);
    printf("overwrite := %d\n", overwrite);
    printf("drymode := %d\n", drymode);
    if(pars!=NULL) printf("pars := '%s'\n", pars);
    if(tacs!=NULL) printf("tacs := '%s'\n", tacs);
  }

  /* 
   *  Try to read the file in which data should be added
   */
  if(verbose>1) printf("trying to read %s\n", parfile1);
  ret=parRead(&par1, parfile1, &status);
  if(ret==TPCERROR_OK) {
    /* File 1 could be read */
    if(overwrite!=0) {
      if(drymode) printf("File %s would be overwritten.\n", parfile1);
      else if(verbose>=0) printf("File %s is overwritten.\n", parfile1);
      newfile=1;
    } else if(verbose>1) {
      printf("fileformat1 := %s\n", parFormattxt(par1.format));
      printf("parNr1 := %d\n", par1.tacNr);
      printf("tacNr1 := %d\n", par1.tacNr);
    }
    parFree(&par1);
  } else if(ret==TPCERROR_CANNOT_OPEN) {
    /* File 1 does not exist */
    if(drymode) printf("File %s would be created.\n", parfile1);
    else if(verbose>=0) printf("File %s did not exist and was created.\n", parfile1);
    newfile=1; parFree(&par1);
  } else {
    /* File 1 exists but cannot be read */
    fprintf(stderr, "Error: %s\n", errorMsg(status.error));
    parFree(&par1); free(pars); free(tacs); return(2);
  }


  /* 
   *  Read the file from where parameters are imported
   */
  if(verbose>1) printf("reading %s\n", parfile2);
  ret=parRead(&par2, parfile2, &status);
  if(ret!=TPCERROR_OK) {
    fprintf(stderr, "Error: %s\n", errorMsg(status.error));
    parFree(&par2); free(pars); free(tacs); return(3);
  }
  if(verbose>1) {
    printf("fileformat2 := %s\n", parFormattxt(par2.format));
    printf("parNr2 := %d\n", par2.parNr);
    printf("tacNr2 := %d\n", par2.tacNr);
  }
  if(verbose>6) {
    printf("file contents:\n");
    parWrite(&par2, stdout, PAR_FORMAT_CSV_UK, 1, NULL);
    printf("----------------\n");
  }

  /* 
   *  Select the data in file 2 to be copied 
   */
  int parNr=0, tacNr=0;
  if(pars==NULL) {
    /* No parameters specified: select all */
    parSelectParameters(&par2, NULL, 0, &status);
  } else {
    char buf[512];
    for(int i=0; i<strTokenNr(pars, ",;"); i++) {
      if(strTokenNCpy(pars, ",;", i+1, buf, 512)<1) continue;
      n=parSelectParameters(&par2, buf, 0, &status);
      if(n==0) fprintf(stderr, "Warning: %d parameter(s) match '%s'\n", n, buf);
      else if(verbose>2) printf("%d parameter(s) match '%s'\n", n, buf);
    }
  }
  parNr=parSelectedParameters(&par2);
  if(parNr==0) {
    fprintf(stderr, "Error: no parameter(s) to add.\n");
    parFree(&par2); free(pars); free(tacs); return(4);
  }
  if(verbose>2 || drymode) {
    for(int i=0; i<par2.parNr; i++) 
      if(par2.n[i].sw) printf("selected_parameter: %s\n", par2.n[i].name);
  }
  if(tacs==NULL) {
    /* No TACs specified: select all */
    parSelectTACs(&par2, NULL, 0, &status);
  } else {
    char buf[512];
    for(int i=0; i<strTokenNr(tacs, ",;"); i++) {
      if(strTokenNCpy(tacs, ",;", i+1, buf, 512)<1) continue;
      n=parSelectTACs(&par2, buf, 0, &status);
      if(n==0) fprintf(stderr, "Warning: %d TAC(s) match '%s'\n", n, buf);
      else if(verbose>2) printf("%d TAC(s) match '%s'\n", n, buf);
    }
  }
  tacNr=parSelectedTACs(&par2);
  if(tacNr==0) {
    fprintf(stderr, "Error: no TAC(s) to add.\n");
    parFree(&par2); free(pars); free(tacs); return(5);
  }
  if(verbose>2 || drymode) {
    for(int i=0; i<par2.tacNr; i++) 
      if(par2.r[i].sw) printf("selected_tac: %s\n", par2.r[i].name);
  }
  /* par and tac strings are not needed later */
  free(pars); free(tacs);

  /*
   *  Delete parameters and TACs that were not selected.
   */
  if(parNr!=par2.parNr) {
    int i=par2.parNr-1;
    while(i>=0) {
      if(par2.n[i].sw==0) parDeletePar(&par2, i);
      i--;
    }
  }
  if(tacNr!=par2.tacNr) {
    int i=par2.tacNr-1;
    while(i>=0) {
      if(par2.r[i].sw==0) parDeleteTAC(&par2, i);
      i--;
    }
  }

  /*
   *  If target file does not exist, then save the selected contents in file, and quit.
   */
  if(newfile) {

    if(drymode) {
      printf("%d parameter(s) and %d TAC(s) would be written in %s\n", par2.parNr, par2.tacNr, parfile1);
      parFree(&par2);
      return(0);
    }

    /* Write new parameter file */
    if(verbose>1) printf("  saving\n");
    fp=fopen(parfile1, "w");
    if(fp==NULL) {
      fprintf(stderr, "Error: cannot open file for writing.\n");
      parFree(&par2); return(11);
    }
    ret=parWrite(&par2, fp, PAR_FORMAT_UNKNOWN, 1, &status);
    fclose(fp); parFree(&par2);
    if(ret!=TPCERROR_OK) {
      fprintf(stderr, "Error: %s\n", errorMsg(status.error));
      return(12);
    }
    if(verbose>0) printf("%s saved.\n", parfile1);
    return(0);

  }


  /*
   *  We are here only if target file exists. 
   */
  /* Read the previous contents. */
  if(verbose>1) printf("reading %s\n", parfile1);
  ret=parRead(&par1, parfile1, &status);
  if(ret!=TPCERROR_OK) {
    fprintf(stderr, "Error: %s\n", errorMsg(status.error));
    parFree(&par1); parFree(&par2); return(6);
  }
  if(verbose>1) {
    printf("fileformat1 := %s\n", parFormattxt(par1.format));
    printf("parNr1 := %d\n", par1.parNr);
    printf("tacNr1 := %d\n", par1.tacNr);
  }

  /* Check whether parameters and/or TACs match between the files */
  int sametacs=0, samepars=0;
  if(parCompareParameterNames(&par1, &par2, -1, 0, &status)==0) samepars=1;
  if(parCompareTacNames(&par1, &par2, -1, 0, &status)==0) sametacs=1;
  if(verbose>1) {
    if(samepars) printf("All parameters match.\n"); else printf("Parameters do not match.\n"); 
    if(sametacs) printf("All TACs match.\n"); else printf("TACs do not match.\n"); 
  }


  /*
   *  If new and old TACs do match, then add new parameter column(s)
   */
  if(sametacs) {

    if(drymode) {
      printf("%d parameter(s) would be written in %s\n", parNr, parfile1);
      parFree(&par1); parFree(&par2);
      return(0);
    }

    if(verbose>1) printf("adding %d parameter(s) in %s\n", parNr, parfile1);

    /* Allocate more space */
    ret=parAllocateMore(&par1, par2.parNr, 0);
    if(ret!=TPCERROR_OK) {
      fprintf(stderr, "Error: cannot add parameters to data.\n");
      parFree(&par1); parFree(&par2); return(7);
    }

    /* Copy */
    int ti, pi, pj=par1.parNr;
    for(pi=0; pi<par2.parNr; pi++, pj++) {
      strcpy(par1.n[pj].name, par2.n[pi].name);
      par1.n[pj].unit=par2.n[pi].unit;
      for(ti=0; ti<par1.tacNr; ti++) {
        par1.r[ti].p[pj]=par2.r[ti].p[pi];
        par1.r[ti].sd[pj]=par2.r[ti].sd[pi];
        par1.r[ti].cl1[pj]=par2.r[ti].cl1[pi];
        par1.r[ti].cl2[pj]=par2.r[ti].cl2[pi];
      }
    }
    par1.parNr+=parNr;
    
    /* Remove WSS data */
    for(ti=0; ti<par1.tacNr; ti++) par1.r[ti].wss=nan("");
    

    /* Save file */
    if(verbose>1) printf("  saving\n");
    fp=fopen(parfile1, "w");
    if(fp==NULL) {
      fprintf(stderr, "Error: cannot open file for writing.\n");
      parFree(&par1); parFree(&par2); return(11);
    }
    ret=parWrite(&par1, fp, PAR_FORMAT_UNKNOWN, 1, &status);
    fclose(fp); parFree(&par1); parFree(&par2);
    if(ret!=TPCERROR_OK) {
      fprintf(stderr, "Error: %s\n", errorMsg(status.error));
      return(12);
    }
    if(verbose>0) printf("%d parameter(s) saved in %s\n", parNr, parfile1);
    return(0);
  }

  /*
   *  If new and old parameters do match but TACs did not, then add new TAC row(s)
   */
  if(samepars) {

    if(drymode) {
      printf("%d TAC(s) would be written in %s\n", tacNr, parfile1);
      parFree(&par1); parFree(&par2);
      return(0);
    }

    if(verbose>1) printf("adding %d TAC(s) in %s\n", tacNr, parfile1);

    /* Allocate more space */
    ret=parAllocateMore(&par1, 0, par2.tacNr);
    if(ret!=TPCERROR_OK) {
      fprintf(stderr, "Error: cannot add TACs to data.\n");
      parFree(&par1); parFree(&par2); return(7);
    }

    /* Copy */
    if(verbose>2) printf("copying...\n");
    int pi, ti, tj=par1.tacNr;
    for(ti=0; ti<par2.tacNr; ti++, tj++) {
      par1.r[tj].model=par2.r[ti].model;
      strcpy(par1.r[tj].name, par2.r[ti].name);
      par1.r[tj].start=par2.r[ti].start;
      par1.r[tj].end=par2.r[ti].end;
      par1.r[tj].fitNr=par2.r[ti].fitNr;
      par1.r[tj].dataNr=par2.r[ti].dataNr;
      par1.r[tj].wss=par2.r[ti].wss;
      for(pi=0; pi<par2.parNr; pi++) {
        par1.r[tj].p[pi]=par2.r[ti].p[pi];
        par1.r[tj].sd[pi]=par2.r[ti].sd[pi];
        par1.r[tj].cl1[pi]=par2.r[ti].cl1[pi];
        par1.r[tj].cl2[pi]=par2.r[ti].cl2[pi];
      }
    }
    par1.tacNr+=tacNr;

    /* Save file */
    if(verbose>1) printf("  saving\n");
    fp=fopen(parfile1, "w");
    if(fp==NULL) {
      fprintf(stderr, "Error: cannot open file for writing.\n");
      parFree(&par1); parFree(&par2); return(11);
    }
    ret=parWrite(&par1, fp, PAR_FORMAT_UNKNOWN, 1, &status);
    fclose(fp); parFree(&par1); parFree(&par2);
    if(ret!=TPCERROR_OK) {
      fprintf(stderr, "Error: %s\n", errorMsg(status.error));
      return(12);
    }
    if(verbose>0) printf("%d TAC(s) saved in %s\n", tacNr, parfile1);
    return(0);
  }


  /*
   *  Check if there are common TAC names
   */
  ret=parSelectByAnother(&par2, &par1, &parNr, &tacNr, &status);
  if(verbose>1) printf("%d common TAC names\n%d common parameter names\n", tacNr, parNr);
  if(ret!=TPCERROR_OK || tacNr<1) {
    fprintf(stderr, "Error: incompatible contents for adding.\n");
    parFree(&par1); parFree(&par2);
    return(1);
  }
  /* Copy */
  parNr=par1.parNr;
  ret=parCombineTACs(&par1, &par2, &status);
  if(ret!=TPCERROR_OK) {
    fprintf(stderr, "Error: incompatible contents for adding.\n");
    parFree(&par1); parFree(&par2);
    return(1);
  }
  //printf("parNr %d -> %d\n", parNr, par1.parNr);
  /* Save */
  if(drymode) {
    printf("%d new parameter(s) would be written in %s\n", par1.parNr-parNr, parfile1);
    parFree(&par1); parFree(&par2);
    return(0);
  }
  {
    if(verbose>1) printf("  saving\n");
    fp=fopen(parfile1, "w");
    if(fp==NULL) {
      fprintf(stderr, "Error: cannot open file for writing.\n");
      parFree(&par1); parFree(&par2); return(11);
    }
    ret=parWrite(&par1, fp, PAR_FORMAT_UNKNOWN, 1, &status);
    fclose(fp);
    if(ret!=TPCERROR_OK) {
      fprintf(stderr, "Error: %s\n", errorMsg(status.error));
      parFree(&par1); parFree(&par2); return(12);
    }
    printf("%d new parameter(s) saved in %s\n", par1.parNr-parNr, parfile1);
    parFree(&par1); parFree(&par2);
    return(0);
  }

  /* Free memory */
  parFree(&par1); parFree(&par2);

  fprintf(stderr, "Error: incompatible contents for adding.\n");
  return(1);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
