/** @file parget.c
 *  @brief Print to stdout a single parameter value from a PAR file.
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
/*****************************************************************************/
#include "tpcextensions.h"
#include "tpcift.h"
#include "tpcpar.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Print to screen the parameter value for specified parameter and TAC,",
  "based on the name or number of the parameter and TAC in the file.",
  " ",
  "Usage: @P [options] parfile parametername tacname",
  " ",
  "Options:",
  " -SD | -CL1 | -CL2",
  "     Parameter SD or CL value is printed instead of parameter itself.",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "Example:",
  "     @P ab123fitk3.res k3 putam",
  " ",
  "See also: paradd, parformat, parrenp, parsort, pardiff, rescoll",
  " ",
  "Keywords: parameter, tool, reporting, software testing",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int ai, help=0, version=0, verbose=1;
  char parfile[FILENAME_MAX], parname[FILENAME_MAX], tacname[FILENAME_MAX];
  int printWhat=0; // 0=parameter, 1=SD, 2=CL1, 3=CL2

  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  parfile[0]=parname[0]=tacname[0]=(char)0;
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') {
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    char *cptr=argv[ai]+1; if(*cptr=='-') {cptr++; if(!*cptr) continue;}
    if(!strcasecmp(cptr, "SD")) {printWhat=1; continue;}
    if(!strcasecmp(cptr, "CL1")) {printWhat=2; continue;}
    if(!strcasecmp(cptr, "CL2")) {printWhat=3; continue;}
    if(!strncasecmp(cptr, "PAR", 1)) {printWhat=0; continue;}
    fprintf(stderr, "Error: invalid option '%s'\n", argv[ai]);
    return(1);
  } else break;

  TPCSTATUS status; statusInit(&status);
  statusSet(&status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  status.verbose=verbose-1;
  
  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  /* Process other arguments, starting from the first non-option */
  if(ai<argc) strlcpy(parfile, argv[ai++], FILENAME_MAX);
  if(ai<argc) strlcpy(parname, argv[ai++], FILENAME_MAX);
  if(ai<argc) strlcpy(tacname, argv[ai++], FILENAME_MAX);
  if(ai<argc) {fprintf(stderr, "Error: too many arguments: '%s'.\n", argv[ai]); return(1);}

  /* Check that we got what we need */
  if(!tacname[0]) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  
  /* In verbose mode print arguments and options */
  if(verbose>1) {
    printf("parfile := %s\n", parfile);
    printf("parname := %s\n", parname);
    printf("tacname := %s\n", tacname);
    printf("printWhat := %d\n", printWhat);
  }


  /* 
   *  Read the file
   */
  if(verbose>1) printf("reading %s\n", parfile);
  PAR par; parInit(&par);
  if(parRead(&par, parfile, &status)!=TPCERROR_OK) {
    fprintf(stderr, "Error: %s\n", errorMsg(status.error));
    parFree(&par); return(2);
  }
  if(verbose>2) {
    printf("fileformat := %s\n", parFormattxt(par.format));
    printf("parNr := %d\n", par.parNr);
    printf("tacNr := %d\n", par.tacNr);
  }

  /* 
   *  Select the parameter to be shown
   */
  int parNr=0, tacNr=0;
  parNr=parSelectParameters(&par, parname, 1, &status);
  tacNr=parSelectTACs(&par, tacname, 1, &status);
  if(verbose>2) {
    printf("%d parameter(s) match '%s'\n", parNr, parname);
    printf("%d TAC(s) match '%s'\n", tacNr, tacname);
  }
  if(parNr==0) fprintf(stderr, "Error: no parameter matches '%s'.\n", parname);
  if(tacNr==0) fprintf(stderr, "Error: no TAC matches '%s'.\n", tacname);
  if(parNr==0 || tacNr==0) {parFree(&par); return(3);}
  if(tacNr>1 || parNr>1) {
    fprintf(stderr, "Error: cannot identify a single parameter value.\n");
    printf("The following parameters would match the given parameter and TAC name:\n");
    for(int pi=0; pi<par.parNr; pi++) if(par.n[pi].sw) {
      for(int ti=0; ti<par.tacNr; ti++) if(par.r[ti].sw)
        printf("  %s\t%s\n", par.n[pi].name, par.r[ti].name);
    }
    parFree(&par); return(3);
  }

  /*
   *  Get and print the selected parameter value
   */
  double v=nan("");
  for(int pi=0; pi<par.parNr; pi++) if(par.n[pi].sw) {
    for(int ti=0; ti<par.tacNr; ti++) if(par.r[ti].sw) {
      if(printWhat==1) v=par.r[ti].sd[pi];
      else if(printWhat==2) v=par.r[ti].cl1[pi];
      else if(printWhat==3) v=par.r[ti].cl2[pi];
      else v=par.r[ti].p[pi];
    }
  }
  /* Free memory */
  parFree(&par);
  /* Check */
  if(isnan(v)) {fprintf(stderr, "Error: requested a missing value.\n"); return(5);}
  /* and print */
  fprintf(stdout, "%g\n", v);

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
