#!/bin/bash
#: Title      : test_fit_wliv
#: Date       : 2019-10-25
#: Author     : "Vesa Oikonen" <vesa.oikonen@utu.fi>
#: Options    : None

# Set the name of executable to test
if [ -n "${OS+1}" ] && [ "$OS" = "Windows_NT" ]; then
EXT=.exe;
else
EXT=;
fi

PROGRAM=../../fit_wliv$EXT;
if [ ! -f $PROGRAM ]; then
  printf "Failed: executable does not exist.\n"
  exit 1
fi

printf "=====================================================================\n"
printf "creating test data for %s\n" $PROGRAM
printf "=====================================================================\n"


printf "\n verifying that required data exists \n"
if [ ! -f radiowater_lv.fit ] || [ ! -f wframes.sif ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
if [ ! -f real_lv.tac ] || [ ! -f real_liver.tac ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi

printf "\n creating data for testing \n"

if [ ! -f liver.par ] || [ ! -f correct.par ]; then
  rm -f liver.tac liver.bld
  printf "# model := radiowater-liver\n" > liver.par
  printf "Parameters	K1a[mL/(mL*min)]	K1p[mL/(mL*min)]	p[mL/mL]	Vb[mL/mL]	LdT[sec]	PdT[sec]	kGI[mL/(mL*min)]\n" >> liver.par
  printf "tac1	0.5	3.5	0.8	0.02	10	15	1\n" >> liver.par
  printf "tac2	0.4	2.6	0.6	0.02	10	15	1\n" >> liver.par
  printf "tac3	0.8	0	0.8	0.02	10	15	1\n" >> liver.par

  printf "# program := fit_wliv 0.7.4 (c) 2019 by Turku PET Centre\n" > correct.par
  printf "# model := radiowater-liver\n" >> correct.par
  printf "Parameters	K1a[mL/(mL*min)]	K1p[mL/(mL*min)]	k2[1/min]	Vb[mL/mL]	LdeltaT[sec]	PdeltaT[sec]	kGI[1/min]	p[mL/mL]	WSS\n" >> correct.par
  printf "tac1	0.5	3.5	5	0.02	10	15	1	0.8	0\n" >> correct.par
  printf "tac2	0.4	2.6	5	0.02	10	15	1	0.6	0\n" >> correct.par
  printf "tac3	0.8	0	1	0.02	10	0	0	0.8	0\n" >> correct.par
fi

if [ ! -f lv.tac ] || [ ! -f lv.bld ] || [ ! -f liver.tac ] || [ ! -f correct.par ]; then
  rm -f liver.tac liver.bld
  fit2dat -c=0,430,0.1 radiowater_lv.fit lv.bld
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  simframe lv.bld wframes.sif lv.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tac2svg -legend=no -mt="LV BTAC" lv.svg -s lv.tac -l lv.bld
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f liver.tac ] || [ ! -f liver.bld ]; then
  sim_wliv liver.par lv.bld liver.bld
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  simframe liver.bld wframes.sif liver.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tac2svg -legend=yes -mt="Simulated TACs" liver.svg -s lv.tac liver.tac -l lv.bld liver.bld
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi


printf "\n\n"
printf "=====================================================================\n"
printf "testing %s\n" $PROGRAM
printf "=====================================================================\n"



printf "\n 1.0.0 \n"
printf " Test case: Quick test with real PET data. \n"
printf " Expected result: No error, and fitted TAC is not too far from original. \n\n"

rm -f output1.svg output1.par output1.tac
$PROGRAM -wf -Vb=0 -sim=output1.tac -svg=output1.svg real_lv.tac real_liver.tac output1.par
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
if [ ! -f output1.tac ] || [ ! -f output1.svg ] || [ ! -f output1.par ]; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tac2svg -legend=no -mt="" output1t.svg real_lv.tac -s real_liver.tac -l output1.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=3000 -rel=10 real_liver.tac output1.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 1.0.1 \n"
printf " Test case: Another quick test with another real PET data. \n"
printf " Test case 2: Vb fixed to zero and 0.1 and fitted freely. \n"
printf " Note: Extra steps to make a plot for the web page. \n"
printf " Expected result: No error, and fitted TAC is not too far from original. \n\n"

rm -f output1vb0.par output1vb0.tac
$PROGRAM -Vb=0 -sim=output1vb0.tac real_lv2.tac real_liver2.tac output1vb0.par
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=4 -rel=5 real_liver2.tac output1vb0.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

rm -f output1vb10.par output1vb10.tac
$PROGRAM -Vb=0.10 -sim=output1vb10.tac real_lv2.tac real_liver2.tac output1vb10.par
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=4 -rel=5 real_liver2.tac output1vb10.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

rm -f output1vbg.par output1vbf.tac
$PROGRAM -sim=output1vbf.tac real_lv2.tac real_liver2.tac output1vbf.par
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=4 -rel=5 real_liver2.tac output1vbf.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

tacren --silent output1vb0.tac 0 Fit_no_Vb
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacren --silent output1vbf.tac 0 Fit_free_Vb
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

tac2svg -legend=yes -mt="" radiowater_fit_wliv.svg -s real_liver2.tac -b real_lv2.tac -l output1vb0.tac output1vbf.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"




printf "\n 1.1.0 \n"
printf " Test case 1: Test with simulated data. \n"
printf " Test case 2: Use frequently sampled input data. \n"
printf " Expected result: Resulting parameters very close to true ones. \n\n"

rm -f output2a.svg output2.par output2.tac
$PROGRAM -sim=output2.tac lv.bld liver.bld output2.par
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tac2svg -legend=no -mt="" output2a.svg -s liver.tac -l output2.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

tacmatch -abs=3000 -rel=10 liver.bld output2.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

parmatch -par=1-3 -rel=5 correct.par output2.par
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parmatch -par=4 -abs=0.02 correct.par output2.par
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parmatch -par=5,6 -abs=2 correct.par output2.par
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parmatch -par=7,8 -abs=0.05 correct.par output2.par
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 1.1.1 \n"
printf " Test case 1: Simulated data, framed tissue. \n"
printf " Test case 2: Fixed Vb and hepatic delay. \n"
printf " Expected result: Resulting parameters very close to true ones. \n\n"

rm -f output2b.svg output2b.par
$PROGRAM -Vb=0.02 -delay=10 -svg=output2b.svg lv.bld liver.tac output2b.par
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

parmatch -par=1-3 -rel=1 correct.par output2b.par
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parmatch -par=4,5 -abs=0.00001 correct.par output2b.par
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parmatch -par=6 -abs=1 correct.par output2b.par
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parmatch -par=7,8 -abs=0.01 correct.par output2b.par
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 1.1.2 \n"
printf " Test case 1: Simulated data, framed tissue and input. \n"
printf " Test case 2: Fixed Vb and both delays. \n"
printf " Expected result: Resulting parameters very close to true ones. \n\n"

rm -f output2c.svg output2c.par
$PROGRAM -Vb=0.02 -delay=10 -delayp=15 -svg=output2c.svg lv.tac liver.tac output2c.par
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

parmatch -par=1 -rel=10 -abs=0.05 correct.par output2c.par
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parmatch -par=2-3 -rel=10 -abs=2 correct.par output2c.par
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parmatch -par=4-6 -abs=0.00001 correct.par output2c.par
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parmatch -par=7 -abs=0.02 correct.par output2c.par
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parmatch -par=8 -abs=0.1 correct.par output2c.par
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n===================================================================\n"
printf "   All passed! \n"
printf "===================================================================\n"
exit 0

