#!/bin/bash
#: Title      : test_fit_wpul
#: Date       : 2019-10-14
#: Author     : "Vesa Oikonen" <vesa.oikonen@utu.fi>
#: Options    : None

# Set the name of executable to test
if [ -n "${OS+1}" ] && [ "$OS" = "Windows_NT" ]; then
EXT=.exe;
else
EXT=;
fi

PROGRAM=../../fit_wpul$EXT;
if [ ! -f $PROGRAM ]; then
  printf "Failed: executable does not exist.\n"
  exit 1
fi

printf "=====================================================================\n"
printf "creating test data for %s\n" $PROGRAM
printf "=====================================================================\n"


printf "\n verifying that required data exists \n"
if [ ! -f radiowater_rv.fit ] || [ ! -f wframes.sif ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
if [ ! -f real_rv.tac ] || [ ! -f real_rvlv.tac ] || [ ! -f real_lung.tac ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi

printf "\n creating data for testing \n"

if [ ! -f rv.tac ] || [ ! -f rv.bld ] || [ ! -f lung.tac ] || [ ! -f correct.par ]; then
  fit2dat -c=0,420,0.1 radiowater_rv.fit rv.bld
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  simframe rv.bld wframes.sif rv.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tac2svg -legend=no -mt="RV BTAC" rv.svg -s rv.tac -l rv.bld
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

  simdisp rv.bld 3 3 rvdispdel1.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  simdisp rv.bld 3 2 rvdispdel2.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

# k1 and k2 0.02 per sec equals 1.2 per min
  p2t_3c -nosub rvdispdel1.tac 0.02 0.02 0 0 0 0 lungtemp1.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  taccalc rvdispdel1.tac x 0.15 vb1.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  taccalc lungtemp1.tac + vb1.tac lungtemp1b.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  taccalc lungtemp1b.tac x 0.30 lungtemp1c.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

# k1 and k2 0.03 per sec equals 1.8 per min
  p2t_3c -nosub rvdispdel2.tac 0.03 0.03 0 0 0 0 lungtemp2.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  taccalc rvdispdel2.tac x 0.15 vb2.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  taccalc lungtemp2.tac + vb2.tac lungtemp2b.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  taccalc lungtemp2b.tac x 0.30 lungtemp2c.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

  simframe lungtemp1c.tac wframes.sif lung1.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  simframe lungtemp2c.tac wframes.sif lung2.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacadd -ovr lung.tac lung1.tac lung2.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacren lung.tac 0 Lung#
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

  tac2svg -mt="Simulated lung TTACs with RV cavity BTAC" -legend=y lung.svg rv.tac lung.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

  printf "# analysis_time := 2019-10-02 22:33:44\n" > correct.par
  printf "# program := fit_wpul 0.7.4 (c) 2019 by Turku PET Centre\n" >> correct.par
  printf "# inputfile := rv.bld\n" >> correct.par
  printf "# datafile := lung.tac\n" >> correct.par
  printf "# dataNr := 21\n" >> correct.par
  printf "# fit_start := 0\n" >> correct.par
  printf "# fit_end := 360\n" >> correct.par
  printf "Parameters	tau[sec]	deltaT[sec]	Vb[mL/mL]	K1[mL/(mL*min)]	k2[1/min]	K1/k2[mL/mL]	WSS\n" >> correct.par
  printf "Lung1	3	3	0.045	0.36	1.2	0.3	0\n" >> correct.par
  printf "Lung2	3	2	0.045	0.54	1.8	0.3	0\n" >> correct.par

fi




printf "\n\n"
printf "=====================================================================\n"
printf "testing %s\n" $PROGRAM
printf "=====================================================================\n"



printf "\n 1.0.0 \n"
printf " Test case 1: Test with simulated data. \n"
printf " Test case 2: Use frequently sampled input data. \n"
printf " Expected result: Resulting parameters very close to true ones. \n\n"

rm -f output1.svg output1.par
$PROGRAM -tau=3 -svg=output1.svg rv.bld lung.tac output1.par
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
if [ ! -f output1.svg ]; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

parmatch -p=y -tacnames=y -parnames=y -parunits=y -wss=n -abs=0.002 -rel=1 correct.par output1.par
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 1.0.1 \n"
printf " Test case 1: Test with simulated data. \n"
printf " Test case 2: Use input data from PET frames. \n"
printf " Expected result: Resulting parameters reasonably close to true ones. \n\n"

rm -f output2.svg output2.par
$PROGRAM -tau=3 -svg=output2.svg rv.tac lung.tac output2.par
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
if [ ! -f output2.svg ]; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

parmatch -par=5 -p=y -tacnames=y -parnames=y -parunits=y -wss=n -abs=0.02 -rel=5 correct.par output2.par
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parmatch -par=4 -p=y -tacnames=n -parnames=n -parunits=n -wss=n -abs=0.05 -rel=10 correct.par output2.par
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parmatch -par=3 -p=y -tacnames=n -parnames=n -parunits=n -wss=n -abs=0.04 -rel=50 correct.par output2.par
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parmatch -par=1,2 -p=y -tacnames=n -parnames=n -parunits=n -wss=n -abs=6 correct.par output2.par
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 1.1.0 \n"
printf " Test case 1: Test with simulated data. \n"
printf " Test case 2: Constrain Vb to the correct value. \n"
printf " Expected result: Resulting parameters very close to true ones. \n\n"

rm -f output1b.svg output1b.par
$PROGRAM -vb=0.045 -tau=3 -svg=output1b.svg rv.bld lung.tac output1b.par
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
if [ ! -f output1b.svg ]; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

parmatch -p=y -tacnames=y -parnames=y -parunits=y -wss=n -abs=0.001 -rel=1 correct.par output1b.par
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n===================================================================\n"

printf "\n 2.0.0 \n"
printf " Test case 1: Test with real PET data. \n"
printf " Test case 2: Dispersion is set to zero. \n"
printf " Test case 3: Vb is set to zero. \n"
printf " Expected result: No error. \n\n"

rm -f output3.svg output3.par
$PROGRAM -wf -tau=0 -vb=0 -svg=output3.svg real_rv.tac real_lung.tac output3.par
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
if [ ! -f output3.svg ]; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"


printf "\n 2.0.1 \n"
printf " Test case 1: Test with real PET data without weighting. \n"
printf " Test case 2: TTAC file has min as time units. \n"
printf " Test case 3: TTAC data is cut to 3.0 min. \n"
printf " Test case 4: Fitted TTACs are saved. \n"
printf " Expected result 1: Fitted TTACs can be plotted with BTAC and measured TTACs. \n"
printf " Expected result 2: Fitted TTACs are reasonably close to measured TTACs. \n\n"

taccut real_lung.tac 0 180 real_lung_min.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacunit -xconv=min real_lung_min.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
cp -f real_rv.tac real_rv_kbq.tac
tacunit -yconv=kBq/mL real_rv_kbq.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

rm -f output3m.tac output3m.svg output3m.par

$PROGRAM -tau=0 -vb=0 -sim=output3m.tac real_rv_kbq.tac real_lung_min.tac output3m.par
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
interpol -f=real_lung_min.tac output3m.tac temp.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tac2svg -legend=n -mt="" -x2=180 output3m.svg -b real_rv_kbq.tac -s real_lung_min.tac -l temp.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacunit -xconv=min -yconv=Bq/mL temp.tac
tacmatch -abs=3000 -rel=10 real_lung_min.tac temp.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"




printf "\n 2.1.0 \n"
printf " Test case 1: RV and LV BTACs are first fitted and fitted RV used as input. \n"
printf " Test case 2: TTAC file has different concentration units than BTAC. \n"
printf " Test case 3: Fitted TTACs are saved. \n"
printf " Expected result 1: Fitted TTACs are saved at BTAC sampling times. \n"
printf " Expected result 2: Fitted TTACs are reasonably close to measured TTACs. \n\n"

cp -f real_lung.tac real_lung_kbq.tac
rm -f output3s.tac output3s.svg
tacunit -yconv=kBq/mL real_lung_kbq.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

fit_wrlv real_rvlv.tac rvlv.fit
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
fit2dat -c=0,420,0.5 rvlv.fit fitted_rvlv.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacadd -ovr fitted_rv.tac fitted_rvlv.tac RV
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

$PROGRAM -wf -tau=0 -vb=0 -sim=output3s.tac fitted_rv.tac real_lung_kbq.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tac2svg -legend=y -mt="" output3s.svg -s real_lung_kbq.tac -l output3s.tac
if [ ! -f output3s.svg ]; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
interpol -f=real_lung.tac output3s.tac temp.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=3 -rel=10 real_lung_kbq.tac temp.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n===================================================================\n"
printf "   All passed! \n"
printf "===================================================================\n"
exit 0

