/** @file tacblend.c
 *  @brief Pool all data samples from given TAC files.
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <unistd.h>
/*****************************************************************************/
#include "tpcextensions.h"
#include "tpcift.h"
#include "tpctac.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Pools all samples in TAC files. Overlapping samples are preserved.",
  "Each TAC file must contain the same number of TACs with equal names,",
  "but sample number can be different.",
  " ",
  "Usage: @P [options] outputfile tacfiles",
  " ",
  "Options:",
  " --force",
  "     Program does not mind if the time or calibration units cannot be",
  "     converted to match, or if TAC names do not match.",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "See also: tacadd, taccat, tacjoin, tacunit, avgbolus, avgttac, tacmsamp",
  " ",
  "Keywords: TAC, tool",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int ai, help=0, version=0, verbose=1;
  int ret;
  int checkData=1;
  char tacfile[FILENAME_MAX], outfile[FILENAME_MAX];
  int fileNr=0, file1=0;


  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  tacfile[0]=outfile[0]=(char)0;
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') {
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    char *cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(!*cptr) continue;
    if(strcasecmp(cptr, "F")==0 || strcasecmp(cptr, "FORCE")==0) {
      checkData=0; continue;
    }
    fprintf(stderr, "Error: invalid option '%s'\n", argv[ai]);
    return(1);
  } else break; // tac name argument may start with '-'

  TPCSTATUS status; statusInit(&status);
  statusSet(&status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  status.verbose=verbose-3;
  
  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  /* Process other arguments, starting from the first non-option */
  if(ai<argc) strlcpy(outfile, argv[ai++], FILENAME_MAX);
  for(; ai<argc; ai++) {
    if(fileNr==0) file1=ai;
    fileNr++;
  }


  /* In verbose mode print arguments and options */
  if(verbose>1) {
    for(ai=0; ai<argc; ai++) printf("%s ", argv[ai]); 
    printf("\n");
    printf("fileNr := %d\n", fileNr);
    printf("outfile := %s\n", outfile);
    fflush(stdout);
  }

  /* Is something missing? */
  if(!outfile[0]) {tpcPrintUsage(argv[0], info, stdout); return(1);}
  if(fileNr==0) {
    fprintf(stderr, "Error: missing command-line argument; try %s --help\n", argv[0]);
    return(1);
  }
  if(fileNr==1) {
    fprintf(stderr, "Error: only one input file specified.\n");
    return(1);
  }

  
  /* Check that all input files do exist, and that their name does not match output file name */
  for(ai=file1; ai<argc; ai++) {
    strlcpy(tacfile, argv[ai], FILENAME_MAX);
    if(access(tacfile, 0) == -1) {
      fprintf(stderr, "Error: input file %s does not exist.\n", tacfile);
      return(2);
    }
    if(strcasecmp(outfile, tacfile)==0) {
      fprintf(stderr, "Error: input file would be overwritten.\n");
      return(2);
    }
  }


  /*
   *  Read the first file
   */
  TAC tac, pool;
  tacInit(&tac); tacInit(&pool);
  strlcpy(tacfile, argv[file1], FILENAME_MAX);
  if(verbose>1) printf("reading %s\n", tacfile);
  ret=tacRead(&pool, tacfile, &status);
  if(ret!=TPCERROR_OK) {
    fprintf(stderr, "Error (%d): %s\n", ret, errorMsg(status.error));
    tacFree(&pool); return(2);
  }
  if(checkData) tacSortByName(&pool, &status);
  if(verbose>2) {
    printf("fileformat := %s\n", tacFormattxt(pool.format));
    printf("tacNr := %d\n", pool.tacNr);
    printf("sampleNr := %d\n", pool.sampleNr);
    printf("xunit := %s\n", unitName(pool.tunit));
    printf("yunit := %s\n", unitName(pool.cunit));
  }
  if(tacXNaNs(&pool)>0) {
    fprintf(stderr, "Error: missing x values in %s\n", tacfile);
    tacFree(&pool); return(2);
  }
#if(0)
  /* Remove any weighting */
  pool.weighting=WEIGHTING_OFF;
#endif


  /*
   *  Add the data from the other files
   */
  int differentTUnits=0;
  int differentCUnits=0;
  for(ai=file1+1; ai<argc; ai++) {
    strlcpy(tacfile, argv[ai], FILENAME_MAX);
    if(verbose>1) printf("reading %s\n", tacfile);
    ret=tacRead(&tac, tacfile, &status);
    if(ret!=TPCERROR_OK) {
      fprintf(stderr, "Error (%d): %s\n", ret, errorMsg(status.error));
      tacFree(&pool); tacFree(&tac); return(2);
    }
    if(checkData) tacSortByName(&tac, &status);
    if(verbose>2) {
      printf("fileformat := %s\n", tacFormattxt(tac.format));
      printf("tacNr := %d\n", tac.tacNr);
      printf("sampleNr := %d\n", tac.sampleNr);
      printf("xunit := %s\n", unitName(tac.tunit));
      printf("yunit := %s\n", unitName(tac.cunit));
    }
    if(tacXNaNs(&pool)>0) {
      fprintf(stderr, "Error: missing x values in %s\n", tacfile);
      tacFree(&pool); tacFree(&tac); return(2);
    }
    /* Check if the TAC number is different */
    if(pool.tacNr!=tac.tacNr) {
      fprintf(stderr, "Error: different number of TACs.\n");
      tacFree(&pool); tacFree(&tac); return(3);
    }
    /* Check that TAC names match, if requested */
    if(checkData && tacCompareNames(&pool, &tac, -1, &status)!=0) {
      fprintf(stderr, "Error: TAC names do not match.\n");
      tacFree(&pool); tacFree(&tac); return(3);
    }
    /* Try to convert time units to time units in the first file */
    ret=tacXUnitConvert(&tac, pool.tunit, &status);
    if(ret==TPCERROR_UNKNOWN_UNIT) ret=TPCERROR_OK;
    if(ret!=TPCERROR_OK) {
      if(verbose>2) fprintf(stderr, "Status: %s\n", errorMsg(status.error));
      if(checkData) {
        fprintf(stderr, "Error: non-compatible TAC time units.\n");
        tacFree(&pool); tacFree(&tac); return(4);
      }
      differentTUnits++;
    }
    /* Try to convert concentration units to time units in the first file */
    ret=tacYUnitConvert(&tac, pool.cunit, &status);
    if(ret==TPCERROR_UNKNOWN_UNIT) ret=TPCERROR_OK;
    if(ret!=TPCERROR_OK) {
      if(verbose>2) fprintf(stderr, "Status: %s\n", errorMsg(status.error));
      if(checkData) {
        fprintf(stderr, "Error: non-compatible TAC concentration units.\n");
        tacFree(&pool); tacFree(&tac); return(5);
      }
      differentCUnits++;
    }
    /* If at least one of TAC files only has frame mid times, then all have */
    if(tac.isframe==0) pool.isframe=0;
    /* Add the new sample values */
    ret=tacAllocateMoreSamples(&pool, tac.sampleNr);
    if(ret!=TPCERROR_OK) {
      fprintf(stderr, "Error: cannot allocate memory.\n");
      tacFree(&pool); tacFree(&tac); return(6);
    }
    for(int i=0; i<tac.sampleNr; i++) {
      pool.x[pool.sampleNr]=tac.x[i];
      pool.x1[pool.sampleNr]=tac.x1[i];
      pool.x2[pool.sampleNr]=tac.x2[i];
      for(int j=0; j<pool.tacNr; j++) pool.c[j].y[pool.sampleNr]=tac.c[j].y[i];
      pool.w[pool.sampleNr]=tac.w[i];
      pool.sampleNr++;
    }
    tacFree(&tac);
  }
  if(verbose>0 && differentTUnits>1) fprintf(stderr, 
    "Warning: time units could not be converted for %d file(s).\n", differentTUnits);
  if(verbose>0 && differentCUnits>1) fprintf(stderr, 
    "Warning: concentration units could not be converted for %d file(s).\n", differentCUnits);

  /*
   *  Sort pooled data by sample time
   */
  ret=tacSortByTime(&pool, &status);
  if(ret!=TPCERROR_OK) {
    fprintf(stderr, "Error: cannot sort the pooled data.\n");
    tacFree(&pool); return(8);
  }
  /* Delete study number, if one exists */
  (void)tacSetHeaderStudynr(&pool.h, "");


  /*
   *  Save data 
   */
  if(verbose>1) printf("writing %s\n", outfile);
  FILE *fp; fp=fopen(outfile, "w");
  if(fp==NULL) {
    fprintf(stderr, "Error: cannot open file for writing (%s)\n", outfile);
    tacFree(&tac); return(11);
  }
  ret=tacWrite(&pool, fp, TAC_FORMAT_UNKNOWN, 1, &status);
  fclose(fp);
  if(ret!=TPCERROR_OK) {
    fprintf(stderr, "Error (%d): %s\n", ret, errorMsg(status.error));
    tacFree(&pool); return(12);
  }
  if(verbose>=0) {
    printf("%d samples from %d TAC files pooled.\n", pool.sampleNr, fileNr);
    printf("%s saved.\n", outfile);
  }
  tacFree(&pool);

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
