/** @file taccuty.c
 *  @brief Extract specified y (concentration) interval from the TAC.
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
/*****************************************************************************/
#include "tpcextensions.h"
#include "tpctac.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Extract specified y (concentration) interval from PET time-activity",
  "curves (TACs).",
  " ",
  "Usage: @P [Options] tacfile lower upper [outputfile]",
  " ",
  "Options:",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "See also: taccut, tacblend, tacinv, tacsety, dftmax, tacdel, tacsplit, taclkup",
  " ",
  "Keywords: TAC, tool, simulation, cropping",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int    ai, help=0, version=0, verbose=1;
  int    origNr, ret;
  char   dfile1[FILENAME_MAX], dfile2[FILENAME_MAX];
  double lowerY, upperY;

  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  dfile1[0]=dfile2[0]=(char)0;
  lowerY=upperY=nan("");
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') {
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    fprintf(stderr, "Error: invalid option '%s'\n", argv[ai]);
    return(1);
  } else break; 

  TPCSTATUS status; statusInit(&status);
  statusSet(&status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  status.verbose=verbose-1;
  
  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  /* Process other arguments, starting from the first non-option */
  if(ai<argc) {strlcpy(dfile1, argv[ai++], FILENAME_MAX);}
  if(ai<argc) {lowerY=atofVerified(argv[ai++]);}
  if(ai<argc) {upperY=atofVerified(argv[ai++]);}
  if(ai<argc) {strlcpy(dfile2, argv[ai++], FILENAME_MAX);}
  if(ai<argc) {
    fprintf(stderr, "Error: too many arguments: '%s'.\n", argv[ai]);
    return(1);
  }

  /* Check that we got what we need */
  if(!dfile1[0] || isnan(lowerY) || isnan(upperY) || upperY<lowerY) {
    tpcPrintUsage(argv[0], info, stderr); return(1);
  }

  /* If output filename was not given, then edit the input file */
  if(!dfile2[0]) strcpy(dfile2, dfile1);

  /* In verbose mode print arguments and options */
  if(verbose>1) {
    printf("dfile1 := %s\n",  dfile1);
    printf("dfile2 := %s\n",  dfile2);
    printf("lowerY := %g\n",  lowerY);
    printf("upperY := %g\n",  upperY);
  }


  /* 
   *  Read data
   */
  if(verbose>1) printf("reading %s\n", dfile1);
  TAC tac; tacInit(&tac);
  ret=tacRead(&tac, dfile1, &status);
  if(ret!=TPCERROR_OK) {
    fprintf(stderr, "Error: %s\n", errorMsg(status.error));
    tacFree(&tac); return(2);
  }
  if(verbose>1) {
    printf("fileformat := %s\n", tacFormattxt(tac.format));
    printf("tacNr := %d\n", tac.tacNr);
    printf("sampleNr := %d\n", tac.sampleNr);
  }
  origNr=tacNotNaNs(&tac, -1); // includes both x and y values

  /* Get y range */
  double ymin, ymax;
  ret=tacYRange(&tac, -1, &ymin, &ymax, NULL, NULL, NULL, NULL);
  if(ret!=0 || origNr<1) {
    fprintf(stderr, "Error: invalid value range in file.\n");
    tacFree(&tac); return(2);
  }
  if(verbose>1) {
    printf("orig_min := %g\n", ymin);
    printf("orig_max := %g\n", ymax);
  }
  /* Check if we already are inside the required range */
  if(ymin>=lowerY && ymax<=upperY) {
    if(verbose>1) printf("all values are inside required range.\n");
    /* Just quit, if user did not even give a new filename */
    if(strcasecmp(dfile1, dfile2)==0) {
      if(verbose>0) printf("nothing done.\n");
      return(0);
    }
    /* Otherwise save the data with new name */
    if(verbose>1) printf("writing %s\n", dfile2);
    FILE *fp; fp=fopen(dfile2, "w");
    if(fp==NULL) {
      fprintf(stderr, "Error: cannot open file for writing (%s)\n", dfile2);
      tacFree(&tac); return(11);
    }
    ret=tacWrite(&tac, fp, TAC_FORMAT_UNKNOWN, 1, &status);
    fclose(fp);
    if(ret!=TPCERROR_OK) {
      fprintf(stderr, "Error: %s\n", errorMsg(status.error));
      tacFree(&tac); return(12);
    }
    if(verbose>0) printf("All sample(s) were extracted.\n");
    /* ... and quit */
    tacFree(&tac);
    return(0);
  }


  /*
   *  Set outlier values to NaN
   */
  for(int i=0; i<tac.sampleNr; i++) {
    int j, n=0;
    for(j=0; j<tac.tacNr; j++) {
      if(tac.c[j].y[i]<lowerY || tac.c[j].y[i]>upperY) tac.c[j].y[i]=nan("");
      if(!isnan(tac.c[j].y[i])) n++;
    }
    /* If no y values left in this time frame, then set also x to NaN */
    if(n==0) tac.x[i]=tac.x1[i]=tac.x2[i]=nan(""); 
  }

  /* Remove time frames that have no data left */
  if(tacXNaNs(&tac)>0) {
    if(verbose>1) printf("deleting missing samples\n");
    ret=tacDeleteMissingSamples(&tac);
  }
  int finalNr=tacNotNaNs(&tac, -1); // includes both x and y values
  if(verbose>1) {
    printf("original_sampleNr := %d\n", origNr);
    printf("final_sampleNr := %d\n", finalNr);
  }
  if(tac.sampleNr==0) {
    fprintf(stderr, "Error: no data inside the range.\n");
    tacFree(&tac); return(9);
  }

  /*
   *  Save the extracted TACs
   */
  if(verbose>1) printf("writing %s\n", dfile2);
  FILE *fp; fp=fopen(dfile2, "w");
  if(fp==NULL) {
    fprintf(stderr, "Error: cannot open file for writing (%s)\n", dfile2);
    tacFree(&tac); return(11);
  }
  ret=tacWrite(&tac, fp, TAC_FORMAT_UNKNOWN, 1, &status);
  fclose(fp);
  if(ret!=TPCERROR_OK) {
    fprintf(stderr, "Error: %s\n", errorMsg(status.error));
    tacFree(&tac); return(12);
  }

  /* Free memory */
  tacFree(&tac);

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
