/** @file tacformat.c
 *  @brief View or set TAC data file format.
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 *  @test Test more file formats.
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
/*****************************************************************************/
#include "tpcextensions.h"
#include "tpcift.h"
#include "tpctac.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Convert regional or input TAC data in filename1 into filename2",
  "in specified file format.",
  "Data can be written only in a few formats (listed below), but more",
  "file formats can be read.",
  " ",
  "Usage: @P [options] filename1 [filename2]",
  " ",
  "Options:",
  " -f=<format-id>",
  "     Accepted format-id's:",
  "     CSV-INT   - CSV format with semicolons and decimal commas.",
  "     CSV-UK    - CSV format with commas and decimal points.",
  "     TSV-INT   - TSV format with tabs and decimal commas.",
  "     TSV-UK    - TSV format with tabs and decimal points.",
  "     PMOD      - PMOD tac and bld format.",
  "     DFT       - TPC TAC format.",
  "     SIMPLE    - txt file with tabs and decimal points.",
  "     XML       - MS Excel compatible XML format.",
  "     Without this option, only the format of the given TAC file is shown.",
  " -hdr=<Yes|no>",
  "     Extra information is (y, default) or is not stored (n) in lines",
  "     starting with '#' character; not effective with all formats.",
  " -mid",
  "     Frame mid times are used instead of frame start and end times.",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "Example 1: convert DFT file into PMOD TAC format",
  "     @P -f=pmod -hdr=no iea446.dft iea446.tac",
  " ",
  "Example 2a: convert all TAC files into DFT format on Windows",
  "     for %g in (*.tac) do @P -f=DFT -hdr=no %g",
  " ",
  "Example 2b: the same on linux and macOS",
  " ",
  "     for file in ./*.tac; do tacformat -f=dft -hdr=no $file; done",
  " ",
  "See also: taclist, tacunit, tacframe, tacnames, tac2xml",
  " ",
  "Keywords: TAC, tool, format, PMOD, CSV, DFT",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int ai, help=0, version=0, verbose=1;
  int ret;
  int new_format=TAC_FORMAT_UNKNOWN;
  int save_header=1;
  int mid_time=0;
  char *cptr, tacfile1[FILENAME_MAX], tacfile2[FILENAME_MAX];
  TAC tac;

  
  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  tacInit(&tac);
  tacfile1[0]=tacfile2[0]=(char)0;
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') {
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(!*cptr) continue;
    if(strncasecmp(cptr, "F=", 2)==0) {
      new_format=tacFormatIdentify(cptr+2);
      if(new_format!=TAC_FORMAT_UNKNOWN) continue;
    } else if(strncasecmp(cptr, "FORMAT=", 7)==0) {
      new_format=tacFormatIdentify(cptr+7);
      if(new_format!=TAC_FORMAT_UNKNOWN) continue;
    } else if(strncasecmp(cptr, "HDR=", 4)==0) {
      cptr+=4;
      if(strncasecmp(cptr, "YES", 1)==0) {save_header=1; continue;}
      else if(strncasecmp(cptr, "NO", 1)==0) {save_header=0; continue;}
    } else if(strncasecmp(cptr, "HEADER=", 7)==0) {
      cptr+=7;
      if(strncasecmp(cptr, "YES", 1)==0) {save_header=1; continue;}
      else if(strncasecmp(cptr, "NO", 1)==0) {save_header=0; continue;}
    } else if(strncasecmp(cptr, "MIDDLE", 3)==0) {
      mid_time=1; continue;
    }
    fprintf(stderr, "Error: invalid option '%s'.\n", argv[ai]);
    return(1);
  } else break; // tac name argument may start with '-'

  TPCSTATUS status; statusInit(&status);
  statusSet(&status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  status.verbose=verbose-1;

  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  /* Process other arguments, starting from the first non-option */
  if(ai<argc) {strlcpy(tacfile1, argv[ai++], FILENAME_MAX);}
  if(ai<argc) {strlcpy(tacfile2, argv[ai++], FILENAME_MAX);}
  if(ai<argc) {fprintf(stderr, "Error: too many arguments: '%s'.\n", argv[ai]); return(1);}

  /* Is something missing? */
  if(!tacfile1[0]) {tpcPrintUsage(argv[0], info, stdout); return(1);}

  /* In verbose mode print arguments and options */
  if(verbose>1) {
    printf("new_format := %s\n", tacFormattxt(new_format));
    printf("save_header := %d\n", save_header);
    printf("tacfile1 := %s\n", tacfile1);
    if(tacfile2[0]) printf("tacfile2 := %s\n", tacfile2);
    fflush(stdout);
  }


  /*
   *  Read TAC data
   */
  if(verbose>1) printf("reading %s\n", tacfile1);
  ret=tacRead(&tac, tacfile1, &status);
  if(ret) { // error
    tacFree(&tac);
    /* If format conversion was requested, then this certainly is bad */
    if(new_format!=TAC_FORMAT_UNKNOWN) {
      fprintf(stderr, "Error: %s (%s)\n", errorMsg(status.error), tacfile1);
      return(2);
    }
    /* User just wanted to know the current format */
    ret=tacFormatDetermine(tacfile1, &status);
    if(ret==TAC_FORMAT_UNKNOWN) {
      fprintf(stderr, "Error: %s (%s)\n", errorMsg(status.error), tacfile1);
      return(2);
    }
    fprintf(stdout, "%s\n", tacFormattxt(ret));
    return(0);
  }
  
  /* If new format was not specified, then just print the current format */
  if(new_format==TAC_FORMAT_UNKNOWN) {
    fprintf(stdout, "%s\n", tacFormattxt(tac.format));
    tacFree(&tac);
    return 0;
  }
  
  /* If output file name was not given by user, then make it here */
  if(!tacfile2[0]) {
    strcpy(tacfile2, tacfile1);
    filenameRmPath(tacfile2); filenameRmExtension(tacfile2);
    strcat(tacfile2, tacDefaultExtension(new_format));
  }
  if(verbose>1) {
    printf("tacfile2 := %s\n", tacfile2);
  }
  
  /* Use mid times if requested */
  if(mid_time) tac.isframe=0;

  /* Print current extra headers for testing */
  if(verbose>4) {
    printf("\n---- tac.h ----\n");
    iftWrite(&tac.h, stdout, NULL);
    printf("---------------\n\n");
  }

  /* Try to save the data in the new format */
  if(verbose>1) printf("writing %s\n", tacfile2);
  FILE *fp; fp=fopen(tacfile2, "w");
  if(fp==NULL) {
    fprintf(stderr, "Error: cannot open file for writing (%s)\n", tacfile2);
    tacFree(&tac); return(5);
  }
  ret=tacWrite(&tac, fp, new_format, save_header, &status);
  fclose(fp);
  tacFree(&tac);
  if(ret==TPCERROR_UNSUPPORTED) {
    fprintf(stderr, "Error: writing format %s is not supported.\n", tacFormattxt(new_format));
    return(6);
  }
  if(ret!=TPCERROR_OK) {
    fprintf(stderr, "Error: %s\n", errorMsg(status.error));
    return(7);
  }
  if(verbose>0) printf("  %s written.\n", tacfile2);
  
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
