/** @file tacren.c
 *  @brief Rename TAC(s) inside TAC file.
 *  @details Combines the functionality of previous applications dftren and dftrenpl. 
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <math.h>
/*****************************************************************************/
#include "tpcextensions.h"
#include "tpcift.h"
#include "tpctac.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Rename the specified TAC(s) inside a TAC file.",
  " ",
  "Usage: @P [options] file tac_id newname",
  " ",
  "Enter the number or (partial) name of the TAC to be renamed.",
  "Set tac_id to 0, if all TAC names are to be changed.",
  " ",
  "For newname, use '_' to separate for example hemisphere or other position",
  "from the target name. Character '#' is replaced by sequential number.",
  "Character '@' specifies that previous partial name is to be kept.",
  " ",
  "Note that not all TAC file formats can store TAC names.",
  "At least PMOD and DFT formats support TAC names.",
  " ",
  "Options:",
  " -dry",
  "     Nothing is actually renamed, application only lists which TAC(s) would",
  "     be renamed and how.",
  " -add=<field_nr>",
  "     New field name is added to given position; enter large number to add",
  "     it to the end.",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "Example 1: Rename TAC number 3 to 'str_dx'",
  "     @P a123.tac 3 str_dx",  
  " ",
  "Example 2: Rename all TACs to TACx where x is the number of the TAC",
  "     @P a234.tac 0 TAC#",  
  " ",
  "Example 3: Rename name field 'sn' to 'sin'",
  "     @P a345.tac sn @_sin_@",
  " ",
  "Example 4: Add TAC nr as the first name field, and keep the next fields intact",
  "     @P -add=1 a456.tac 0 #_@_@_@",
  " ",
  "See also: taclist, tacnames, tacadd, tacformat, tacsort, tacunit, tacdel",
  " ",
  "Keywords: TAC, tool, rename, simulation",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int ai, help=0, version=0, verbose=1;
  int ret, dryMode=0, addField=0;
  char *cptr, tacfile[FILENAME_MAX], tacid[256], tacname[256];
  TAC tac;
  unsigned int n;

  
  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  tacInit(&tac);
  tacfile[0]=tacid[0]=tacname[0]=(char)0;
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') {
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(!*cptr) continue;
    if(strcasecmp(cptr, "DRY")==0) {
      dryMode=1; continue;
    } else if(strncasecmp(cptr, "ADD=", 4)==0) {
      addField=atoi(cptr+4); if(addField>0) continue;
    }
    fprintf(stderr, "Error: invalid option '%s'\n", argv[ai]);
    return(1);
  } else break; // tac name argument may start with '-'

  TPCSTATUS status; statusInit(&status);
  statusSet(&status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  if(verbose>1) status.verbose=verbose-1;
  
  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  /* The first argument (non-option) is the file name */
  if(ai<argc) {
    strlcpy(tacfile, argv[ai], FILENAME_MAX); ai++;
  } else {
    fprintf(stderr, "Error: missing filename.\n");
    return(1);
  }
  /* Then user should have given the TAC id */
  if(ai<argc) {
    strlcpy(tacid, argv[ai], 256); ai++;
  } else {
    fprintf(stderr, "Error: missing TAC name or number.\n");
    return(1);
  }
  /* And the rest should be the new name */
  if(ai>=argc && addField==0) {
    fprintf(stderr, "Error: missing new name for TAC(s).\n");
    return(1);
  }
  n=0;
  for(; ai<argc; ai++) {
    if( (1+strlen(argv[ai])+strlen(tacname)) >= 255 ) {
      fprintf(stderr, "Error: new name for TAC(s) is too long.\n");
      return(1);
    }
    if(n>0) strncat(tacname, "_", 255-strlen(tacname));
    strncat(tacname, argv[ai], 255-strlen(tacname)); 
    n++;
  }
  
  
  /* In verbose mode print arguments and options */
  if(verbose>1) {
    printf("tacfile := %s\n", tacfile);
    printf("dryMode := %d\n", dryMode);
    printf("addField := %d\n", addField);
    printf("tac_id := '%s'\n", tacid);
    printf("new_tacname := '%s'\n", tacname);
    fflush(stdout);
  }


  /* Check that new name contains only accepted characters */
  /* And check if it contains # or @ characters */
  int toNum=0, toKeep=0;
  for(int unsigned i=0; i<strlen(tacname); i++) {
    if(tacname[i]=='#') {toNum++; continue;}
    if(tacname[i]=='@') {toKeep++; continue;}
    if(tacname[i]=='_') continue;
    if(tacname[i]=='-') continue;
    if(isalnum(tacname[i])) continue;
    fprintf(stderr, "Error: invalid character '%c' in new tacname.\n", tacname[i]);
    return(1);
  }
  if(verbose>1) {
    printf("edited_new_tacname := %s\n", tacname);
    printf("toNum := %d\n", toNum);
    printf("toKeep := %d\n", toKeep);
    fflush(stdout);
  }
  if(toNum>1) {
    fprintf(stderr, "Error: do not use '#' more than once.\n"); 
    return(1);
  }
  /* Check that we have *_@_* or @_* or *_@ */
  if(toKeep>0) {
    ret=0;
    for(int unsigned i=1; i<strlen(tacname)-1; i++) {
      if(tacname[i]!='@') continue;
      if(tacname[i-1]!='_' || tacname[i+1]!='_') ret++;
    }
    if(ret) {
      fprintf(stderr, "Error: '@' must be separated by '_'.\n"); 
      return(1);
    }
  } 

  /* 
   *  Read the file 
   */
  if(verbose>1) printf("reading %s\n", tacfile);
  ret=tacRead(&tac, tacfile, &status);
  if(ret!=TPCERROR_OK) {
    fprintf(stderr, "Error: %s\n", errorMsg(status.error));
    tacFree(&tac); return(2);
  }
  if(verbose>2) {
    printf("fileformat := %s\n", tacFormattxt(tac.format));
    printf("tacNr := %d\n", tac.tacNr);
    printf("sampleNr := %d\n", tac.sampleNr);
    fflush(stdout);
  }
  if(tac.tacNr<1) {
    fprintf(stderr, "Error: no TACs in %s\n", tacfile);
    tacFree(&tac); return(2);
  }

  /* Select the regions */
  if(strcmp(tacid, "0")==0) {
    for(int j=0; j<tac.tacNr; j++) tac.c[j].sw=1;
    n=tac.tacNr;
    if(verbose>1) printf("All %d tac(s) selected.\n", n);
  } else {
    n=tacSelectTACs(&tac, tacid, 1, &status);
    if(n<1) {
      fprintf(stderr, "Error: specified TACs not found in %s.\n", tacfile);
      tacFree(&tac); return(3);
    } else if(verbose>1) 
      printf("%d tac(s) match tac_id '%s'\n", n, tacid);
  }


  /*
   *  Rename the selected TACs
   */
  char oldname[MAX_TACNAME_LEN+1], newname[MAX_TACNAME_LEN+1];
  char tmp[1024];
  for(int j=0; j<tac.tacNr; j++) if(tac.c[j].sw) {
    if(verbose>1) printf("\nold_name := '%s'\n", tac.c[j].name);
    /* Get the current TAC name, without extra spaces or quotation marks */
    strncpyClean(oldname, tac.c[j].name, MAX_TACNAME_LEN);
    /* '.' means that name is actually empty */
    if(strcmp(oldname, ".")==0) strcpy(oldname, "");
    /* Add TAC name field, if required */
    if(addField>0) {
      if(!roinameAddField(oldname, "", addField-1, MAX_TACNAME_LEN)) {
        fprintf(stderr, "Error: cannot add name field.\n");
        tacFree(&tac); return(4);
      }
    }
    /* Pre-process the template into tmp */
    if(strlen(tacname)==0) {
      /* Empty template can be given, if user wants to delete ROI name, or
         if user just wants to add a name field */
      if(addField>0) strcpy(tmp, oldname); else strcpy(tmp, "");
    } else {
      /* Replace the template # with TAC number */
      strcpy(tmp, tacname);
      if(toNum) {
        unsigned int i;
        if(verbose>10) printf("j+1=%d tmp := '%s'\n", j+1, tmp);
        i=strcspn(tacname, "#"); tmp[i]=(char)0; 
        if(verbose>10) printf("tmp := '%s' i=%d\n", tmp, i);
        strncatIntZ(tmp, 1+j, tac.tacNr, 1024-strlen(tmp)-1);
        if(verbose>10) printf("tmp := '%s'\n", tmp);
        strncat(tmp, tacname+i+1, 1024-strlen(tmp)-1);
        if(verbose>10) printf("tmp := '%s'\n", tmp);
        if(strlen(tmp)>MAX_TACNAME_LEN) {
          fprintf(stderr, "Error: too long TAC name '%s'\n", tmp);
          tacFree(&tac); return(5);
        }
      }
    }
    /* Set the new name based on the template */
    if(!roinameEditByTemplate(tmp, oldname, newname, MAX_TACNAME_LEN+1)) {
      fprintf(stderr, "Error: invalid TAC name '%s'\n", tmp);
      tacFree(&tac); return(6);
    }

    /* Tell user what we are doing */
    if(verbose>0 || dryMode) {printf("%s -> %s\n", tac.c[j].name, newname); fflush(stdout);}

    
    /* Replace old name with the new name */
    if(strlen(newname)>MAX_TACNAME_LEN) {
      fprintf(stderr, "Error: too long TAC name '%s'\n", newname);
      tacFree(&tac); return(7);
    }
    strcpy(tac.c[j].name, newname);

    /* Do not let data to be saved in simple format if there is at least one TAC name */
    if(tac.format==TAC_FORMAT_SIMPLE && strlen(newname)>0) tac.format=TAC_FORMAT_DFT;
    
  } // next TAC


  if(dryMode) {
    tacFree(&tac); return(0);
  } 

  /*
   *  Save data 
   */
  if(verbose>1) printf("writing %s\n", tacfile);
  FILE *fp; fp=fopen(tacfile, "w");
  if(fp==NULL) {
    fprintf(stderr, "Error: cannot open file for writing (%s)\n", tacfile);
    tacFree(&tac); return(11);
  }
  if(verbose>2) printf("format := %s\n", tacFormattxt(tac.format));
  ret=tacWrite(&tac, fp, TAC_FORMAT_UNKNOWN, 1, &status);
  fclose(fp); tacFree(&tac);
  if(ret!=TPCERROR_OK) {
    fprintf(stderr, "Error (%d): %s\n", ret, errorMsg(status.error));
    return(12);
  }

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
