#!/bin/bash
#: Title      : test_fit_h2o
#: Date       : 2022-03-29
#: Author     : "Vesa Oikonen" <vesa.oikonen@utu.fi>
#: Options    : None

# Set the name of executable to test
if [ -n "${OS+1}" ] && [ "$OS" = "Windows_NT" ]; then
EXT=.exe;
else
EXT=;
fi

PROGRAM=../../fit_h2o$EXT;

if [ ! -f $PROGRAM ]; then
  printf "Failed: executable does not exist.\n"
  exit 1
fi

printf "=====================================================================\n"
printf "creating test data for %s\n" $PROGRAM
printf "=====================================================================\n"

printf "\n verifying that required data exists \n"

if [ ! -f blood.dat ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
if [ ! -f frames.dat ] || [ ! -f frames_smallgap.dat ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
if [ ! -f frames_biggap.dat ] || [ ! -f frames_smalloverlap.dat ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
if [ ! -f frames_bigoverlap.dat ] || [ ! -f frames_quick.dat ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
if [ ! -f correct_quick.res ] || [ ! -f correct_quick_ml.res ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
if [ ! -f correct_quick_bs.res ] || [ ! -f correct_basicsim.res ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi


printf "\n making data for testing \n"

if [ ! -f blood.bld ]; then
  rm -f blood_na.bld
  rm -f blood_3.bld
  rm -f tac_f003_va1.dft tac_f030_va3.dft tac_f300_va6.dft
  printf "\n making blood.bld \n"
  tacformat -f=PMOD blood.dat blood.bld
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tac2svg -mt="Input" -legend=y input.svg -l blood.bld 
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f blood_na.bld ]; then
  printf "\n Make input with NaN \n"
  cp -f blood.bld blood_na.bld
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacsety blood_na.bld 1 10 nan
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f blood_3.bld ]; then
  printf "\n Make input with only 3 samples \n"
  interpol -x=5,18,68 blood.bld blood_3.bld
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f tac_f003_va1.dft ] || [ ! -f tac_f003_va1.dft ]; then
  rm -f basicsim.dft basicsim.tac
  printf "\n Simulate TTAC \n"
  b2t_h2o -nosub blood.dat 3 0.8 1 1 100 tac_f003_va1.dft
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacren --silent tac_f003_va1.dft 1 f003_va1
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tactime tac_f003_va1.dft -5 tac_f003_va1_m05.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  simframe --silent tac_f003_va1_m05.dat frames.dat tac_f003_va1_m05.dft O-15
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f tac_f030_va3.dft ] || [ ! -f tac_f030_va3_p05f.dft ]; then
  rm -f basicsim.dft basicsim.tac
  printf "\n Simulate TTAC \n"
  b2t_h2o -nosub blood.dat 30 0.8 1 3 100 tac_f030_va3.dft
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacren --silent tac_f030_va3.dft 1 f030_va3
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tactime tac_f030_va3.dft +5 tac_f030_va3_p05.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  simframe --silent tac_f030_va3_p05.dat frames.dat tac_f030_va3_p05.dft O-15
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f tac_f300_va6.dft ] || [ ! -f tac_f300_va6_m10f.dft ]; then
  rm -f basicsim.dft basicsim.tac
  printf "\n Simulate TTAC \n"
  b2t_h2o -nosub blood.dat 300 0.8 1 6 100 tac_f300_va6.dft
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacren --silent tac_f300_va6.dft 1 f300_va6
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tactime tac_f300_va6.dft -10 tac_f300_va6_m10.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  simframe --silent tac_f300_va6_m10.dat frames.dat tac_f300_va6_m10.dft O-15
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f basicsim.dft ] || [ ! -f basicsim.tac ]; then
  printf "\n Making basicsim.tac \n"
  tacadd --silent -ovr basicsim.dft tac_f003_va1_m05.dft
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacadd --silent basicsim.dft tac_f030_va3_p05.dft
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacadd --silent basicsim.dft tac_f300_va6_m10.dft
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacformat -f=PMOD basicsim.dft basicsim.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tac2svg -mt="Simulated TTACs" -legend=y basicsim.svg -l basicsim.tac 
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi


if [ ! -f quick.dft ] || [ ! -f quick.tac ] || [ ! -f quick_fpt.dft ]; then
  printf "\n Simulates TTAC for quick fitting \n"
  rm -f quick_na.dft quick_3.dft 
  rm -f normal.dft smallgap.dft biggap.dft smalloverlap.dft bigoverlap.dft 
  b2t_h2o -nosub blood.dat 20 1.0 1 4 50 tac_f20_va2.dft
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacren --silent tac_f20_va2.dft 1 f20_va2_std
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  simframe --silent tac_f20_va2.dft frames_quick.dat quick.dft O-15
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacformat -f=PMOD quick.dft quick.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

  b2t_h2o -fpt -nosub blood.dat 20 1.0 1 4 50 tac_f20_va2_fpt.dft
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacren --silent tac_f20_va2_fpt.dft 1 f20_va2_fpt
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  simframe --silent tac_f20_va2_fpt.dft frames_quick.dat quick_fpt.dft O-15
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacformat -f=PMOD quick_fpt.dft quick_fpt.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tac2svg -mt="Simulated TTACs" -legend=y quick.svg -l quick.tac quick_fpt.tac 
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi


if [ ! -f quick_na.dft ]; then
  printf "\n Make tissue data with NaN \n"
  cp -f quick.dft quick_na.dft
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacsety quick_na.dft 1 4 nan
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi


if [ ! -f quick_3.dft ] || [ ! -f frames_3.dat ]; then
  rm -f frames_3.dat
  printf "\n Make tissue data with just three frames \n"
  printf "0 30\n" > frames_3.dat
  printf "30 30\n" >> frames_3.dat
  printf "60 30\n" >> frames_3.dat
  simframe --silent tac_f20_va2.dft frames_3.dat quick_3.dft O-15
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f normal.dft ]; then
  simframe --silent tac_f20_va2.dft frames.dat normal.dft O-15
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f smallgap.dft ]; then
  simframe --silent tac_f20_va2.dft frames_smallgap.dat smallgap.dft O-15
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f biggap.dft ]; then
  simframe --silent tac_f20_va2.dft frames_biggap.dat biggap.dft O-15
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f smalloverlap.dft ]; then
  simframe --silent tac_f20_va2.dft frames_smalloverlap.dat smalloverlap.dft O-15
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f bigoverlap.dft ]; then
  simframe --silent tac_f20_va2.dft frames_bigoverlap.dat bigoverlap.dft O-15
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi


printf "\n\n"
printf "=====================================================================\n"
printf "testing %s\n" $PROGRAM
printf "=====================================================================\n"





printf "\n 1.0.0 \n"
printf " Test case: Fit simulated TACs with realistic range of parameters. \n"
printf " Expected result: Result file is created, and results are close to correct. \n"

rm -f output.res
$PROGRAM blood.dat quick.dft 999 output.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parmatch -abs=0.001 -rel=3 -tacnames=y -parunits=y -par=1-4 correct_quick.res output.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 1.0.1 \n"
printf " Test case: Same as 1.0.0, but using option -fpt. \n"
printf " Expected result: Result file is created, and results are close to correct. \n"

rm -f output.res
$PROGRAM -fpt blood.dat quick_fpt.dft 999 output.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parmatch -abs=0.001 -rel=3 -tacnames=n -parunits=y -par=1-4 correct_quick.res output.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 1.0.2 \n"
printf " Test case: Same as 1.0.0, but using options -va=2 and -delay=0. \n"
printf " Expected result: Result file is created, and results are close to correct. \n"

rm -f output.res
$PROGRAM -Va=2 -delay=0 blood.dat quick.dft 999 output.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parmatch -abs=0.001 -rel=3 -tacnames=y -parunits=y -par=1-2 correct_quick.res output.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parmatch -tacnames=y -parunits=y -par=3-4 correct_quick.res output.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 1.0.3 \n"
printf " Test case 1: Same as 1.0.0, but using option -pH2O=1 and delay=0. \n"
printf " Test case 2: Extra files saved with option -svg, -fit, -input, and -resid. \n"
printf " Expected result: Extra files are created, and results are close to correct. \n"
printf " Expected result 2: Parameter SD and CL are not saved by default. \n\n"

rm -f output.res fit.svg fit.dft resid.dft dcinput.dft
$PROGRAM -pH2O=1 -delay=0 -svg=fit.svg -fit=fit.dft -resid=resid.dft -input=dcinput.dft blood.dat quick.dft 999 output.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parmatch -abs=0.001 -rel=3 -tacnames=y -parunits=y -par=1,3 correct_quick.res output.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parmatch -tacnames=y -parunits=y -par=2,4 -cl=y -sd=y correct_quick.res output.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
if [ ! -f fit.svg ] || [ ! -f fit.dft ] || [ ! -f resid.dft ] || [ ! -f dcinput.dft ]; then
  printf "\n Error: extra files were not created. \n"
  printf "Failed!\n" ; exit 1 ;
fi
printf "\n Testing also that extra TAC files can be plotted... \n"
tac2svg -mt="Fitted TTAC" -legend=y fit103.svg -l fit.dft 
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tac2svg -mt="Residual TAC" -legend=y resid103.svg -l resid.dft 
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tac2svg -mt="Delay-corrected BTAC" -legend=y dcinput103.svg -l dcinput.dft 
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 1.0.4 \n"
printf " Test case: Using options -SD=y and CL=y. \n"
printf " Expected result: Parameter SD and CL are calculated and saved. \n\n"

rm -f output.res
$PROGRAM -Va=2 -delay=0 -sd=y -cl=y blood.dat quick.dft 999 output.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parmatch -abs=0.001 -rel=3 -tacnames=y -parunits=y -par=1-4 correct_quick_bs.res output.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n-----------------------------------------------------------------\n"

printf "\n 1.1.0 \n"
printf " Test case: Using PMOD files. \n"
printf " Expected result: Results are close to correct. \n\n"

rm -f output.res
$PROGRAM blood.bld quick.tac 999 output.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parmatch -abs=0.001 -rel=3 -tacnames=y -parunits=y -par=1-4 correct_quick.res output.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n===================================================================\n"
printf "   Basic usability tests passed! \n"
printf "   Now testing error handling. \n"
printf "===================================================================\n"

printf "\n 2.0.0 \n"
printf " Test case: Unknown option. \n"
printf " Expected result: Error. \n\n"

rm -f stdout.txt
$PROGRAM -nonexistingoption blood.dat quick.dft 999 output.res &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "invalid option '-nonexistingoption'."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 2.0.1 \n"
printf " Test case: Missing argument. \n"
printf " Expected result: Error. \n\n"

rm -f stdout.txt
$PROGRAM  blood.dat  999 output.res &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

rm -f stdout.txt
$PROGRAM  blood.dat quick.dft 999  &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

rm -f stdout.txt output.res
$PROGRAM blood.dat quick.dft output.res &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 2.1.0 \n"
printf " Test case: Nonexisting data file. \n"
printf " Expected result: Error. \n\n"

rm -f stdout.txt
$PROGRAM nonexistingfile.dat quick.dft 999 output.res &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "cannot read 'nonexistingfile.dat': cannot open file"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

rm -f stdout.txt
$PROGRAM blood.dat nonexistingfile.dat 999 output.res &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "cannot read 'nonexistingfile.dat': cannot open file"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 2.1.1 \n"
printf " Test case: Forgotten fit duration. \n"
printf " Expected result: Error. \n\n"

rm -f stdout.txt
$PROGRAM blood.dat quick.dft  output.res &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "invalid fit time 'output.res'."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 2.2.0 \n"
printf " Test case: Invalid data file contents. \n"
printf " Expected result: Error. \n\n"

rm -f badfile.dat
printf "# badfile\n" > badfile.dat
printf "ugh banzai\n" >> badfile.dat
printf "2 meh\n" >> badfile.dat

rm -f stdout.txt
$PROGRAM badfile.dat quick.dft 999 output.res &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "cannot read 'badfile.dat': wrong format"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

rm -f stdout.txt
$PROGRAM blood.dat badfile.dat 999 output.res &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "cannot read 'badfile.dat': wrong format"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"




printf "\n 2.3.0 \n"
printf " Test case: Input file contains more than one TAC. \n"
printf " Expected result 1: Warning. \n"
printf " Expected result 2: First TAC is used. \n\n"

taccalc blood.bld . 0.001 temp.bld
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacadd -ovr blood_2_3_0.bld blood.bld
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacadd blood_2_3_0.bld temp.bld
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

rm -f stdout.txt output.res
$PROGRAM -delay=0 -Va=2 blood_2_3_0.bld quick.tac 999 output.res &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Warning "using only first TAC in blood_2_3_0.bld"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"
parmatch -abs=0.001 -rel=3 -par=1-4 correct_quick.res output.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 2.4.0 \n"
printf " Test case: Data file contains NaN. \n"
printf " Expected result: Error. \n\n"

rm -f stdout.txt
$PROGRAM blood_na.bld quick.dft 999 output.res &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "missing sample(s)"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

rm -f stdout.txt
$PROGRAM blood.bld quick_na.dft 999 output.res &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "missing sample(s) in quick_na.dft"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n-----------------------------------------------------------------\n"


printf "\n 2.5.0 \n"
printf " Test case: Data file contains only 3 samples. \n"
printf " Expected result: Error. \n\n"

rm -f stdout.txt
$PROGRAM blood_3.bld quick.dft 999 output.res &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "blood_3.bld contains too few samples"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


rm -f stdout.txt
$PROGRAM blood.bld quick_3.dft 999 output.res &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "too few samples in specified fit duration."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n-----------------------------------------------------------------\n"


printf "\n 2.6.0 \n"
printf " Test case: 3 sec gap in beginning of TTAC. \n"
printf " Expected result: not even an error. \n\n"
rm -f stdout.txt output.res
tactime quick.dft +3 temp.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
$PROGRAM -Va=2 -pH2O=1 blood.bld temp.dft 999 output.res &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Warning
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 2.6.1 \n"
printf " Test case: 6 sec gap in beginning of TTAC. \n"
printf " Expected result: warning. \n\n"
rm -f stdout.txt output.res
tactime quick.dft +6 temp.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacsety temp.dft 1 1 1.0
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
$PROGRAM -Va=2 -pH2O=1 blood.bld temp.dft 999 output.res &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Warning "TACs should start at time zero."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 2.6.2 \n"
printf " Test case: 1 min gap in beginning of TTAC. \n"
printf " Expected result: error, if first concentration is not zero. \n\n"
rm -f stdout.txt output.res
tactime quick.dft +60 temp.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacsety temp.dft 1 1 1.0
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
$PROGRAM -Va=2 -pH2O=1 blood.bld temp.dft 999 output.res &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error" "TACs must start at time zero."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n-----------------------------------------------------------------\n"

printf "\n 2.7.0 \n"
printf " Test case: Short gap between frames. \n"
printf " Expected result: almost the same results as wo the gap. \n\n"

rm -f flow1.res
$PROGRAM -Va=2 blood.bld normal.dft 999 flow1.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

rm -f stdout.txt output.res
$PROGRAM -Va=2 blood.bld smallgap.dft 999 flow2.res &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parmatch -abs=0.01 -rel=1 -par=1-4 flow1.res flow2.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 2.7.1 \n"
printf " Test case: Longer gap between frames. \n"
printf " Expected result: about the same results as wo the gap. \n\n"

rm -f stdout.txt output.res
$PROGRAM -Va=2 blood.bld biggap.dft 999 flow3.res &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parmatch -abs=0.01 -rel=3 -par=1-4 flow1.res flow3.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 2.8.0 \n"
printf " Test case: Short overlap between frames. \n"
printf " Expected result: almost the same results as wo the overlap. \n\n"

rm -f stdout.txt output.res
$PROGRAM -Va=2 blood.bld smalloverlap.dft 999 flow4.res &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parmatch -abs=0.01 -rel=1 -par=1-4 flow1.res flow4.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 2.8.1 \n"
printf " Test case: Longer overlap between frames. \n"
printf " Expected result: almost the same results as wo the overlap. \n\n"

rm -f stdout.txt output.res
$PROGRAM -Va=2 blood.bld bigoverlap.dft 999 flow5.res &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parmatch -abs=0.01 -rel=3 -par=1-4 flow1.res flow5.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n-----------------------------------------------------------------\n"


printf "\n 2.9.0 \n"
printf " Test case: User-specified fit time is zero. \n"
printf " Expected result: using all data, correct results. \n\n"

rm -f stdout.txt output.res
$PROGRAM -Va=2 -delay=0 blood.bld quick.dft 0 output.res &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parmatch -abs=0.01 -rel=3 -par=1-4 correct_quick.res output.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval output.res "Data range" "0 - 330 sec"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 2.9.1 \n"
printf " Test case: User-specified fit time is letter o. \n"
printf " Expected result: error. \n\n"

rm -f stdout.txt output.res
$PROGRAM -Va=2 blood.bld quick.dft O output.res &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 2.9.2 \n"
printf " Test case: User-specified fit time is in minutes. \n"
printf " Expected result: error, if too short time. \n\n"

rm -f stdout.txt output.res
$PROGRAM -Va=2 blood.bld quick.dft 4 output.res &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 2.9.3 \n"
printf " Test case: User-specified fit time limits fit time. \n"
printf " Expected result: correct results. \n\n"

rm -f stdout.txt output.res
$PROGRAM -Va=2 -delay=0 blood.bld quick.dft 270 output.res &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parmatch -abs=0.01 -rel=5 -par=1-4 correct_quick.res output.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval output.res "Data range" "0 - 270 sec"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n===================================================================\n"
printf "   Data units \n"
printf "===================================================================\n"

printf "\n 3.0.0 \n"
printf " Test case: Data has different concentration units. \n"
printf " Expected result: automatic conversion, correct result. \n\n"

rm -f quick_bq.dft blood_bq.bld
cp quick.dft quick_bq.dft
cp blood.bld blood_bq.bld
tacunit -yconv=Bq/cc quick_bq.dft blood_bq.bld
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

rm -f output.res
$PROGRAM -svg=fit300a.svg blood_bq.bld quick.dft 999 output.res &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parmatch -abs=0.001 -rel=3 -parunits=y -par=1-4 correct_quick.res output.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

rm -f output.res
$PROGRAM -svg=fit300b.svg blood.bld quick_bq.dft 999 output.res &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parmatch -abs=0.001 -rel=3 -parunits=y -par=1-4 correct_quick.res output.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 3.1.0 \n"
printf " Test case: Data has different time units. \n"
printf " Expected result: automatic conversion, correct result. \n\n"

rm -f quick_min.dft blood_min.bld
cp quick.dft quick_min.dft
cp blood.bld blood_min.bld
tacunit -xconv=min quick_min.dft blood_min.bld
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

rm -f output.res stdout.txt
$PROGRAM -svg=fit310a.svg blood_min.bld quick.dft 999 output.res &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parmatch -abs=0.001 -rel=3 -parunits=y -par=1-4 correct_quick.res output.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

rm -f output.res stdout.txt
$PROGRAM -svg=fit310b.svg blood.bld quick_min.dft 999 output.res &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parmatch -abs=0.001 -rel=3 -parunits=y -par=1-4 correct_quick.res output.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"




printf "\n 3.2.0 \n"
printf " Test case: Data has missing concentration units. \n"
printf " Expected result: correct result. \n\n"

rm -f quick_uc.dft blood_uc.bld
cp quick.dft quick_uc.dft
cp blood.bld blood_uc.bld
tacunit -yset=unknown quick_uc.dft blood_uc.bld
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

rm -f output.res stdout.txt
$PROGRAM -svg=fit320a.svg blood_uc.bld quick.dft 999 output.res &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parmatch -abs=0.001 -rel=3 -parunits=y -par=1-4 correct_quick.res output.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Note" "check the units of input and tissue data."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

rm -f output.res stdout.txt
$PROGRAM -svg=fit320b.svg blood.bld quick_uc.dft 999 output.res &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parmatch -abs=0.001 -rel=3 -parunits=y -par=1-4 correct_quick.res output.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Note" "check the units of input and tissue data."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

rm -f output.res stdout.txt
$PROGRAM -svg=fit320c.svg blood_uc.bld quick_uc.dft 999 output.res &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parmatch -abs=0.001 -rel=3 -parunits=y -par=1-4 correct_quick.res output.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Note" "check the units of input and tissue data."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 3.3.0 \n"
printf " Test case: Data has missing time units. \n"
printf " Expected result: correct result. \n\n"

rm -f quick_ut.dft blood_ut.bld
cp quick.dft quick_ut.dft
cp blood.bld blood_ut.bld
tacunit -xset=unknown quick_ut.dft blood_ut.bld
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Note" "check the units of input and tissue data."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

rm -f output.res stdout.txt
$PROGRAM -svg=fit330a.svg blood_ut.bld quick.dft 999 output.res &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parmatch -abs=0.001 -rel=3 -parunits=y -par=1-4 correct_quick.res output.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

rm -f output.res stdout.txt
$PROGRAM -svg=fit330b.svg blood.bld quick_ut.dft 999 output.res &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parmatch -abs=0.001 -rel=3 -parunits=y -par=1-4 correct_quick.res output.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 3.3.1 \n"
printf " Test case: Neither data file contains time units, but data is in sec.\n"
printf " Expected result: error, if reasonable fit time is set. \n\n"

rm -f output.res stdout.txt
$PROGRAM blood_ut.bld quick_ut.dft 330 output.res &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Warning"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 3.3.2 \n"
printf " Test case: Neither data file contains time units, but data is in min.\n"
printf " Expected result: correct results. \n\n"

cp quick_min.dft quick_ut.dft
cp blood_min.bld blood_ut.bld
tacunit -xset=unknown quick_ut.dft blood_ut.bld

rm -f output.res stdout.txt
$PROGRAM -svg=fit332.svg -Va=2 blood_ut.bld quick_ut.dft 999 output.res &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parmatch -abs=0.001 -rel=3 -parunits=y -par=1-4 correct_quick.res output.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n===================================================================\n"
printf "   Constraints \n"
printf "===================================================================\n"

printf "\n 4.0.0 \n"
printf " Test case: constraint file is created with option -lim=fname.\n"
printf " Expected result: File is created with correct contents. \n\n"

rm -f output.par
$PROGRAM -lim=output.par
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

iftisval output.par K1_lower 0
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
iftisval output.par K1_upper 600
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
iftisval output.par K1k2_lower 0.3
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
iftisval output.par K1k2_upper 1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
iftisval output.par Va_lower 0
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
iftisval output.par Va_upper 25
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
iftisval output.par Delay_lower -60
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
iftisval output.par Delay_upper 60
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 4.0.1 \n"
printf " Test case: existing constraint file and option -lim=fname.\n"
printf " Expected result: Error. \n\n"

rm -f stdout.txt
$PROGRAM -lim=output.par &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error" "parameter constraint file output.par exists."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 4.0.2 \n"
printf " Test case: trying to use nonexisting constraint file.\n"
printf " Expected result: Error. \n\n"

rm -f stdout.txt
$PROGRAM -lim=nonexisting.par blood.bld quick.dft 999 output.res &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error in reading 'nonexisting.par'" "cannot open file"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 4.0.3 \n"
printf " Test case: trying to use constraint file with bad contents.\n"
printf " Expected result: Error. \n\n"

rm -f stdout.txt
$PROGRAM -lim=badfile.dat blood.bld quick.tac 999 output.res &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error in reading 'badfile.dat'" "file contains no data"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 4.1.0 \n"
printf " Test case: set constraints for one parameter with -lim.\n"
printf " Expected result: Limits are set correctly. \n\n"

printf "K1_lower := 6\n"  > limit.par
printf "K1_upper := 60\n" >> limit.par

rm -f stdout.txt
$PROGRAM -d -lim=limit.par blood.bld quick.tac 0 output.res &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "def_pmin[1]" "0.001"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "def_pmax[1]" "0.01"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 4.1.1 \n"
printf " Test case: set constraints for all parameters with -lim.\n"
printf " Expected result: Limits are set correctly. \n\n"

printf "K1k2_lower := 0.5\n" >> limit.par
printf "K1k2_upper := 2.0\n" >> limit.par
printf "Va_lower := 2\n"  >> limit.par
printf "Va_upper := 2\n"  >> limit.par
printf "Delay_lower := -5\n"  >> limit.par
printf "Delay_upper := +5\n"  >> limit.par

rm -f stdout.txt
$PROGRAM -d -lim=limit.par blood.bld quick.dft 0 output.res &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "def_pmin[2]" "0.5"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "def_pmax[2]" "2"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "def_pmin[3]" "0.02"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "def_pmax[3]" "0.02"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "def_pmin[4]" "-5"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "def_pmax[4]" "5"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 4.2.0 \n"
printf " Test case: option -Va=0\n"
printf " Expected result: Limits are set correctly. \n\n"

rm -f stdout.txt
$PROGRAM -d -Va=0 blood.bld quick.tac 0 output.res &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "def_pmin[3]" "0"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "def_pmax[3]" "0"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 4.2.1 \n"
printf " Test case: option -Va=4.5\n"
printf " Expected result: Limits are set correctly. \n\n"

rm -f stdout.txt
$PROGRAM -d -Va=4.5 blood.bld quick.dft 0 output.res &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "def_pmin[3]" "0.045"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "def_pmax[3]" "0.045"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 4.2.2 \n"
printf " Test case: option -Va=0.045 by mistake\n"
printf " Expected result: Warning. \n\n"

rm -f stdout.txt
$PROGRAM -Va=0.045 blood.bld quick.dft 0 output.res &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Warning "Va was set to 0.045%"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 4.3.0 \n"
printf " Test case: constraints set with option and parameter file\n"
printf " Expected result: options outstrip constrains from parameter file. \n\n"

printf "K1_lower := 6\n"  > limit2.par
printf "K1_upper := 60\n" >> limit2.par
printf "K1k2_lower := 0.5\n" >> limit2.par
printf "K1k2_upper := 2.0\n" >> limit2.par
printf "Va_lower := 1\n"  >> limit2.par
printf "Va_upper := 10\n"  >> limit2.par
printf "Delay_lower := -30\n"  >> limit2.par
printf "Delay_upper := +50\n"  >> limit2.par

rm -f stdout.txt
$PROGRAM -d -Va=2 -delay=0 -pH2O=1 -lim=limit2.par blood.bld quick.tac 0 output.res &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "def_pmin[1]" "0.001"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "def_pmax[1]" "0.01"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "def_pmin[2]" "1"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "def_pmax[2]" "1"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "def_pmin[3]" "0.02"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "def_pmax[3]" "0.02"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "def_pmin[4]" "0"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "def_pmax[4]" "0"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n===================================================================\n"
printf "   Weights \n"
printf "===================================================================\n"

printf "\n 5.0.0 \n"
printf " Test case: by default weights are 1\n"
printf " Expected result: correct weights are printed in test mode. \n\n"

rm -f stdout.txt
$PROGRAM -d -Va=2 blood.bld quick.tac 0 output.res &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt common_data_weights "1.000, 1.000, 1.000, 1.000, 1.000, 1.000, 1.000, 1.000, 1.000, 1.000, 1.000, 1.000, 1.000, 1.000, 1.000"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 5.0.1 \n"
printf " Test case: Datafile contains weights.\n"
printf " Expected result: correct weights are printed in test mode. \n\n"

cp -f quick.tac temp.tac
tacweigh -wf temp.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

rm -f stdout.txt
$PROGRAM -d -Va=2 blood.bld temp.tac 0 output.res &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt common_data_weights "0.455, 0.227, 0.227, 0.455, 0.455, 0.455, 0.455, 0.682, 0.682, 1.364, 1.364, 1.364, 1.364, 2.727, 2.727"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n===================================================================\n"
printf "   Data with several TTACs and delay \n"
printf "===================================================================\n"

printf "\n 6.0.0 \n"
printf " Test case: default fitting.\n"
printf " Expected result: results are close to correct. \n\n"

rm -f output.res
$PROGRAM -svg=fit600.svg blood.bld basicsim.tac 0 output.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parmatch -abs=0.001 -rel=2 -parunits=y -par=1 correct_basicsim.res output.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parmatch -abs=0.02 -rel=5 -parunits=y -par=2 correct_basicsim.res output.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parmatch -abs=0.1 -rel=10 -parunits=y -par=3 correct_basicsim.res output.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parmatch -abs=0.5 -rel=10 -parunits=y -par=4 correct_basicsim.res output.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n===================================================================\n"
printf "   Other options \n"
printf "===================================================================\n"

printf "\n 7.0.0 \n"
printf " Test case: Option -ml.\n"
printf " Expected result: results are written in correct units. \n\n"

rm -f output.res
$PROGRAM -ml -delay=0 blood.bld quick.tac 0 output.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parmatch -abs=0.001 -rel=3 -parunits=y -par=1-4 correct_quick_ml.res output.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 7.0.1 \n"
printf " Test case: Option -dl.\n"
printf " Expected result: results are written in correct units. \n\n"

rm -f output.res
$PROGRAM -dl -delay=0 blood.bld quick.tac 0 output.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parmatch -abs=0.001 -rel=3 -parunits=y -par=1-4 correct_quick.res output.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"






printf "\n 7.1.0 \n"
printf " Test case: Option -k2 with -fpt.\n"
printf " Expected result: Correct k2 values are saved in results. \n\n"

rm -f output.res
$PROGRAM -k2 -fpt -delay=0 blood.bld quick.tac 0 output.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parformat -f=ift output.res output.ift
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.001 output.ift k2 0.2083
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.01 output.ift Flow 20.833
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 7.1.1 \n"
printf " Test case: Option -k2 without -fpt.\n"
printf " Expected result: Correct k2 values are saved in results. \n\n"

rm -f output.res
$PROGRAM -k2 -delay=0 blood.bld quick.tac 0 output.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parformat -f=ift output.res output.ift
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.001 output.ift k2 0.2083
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.01 output.ift Flow 20.417
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"








printf "\n===================================================================\n"
printf "   All passed! \n"
printf "===================================================================\n"
exit 0

