#!/bin/bash
#: Title      : test_fitk5
#: Date       : 2019-08-26
#: Author     : "Vesa Oikonen" <vesa.oikonen@utu.fi>
#: Options    : None

# Set the name of executable to test
if [ -n "${OS+1}" ] && [ "$OS" = "Windows_NT" ]; then
EXT=.exe;
else
EXT=;
fi

PROGRAM=../../fitk5$EXT;
if [ ! -f $PROGRAM ]; then
  printf "Failed: executable does not exist.\n"
  exit 1
fi

printf "=====================================================================\n"
printf "creating test data for %s\n" $PROGRAM
printf "=====================================================================\n"


printf "\n verifying that required data exists \n"


if [ ! -f plasma.dat ] || [ ! -f frames.dat ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
if [ ! -f frames_few.dat ] || [ ! -f frames_long.dat ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
if [ ! -f paral.lim ] || [ ! -f serie.lim ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
if [ ! -f correct_pk5.res ] || [ ! -f correct_sk5.res ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
if [ ! -f correct_1_2_0.res ] || [ ! -f correct_1_2_0.res ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi



printf "\n making data for testing \n"


if [ ! -f plasma.bld ] || [ ! -f blood.bld ]; then
  rm -f plasma_na.bld blood_na.bld
  rm -f plasma_3.bld blood_3.bld
  rm -f sim_pk5.tac pet_pk5.tac sim_sk5.tac pet_sk5.tac
  printf "\n making plasma.bld \n"
  tacformat -f=PMOD plasma.dat plasma.bld
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacunit -xset=min -yset=kBq/ml plasma.bld
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacren plasma.bld 0 plasma
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  printf "\n making blood.bld \n"
  p2blood FTHA plasma.bld 0.42 blood.bld
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacren blood.bld 0 blood
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tac2svg -mt="Input" -legend=y input.svg -l plasma.bld blood.bld 
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f plasma_na.bld ] || [ ! -f blood_na.bld ]; then
  printf "\n Make input with NaN \n"
  cp -f plasma.bld plasma_na.bld
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacsety plasma_na.bld 1 20 nan
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  cp -f blood.bld blood_na.bld
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacsety blood_na.bld 1 10 nan
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f plasma_3.bld ] || [ ! -f blood_3.bld ]; then
  printf "\n Make input with only 3 samples \n"
  interpol -x=1.2,8,28 plasma.bld plasma_3.bld
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  interpol -x=1.2,8,28 blood.bld blood_3.bld
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi


if [ ! -f sim_pk5.tac ] || [ ! -f pet_pk5.tac ]; then
  rm -f pet_pk5.tac pet_pk5_few.tac pet_pk5_long.tac
  printf "\n simulate parallel cm tissue TACs \n"

  p2t_v3c --silent -nosub -parallel plasma.bld blood.bld 1 1.25 0.7 0.5 0.00 0  0 5 100 temp.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacren --silent temp.tac 1 ref_1
  tacadd --silent -ovr sim_pk5.tac temp.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

  p2t_v3c --silent -nosub -parallel plasma.bld blood.bld 1 1.25 0.2 0.2 0.00 0  0 5 100 temp.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacren --silent temp.tac 1 ref_2
  tacadd --silent sim_pk5.tac temp.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

  p2t_v3c --silent -nosub -parallel plasma.bld blood.bld 1 1.25 0.7 0.5 0.10 0  0 5 100 temp.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacren --silent temp.tac 1 tis_1
  tacadd --silent sim_pk5.tac temp.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

  p2t_v3c --silent -nosub -parallel plasma.bld blood.bld 1 1.25 1.0 1.0 0.10 0  0 5 100 temp.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacren --silent temp.tac 1 tis_2
  tacadd --silent sim_pk5.tac temp.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

  p2t_v3c --silent -nosub -parallel plasma.bld blood.bld 1 1.25 0.7 1.4 0.10 0  0 5 100 temp.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacren --silent temp.tac 1 tis_3
  tacadd --silent sim_pk5.tac temp.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

  p2t_v3c --silent -nosub -parallel plasma.bld blood.bld 1 1.25 0.7 1.4 0.20 0  0 5 100 temp.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacren --silent temp.tac 1 tis_4
  tacadd --silent sim_pk5.tac temp.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

  p2t_v3c --silent -nosub -parallel plasma.bld blood.bld 1 1.25 0.7 1.4 0.05 0  0 5 100 temp.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacren --silent temp.tac 1 tis_5
  tacadd --silent sim_pk5.tac temp.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

fi

if [ ! -f pet_pk5.tac ] || [ ! -f pet_pk5_few.tac ] || [ ! -f pet_pk5_long.tac ]; then
  printf "\n simulate PET frames \n"
  rm -f pet_pk5_t2.tac pet_pk5_t2_long.tac pet_pk5_na.tac

  simframe --silent sim_pk5.tac frames.dat pet_pk5.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacstudy pet_pk5.tac paral1
  tac2svg -legend=y -mt="Simulated parallel CM TTACs" pet_pk5.svg pet_pk5.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

  simframe --silent sim_pk5.tac frames_few.dat pet_pk5_few.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacstudy pet_pk5_few.tac paral2

  simframe --silent sim_pk5.tac frames_long.dat pet_pk5_long.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacstudy pet_pk5_long.tac paral3
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

fi

if [ ! -f pet_pk5_t2.tac ] || [ ! -f pet_pk5_t2_long.tac ]; then
  printf "\n extract just one TAC for quick testing \n"
  tacadd -ovr pet_pk5_t2.tac pet_pk5.tac tis_2
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacadd -ovr pet_pk5_t2_long.tac pet_pk5_long.tac tis_2
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f pet_pk5_na.tac ]; then
  printf "\n Make TAC with NaN \n"
  cp -f pet_pk5.tac pet_pk5_na.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacsety pet_pk5_na.tac 1 12 nan
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi


if [ ! -f sim_sk5.tac ] || [ ! -f pet_sk5.tac ]; then
  rm -f pet_sk5.tac
  printf "\n simulate in series cm tissue TACs \n"

  p2t_v3c --silent -nosub plasma.bld blood.bld 1 1.25 0.35 0.25  0.00 0  0 5 100 temp.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacren --silent temp.tac 1 ref_1
  tacadd --silent -ovr sim_sk5.tac temp.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

  p2t_v3c --silent -nosub plasma.bld blood.bld 1 1.25 0.35 0.70  0.00 0  0 5 100 temp.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacren --silent temp.tac 1 ref_2
  tacadd --silent sim_sk5.tac temp.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

  p2t_v3c --silent -nosub plasma.bld blood.bld 1 1.25 0.35 0.25  0.10 0  0 5 100 temp.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacren --silent temp.tac 1 tis_1
  tacadd --silent sim_sk5.tac temp.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

  p2t_v3c --silent -nosub plasma.bld blood.bld 1 1.25 0.35 0.25  0.20 0  0 5 100 temp.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacren --silent temp.tac 1 tis_2
  tacadd --silent sim_sk5.tac temp.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

  p2t_v3c --silent -nosub plasma.bld blood.bld 1 1.25 0.35 0.70  0.10 0  0 5 100 temp.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacren --silent temp.tac 1 tis_3
  tacadd --silent sim_sk5.tac temp.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

  p2t_v3c --silent -nosub plasma.bld blood.bld 1 1.25 0.35 0.70  0.20 0  0 5 100 temp.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacren --silent temp.tac 1 tis_4
  tacadd --silent sim_sk5.tac temp.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

  p2t_v3c --silent -nosub plasma.bld blood.bld 1 1.25 0.35 0.70  0.05 0  0 5 100 temp.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacren --silent temp.tac 1 tis_5
  tacadd --silent sim_sk5.tac temp.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

fi


if [ ! -f pet_sk5.tac ]; then
  printf "\n simulate PET frames \n"

  simframe --silent sim_sk5.tac frames.dat pet_sk5.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacstudy pet_sk5.tac serie1
  tac2svg -legend=y -mt="Simulated CM in series TTACs" pet_sk5.svg pet_sk5.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

fi



printf "\n\n"
printf "=====================================================================\n"
printf "testing %s\n" $PROGRAM
printf "=====================================================================\n"


printf "\n 1.0.0 \n"
printf " Test case 1: Fit simulated TACs with realistic range of parameters. \n"
printf " Test case 2: Parallel model. \n"
printf " Test case 3: Fixed Vb. \n"
printf " Expected result 1: Result file is created, and results are close to correct \n"
printf " Expected result 2: SVG plot is created. \n"
printf " Expected result 3: Fitted TACs should be close to original TACs.\n"
printf " Expected result 4: Saved fitted TAC can be used for plotting, too. \n\n"

rm -f output_1_0_0.res fit_1_0_0.tac fit_1_0_0.svg


$PROGRAM -Vb=5 -lim=paral.lim -fit=fit_1_0_0.tac -svg=fit_1_0_0.svg plasma.bld blood.bld pet_pk5.tac 9999 output_1_0_0.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

resmatch -abs=0.001 -header=y -param=y -regions=y -res=1,5,6,7,8 correct_pk5.res output_1_0_0.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
resmatch -abs=0.01 -res=2-4 correct_pk5.res output_1_0_0.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

tacmatch -abs=0.1 -rel=2 pet_pk5.tac fit_1_0_0.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

tac2svg -legend=y fit_1_0_0b.svg -s pet_pk5.tac -l fit_1_0_0.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 1.1.0 \n"
printf " Test case 1: Fit simulated TACs with realistic range of parameters. \n"
printf " Test case 2: Series model. \n"
printf " Test case 3: Fixed Vb. \n"
printf " Expected result 1: Result file is created, and results are close to correct \n"
printf " Expected result 2: SVG plot is created. \n"
printf " Expected result 3: Fitted TACs should be close to original TACs.\n"
printf " Expected result 4: Saved fitted TAC can be used for plotting, too. \n\n"

rm -f output_1_1_0.res fit_1_1_0.tac fit_1_1_0.svg


$PROGRAM -model=series -Vb=5 -lim=serie.lim -fit=fit_1_1_0.tac -svg=fit_1_1_0.svg plasma.bld blood.bld pet_sk5.tac 9999 output_1_1_0.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

resmatch -abs=0.001 -header=y -param=y -regions=y -res=1,5,6,7,8 correct_sk5.res output_1_1_0.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
resmatch -abs=0.01 -res=2-4 correct_sk5.res output_1_1_0.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

tacmatch -abs=0.1 -rel=2 pet_sk5.tac fit_1_1_0.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

tac2svg -legend=y fit_1_1_0b.svg -s pet_sk5.tac -l fit_1_1_0.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 1.1.1 \n"
printf " Test case 1: Constrain K1/k2 based on previously fitted median. \n"
printf " Test case 2: Series model. \n"
printf " Test case 3: Not fixed Vb. \n"
printf " Expected result: Results are close to correct \n\n"

rm -f output_1_1_1.res fit_1_1_1.svg
$PROGRAM -fk1k2=output_1_1_0.res -model=series -lim=serie.lim -svg=fit_1_1_1.svg plasma.bld blood.bld pet_sk5.tac 9999 output_1_1_1.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

parmatch -par=1-6 -abs=0.001 -rel=2 -tacnames=y -parunits=y correct_sk5.res output_1_1_1.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n --------------------------------------------------------------- \n"



printf "\n 1.2.0 \n"
printf " Test case: Bootstrapping with options -sd=y and -cl=y. \n"
printf " Expected result: Result file contains sd and cl for parameters \n\n"

rm -f output.res
$PROGRAM -sd=y -cl=y -Vb=5 plasma.bld blood.bld pet_pk5_t2_long.tac 90 output.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parmatch -abs=0.01 -rel=2 -tacnames=y -parunits=y -par=1-6 -sd=y -cl=y correct_1_2_0.res output.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n --------------------------------------------------------------- \n"


printf "\n 1.3.0 \n"
printf " Test case: Fit end time is shorter than duration of data. \n"
printf " Expected result: Given fit time is applied. \n\n"

rm -f output.res
$PROGRAM -lim=paral.lim plasma.bld blood.bld pet_pk5_t2_long.tac 70 output.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval output.res "Data range" "0 - 60 min"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n===================================================================\n"
printf "   Basic usability tests passed! \n"
printf "   Now testing error handling. \n"
printf "===================================================================\n"


printf "\n 2.0.0 \n"
printf " Test case: Unknown option. \n"
printf " Expected result: Error. \n\n"

rm -f stdout.txt
$PROGRAM -nonexistingoption plasma.bld blood.bld pet_pk5_t2_long.tac 30 output.res &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "invalid option '-nonexistingoption'."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"




printf "\n 2.0.1 \n"
printf " Test case: Missing filename. \n"
printf " Expected result: Error. \n\n"

rm -f stdout.txt
$PROGRAM  blood.bld pet_pk5_t2_long.tac 30 output.res &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

rm -f stdout.txt
$PROGRAM  plasma.bld pet_pk5_t2_long.tac 30 output.res &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

rm -f stdout.txt
$PROGRAM plasma.bld blood.bld  30 output.res &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

rm -f stdout.txt
$PROGRAM plasma.bld blood.bld pet_pk5_t2_long.tac 30  &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n --------------------------------------------------------------- \n"


printf "\n 2.1.0 \n"
printf " Test case: Nonexisting data file. \n"
printf " Expected result: Error. \n\n"

rm -f stdout.txt
$PROGRAM nonexistingfile.dat blood.bld pet_pk5.tac 30 output.res &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "cannot read 'nonexistingfile.dat': cannot open file"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

rm -f stdout.txt
$PROGRAM plasma.bld nonexistingfile.dat pet_pk5.tac 30 output.res &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "cannot read 'nonexistingfile.dat': cannot open file"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

rm -f stdout.txt
$PROGRAM plasma.bld blood.bld nonexistingfile.dat 30 output.res &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "cannot read 'nonexistingfile.dat': cannot open file"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 2.1.1 \n"
printf " Test case: Nonexisting blood file. \n"
printf " Expected result: No error if Vb=0. \n\n"

rm -f stdout.txt
$PROGRAM -Vb=0 plasma.bld nonexistingfile.dat pet_pk5_t2_long.tac 30 output.res &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n --------------------------------------------------------------- \n"



printf "\n 2.1.2 \n"
printf " Test case: Nonexisting ref file. \n"
printf " Expected result: Error. \n\n"

rm -f stdout.txt
$PROGRAM -R=nonexistingfile.dat plasma.bld blood.bld pet_pk5.tac 30 output.res &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error in reading 'nonexistingfile.dat'" "cannot find region"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 2.1.3 \n"
printf " Test case: Given reference region name does not exist. \n"
printf " Expected result: Error. \n\n"

rm -f stdout.txt
$PROGRAM -r=stupid plasma.bld blood.bld pet_pk5.tac 30 output.res &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error in reading 'stupid'" "cannot find region"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 2.1.4 \n"
printf " Test case: Given reference region nr does not exist. \n"
printf " Expected result: Error. \n\n"

rm -f stdout.txt
$PROGRAM -R=16 plasma.bld blood.bld pet_pk5.tac 30 output.res &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error in reading '16'" "cannot find region"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

rm -f stdout.txt
$PROGRAM -R=0 plasma.bld blood.bld pet_pk5.tac 30 output.res &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error in reading '0'" "cannot find region"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n --------------------------------------------------------------- \n"


printf "\n 2.2.0 \n"
printf " Test case: Invalid data file contents. \n"
printf " Expected result: Error. \n\n"

rm -f badfile.dat
printf "# badfile\n" > badfile.dat
printf "ugh banzai\n" >> badfile.dat
printf "2 meh\n" >> badfile.dat

rm -f stdout.txt
$PROGRAM badfile.dat blood.bld pet_pk5.tac 30 output.res &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "cannot read 'badfile.dat': wrong format"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

rm -f stdout.txt
$PROGRAM plasma.bld badfile.dat pet_pk5.tac 30 output.res &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "cannot read 'badfile.dat': wrong format"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

rm -f stdout.txt
$PROGRAM plasma.bld blood.bld badfile.dat 30 output.res &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "cannot read 'badfile.dat': wrong format"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 2.2.1 \n"
printf " Test case: Invalid blood file contents. \n"
printf " Expected result: No error if Vb=0. \n\n"

rm -f stdout.txt
$PROGRAM -Vb=0 plasma.bld badfile.dat pet_pk5_t2_long.tac 30 output.res &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 2.2.2 \n"
printf " Test case: Invalid ref file contents. \n"
printf " Expected result: Error. \n\n"

rm -f stdout.txt
$PROGRAM -r=badfile.dat plasma.bld blood.bld pet_pk5.tac 30 output.res &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error in reading 'badfile.dat'"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n --------------------------------------------------------------- \n"

printf "\n 2.3.0 \n"
printf " Test case: Input file contains more than one TAC. \n"
printf " Expected result 1: Warning. \n"
printf " Expected result 2: First TAC is used. \n\n"

taccalc plasma.bld . 0.001 temp.bld
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacadd -ovr plasma_2_3_0.bld plasma.bld
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacadd plasma_2_3_0.bld temp.bld
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

taccalc blood.bld . 0.001 temp.bld
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacadd -ovr blood_2_3_0.bld blood.bld
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacadd blood_2_3_0.bld temp.bld
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

rm -f stdout.txt output.res
$PROGRAM -lim=paral.lim plasma_2_3_0.bld blood_2_3_0.bld pet_pk5_t2_long.tac 90 output.res &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Warning "using only first TAC in plasma_2_3_0.bld"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"
iftisval stdout.txt Warning "using only first TAC in blood_2_3_0.bld"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"
parmatch -abs=0.01 -rel=1 -par=1-5 -sd=n -cl=n correct_1_2_0.res output.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 2.3.1 \n"
printf " Test case: Ref file contains more than one TAC. \n"
printf " Expected result 1: Warning. \n\n"
printf " Expected result 2: First TAC is used. \n\n"

tacadd -ovr reftemp.tac pet_pk5_long.tac "ref 1"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacadd reftemp.tac pet_pk5_long.tac "tis 4"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

rm -f stdout.txt output.res
$PROGRAM -lim=paral.lim -r=reftemp.tac plasma.bld blood.bld pet_pk5_t2_long.tac 90 output.res &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Warning "ref 1 selected of 2 reference regions."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"
iftisval output.res "Reference region" "ref 1"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval output.res "Reference file" "reftemp.tac"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
resdel output.res "ref 1"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
resdel output.res "tis 4"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parmatch -abs=0.01 -rel=1 -par=1-5 -sd=n -cl=n correct_1_2_0.res output.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"




printf "\n --------------------------------------------------------------- \n"

printf "\n 2.4.0 \n"
printf " Test case: Data file contains NaN. \n"
printf " Expected result: Error. \n\n"

rm -f stdout.txt
$PROGRAM plasma_na.bld blood.bld pet_pk5.tac 60 output.res &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "missing sample(s) in data"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

rm -f stdout.txt
$PROGRAM plasma.bld blood_na.bld pet_pk5.tac 60 output.res &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "missing sample(s)"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

rm -f stdout.txt
$PROGRAM plasma.bld blood.bld pet_pk5_na.tac 60 output.res &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "missing sample(s) in pet_pk5_na.tac"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n --------------------------------------------------------------- \n"


printf "\n 2.5.0 \n"
printf " Test case: Data file contains only 3 samples. \n"
printf " Expected result: Error. \n\n"

rm -f stdout.txt
$PROGRAM plasma_3.bld blood.bld pet_pk5.tac 60 output.res &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "plasma_3.bld contains too few samples"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

rm -f stdout.txt
$PROGRAM plasma.bld blood_3.bld pet_pk5.tac 60 output.res &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "blood_3.bld contains too few samples"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

rm -f stdout.txt
$PROGRAM plasma.bld blood.bld pet_pk5_few.tac 60 output.res &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "too few samples in specified fit duration."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 2.5.1 \n"
printf " Test case: Blood file contains only 3 samples. \n"
printf " Expected result: No error if Vb=0. \n\n"

rm -f stdout.txt
$PROGRAM -Vb=0 plasma.bld blood_3.bld pet_pk5_t2_long.tac 60 output.res &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"





printf "\n===================================================================\n"
printf "   Constraints \n"
printf "===================================================================\n"

printf "\n 3.0.0 \n"
printf " Test case: constraint file is created with option -lim=fname.\n"
printf " Expected result: File is created with correct contents. \n\n"

rm -f output.par
$PROGRAM -lim=output.par
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

iftisval output.par K1_lower 0
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
iftisval output.par K1_upper 5
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
iftisval -abs=0.01 output.par K1k2_lower 0.00001
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
iftisval output.par K1k2_upper 10
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
iftisval output.par k3_lower 0
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
iftisval output.par k3_upper 2
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
iftisval -abs=0.01 output.par k3k4_lower 0
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
iftisval output.par k3k4_upper
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
iftisval output.par k5_lower 0
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
iftisval output.par k5_upper
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
iftisval output.par Vb_lower 0
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
iftisval output.par Vb_upper 
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 3.0.1 \n"
printf " Test case: existing constraint file and option -lim=fname.\n"
printf " Expected result: Error. \n\n"

rm -f stdout.txt
$PROGRAM -lim=output.par &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error" "parameter constraint file output.par exists."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 3.0.2 \n"
printf " Test case: trying to use nonexisting constraint file.\n"
printf " Expected result: Error. \n\n"

rm -f stdout.txt
$PROGRAM -lim=nonexisting.par plasma.bld blood.bld pet_pk5_long.tac 60 output.res &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error in reading 'nonexisting.par'" "cannot open file"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 3.0.3 \n"
printf " Test case: trying to use constraint file with bad contents.\n"
printf " Expected result: Error. \n\n"

rm -f stdout.txt
$PROGRAM -lim=badfile.dat plasma.bld blood.bld pet_pk5_long.tac 60 output.res &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error in reading 'badfile.dat'" "file contains no data"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"




printf "\n===================================================================\n"
printf "   All passed! \n"
printf "===================================================================\n"
exit 0

