/** @file efixplnr.c
 *  @brief Set plane, frame, gate and/or bed numbers in ECAT files to 
 *         continuous sequence.
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 *  @test Add tests.
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include <time.h>
/*****************************************************************************/
#include "libtpcmisc.h"
#include "libtpcimgio.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Convert ECAT 6.3 or 7.x matrix numbers (planes, frames, gates, and/or beds)",
  "into continuous sequence, for instance, plane numbers 8,10,12 to 1,2,3.",
  " ",
  "Usage: @P [Options] ecatfile(s)",
  " ",
  "Options:",
  " -planes=<Y|n>",
  "     Plane numbers are changed to be continuous and start from 1 (Y, default)",
  "     or not changed (n).",
  " -frames=<y|N>",
  "     Frame numbers are changed to be continuous and start from 1 (y)",
  "     or not changed (N, default).",
  " -gates=<y|N>",
  "     Gate numbers are changed to be continuous and start from 1 (y)",
  "     or not changed (N, default).",
  " -beds=<y|N>",
  "     Bed numbers are changed to be continuous and start from 0 (y)",
  "     or not changed (N, default).",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "See also: esplit, lmlist, e7vplavg, e63mreg, img2flat, ecat2ana",
  " ",
  "Keywords: ECAT, matrixlist, image, sinogram, plane, frame, tool",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
/** Create a new ECAT 6.3 file with name outfile, containing data from
    existing file (file pointer given), with given mainheader and matrixlist.

    @return errstatus, which is STATUS_OK (0) when call was successful,
    and >0 in case of an error.  
 */
int ecat63CopyFile(
  /** Pointer to mainheader which is appropriate for both input and output data */
  ECAT63_mainheader *mh,
  /** Pointer to matrixlist which is appropriate for both input and output data */
  MATRIXLIST *ml,
  /** File pointer to input file opened in binary mode */
  FILE *fp,
  /** Filename for output file */
  char *resfile
) {
  FILE *fp2=NULL;
  int m, blkNr, nxtblk, bi;
  char buf[MatBLKSIZE];


  if(mh==NULL || ml==NULL || fp==NULL || resfile==NULL) return STATUS_FAULT;
  /* Open output file */
  fp2=ecat63Create(resfile, mh); if(fp2==NULL) return STATUS_NOWRITEPERM;
  /* Copy the matrices */
  for(m=0; m<ml->matrixNr; m++) if(ml->matdir[m].matstat==1) {
    blkNr=1+ml->matdir[m].endblk-ml->matdir[m].strtblk;
    /* Get block number for matrix header and data */
    nxtblk=ecat63Matenter(fp2, ml->matdir[m].matnum, blkNr-1);
    if(nxtblk<1) {fclose(fp2); remove(resfile); return STATUS_INVALIDMATLIST;}
    /* Copy each block */
    for(bi=ml->matdir[m].strtblk; bi<=ml->matdir[m].endblk; bi++) {
      /* Read block */
      fseek(fp, (bi-1)*MatBLKSIZE, SEEK_SET);
      if(ftell(fp)!=(bi-1)*MatBLKSIZE) {
        fclose(fp2); remove(resfile); return STATUS_NOMATRIX;}
      if(fread(buf, MatBLKSIZE, 1, fp)<1) {
        fclose(fp2); remove(resfile); return STATUS_NOMATRIX;}
      /* Write block */
      fseek(fp2, (nxtblk-1)*MatBLKSIZE, SEEK_SET);
      if(ftell(fp2)!=(nxtblk-1)*MatBLKSIZE) {
        fclose(fp2); remove(resfile); return STATUS_INVALIDMATLIST;}
      if(fwrite(buf, 1, MatBLKSIZE, fp2)<1) {
        fclose(fp2); remove(resfile); return STATUS_DISKFULL;}
      nxtblk++;
    }
  }
  fclose(fp2);
  return STATUS_OK;
}
/*****************************************************************************/

/*****************************************************************************/
/** Create a new ECAT 7 file with name outfile, containing data from
    existing file (file pointer given), with given mainheader and matrixlist.
    @return errstatus, which is STATUS_OK (0) when call was successful,
    and >0 in case of an error.  
 */
int ecat7CopyFile(
  /** Pointer to mainheader which is appropriate for both input and output data */
  ECAT7_mainheader *mh,
  /** Pointer to matrixlist which is appropriate for both input and output data */
  ECAT7_MATRIXLIST *ml,
  /** File pointer to input file opened in binary mode */
  FILE *fp,
  /** Filename for output file */
  char *resfile
) {
  FILE *fp2=NULL;
  int m, blkNr, nxtblk, bi;
  char buf[MatBLKSIZE];


  if(mh==NULL || ml==NULL || fp==NULL || resfile==NULL) return STATUS_FAULT;
  /* Open output file */
  fp2=ecat7Create(resfile, mh); if(fp2==NULL) return STATUS_NOWRITEPERM;
  /* Copy the matrices */
  for(m=0; m<ml->matrixNr; m++) if(ml->matdir[m].status==1) {
    blkNr=1+ml->matdir[m].endblk-ml->matdir[m].strtblk;
    /* Get block number for matrix header and data */
    nxtblk=ecat7EnterMatrix(fp2, ml->matdir[m].id, blkNr-1);
    if(nxtblk<1) {fclose(fp2); remove(resfile); return STATUS_INVALIDMATLIST;}
    /* Copy each block */
    for(bi=ml->matdir[m].strtblk; bi<=ml->matdir[m].endblk; bi++) {
      /* Read block */
      fseek(fp, (bi-1)*MatBLKSIZE, SEEK_SET);
      if(ftell(fp)!=(bi-1)*MatBLKSIZE) {
        fclose(fp2); remove(resfile); return STATUS_NOMATRIX;}
      if(fread(buf, MatBLKSIZE, 1, fp)<1) {
        fclose(fp2); remove(resfile); return STATUS_NOMATRIX;}
      /* Write block */
      fseek(fp2, (nxtblk-1)*MatBLKSIZE, SEEK_SET);
      if(ftell(fp2)!=(nxtblk-1)*MatBLKSIZE) {
        fclose(fp2); remove(resfile); return STATUS_INVALIDMATLIST;}
      if(fwrite(buf, 1, MatBLKSIZE, fp2)<1) {
        fclose(fp2); remove(resfile); return STATUS_DISKFULL;}
      nxtblk++;
    }
  }
  fclose(fp2);
  return STATUS_OK;
}
/*****************************************************************************/

/*****************************************************************************/
/** Set plane, frame, gate and/or bed numbers in ECAT files to continuous
    sequence.
    @return Returns 0 when successful.
 */
int ecatFixMatrixlist(
  /** Pointer to ECAT filename */
  char *ecatfile,
  /** Do or do not fix plane numbers (1/0) */
  int do_planes,
  /** Do or do not fix frame numbers (1/0) */
  int do_frames,
  /** Do or do not fix gate numbers (1/0) */
  int do_gates,
  /** Do or do not fix bed numbers (1/0) */
  int do_beds,
  /** Verbose level; set to <=0 to print nothing in stdout */
  int verbose
) {
  int               ret;
  int               ecat_format=0; // 6 or 7 
  char              outfile[FILENAME_MAX];
  FILE             *fp=NULL;
  ECAT7_mainheader  ecat7_main_header;
  ECAT63_mainheader ecat63_main_header;
  ECAT7_MATRIXLIST  ml7;
  MATRIXLIST        ml6;


  /*
   *  Open the file
   */
  if(verbose>1) printf("  opening %s in binary format\n", ecatfile);
  if((fp=fopen(ecatfile, "rb")) == NULL) {
    fprintf(stderr, "Error: cannot read file %s\n", ecatfile);
    return(2);
  }
  
  /*
   *  Determine the file format (ECAT 6 or 7)
   *  Read mainheader   
   */
  if(verbose>2) printf("  reading main header\n");
  ecat_format=0;
  /* Try to read ECAT 7.x main header */
  ret=ecat7ReadMainheader(fp, &ecat7_main_header);
  if(ret) {
    fprintf(stderr, "Error: unsupported file format %s\n", ecatfile);
    fclose(fp); return(3);
  }
  /* If header could be read, check for magic number */
  if(strncmp(ecat7_main_header.magic_number, ECAT7V_MAGICNR, 7)==0) {
    ecat_format=7;
  } else { // maybe this is ECAT 6.3
    ret=ecat63ReadMainheader(fp, &ecat63_main_header);
    if(ret==0) ecat_format=6;
  }
  if(verbose>2) printf("  ecat_format := %d\n", ecat_format);

  /*
   *  Read matrixlist
   */   
  if(verbose>1) printf("  reading matrix list\n");
  ecat7InitMatlist(&ml7);
  ecat63InitMatlist(&ml6);
  if(ecat_format==7) {
    ret=ecat7ReadMatlist(fp, &ml7, verbose-1);
    if(ret==0 && (ml7.matrixNr<1 || ecat7CheckMatlist(&ml7))) ret=99;
  } else if(ecat_format==6) {
    ret=ecat63ReadMatlist(fp, &ml6, verbose-1);
    if(ret==0 && (ml6.matrixNr<1 || ecat63CheckMatlist(&ml6))) ret=99;
  } else {
    fprintf(stderr, "Error: unsupported file format %s\n", ecatfile);
    fclose(fp); return(3);  
  }
  if(ret) {
    fprintf(stderr, "Error %d: invalid matrix list.\n", ret);
    fclose(fp); return(4);
  }
  
  /*
   *  Fix matrixlist
   */
  if(verbose>1) printf("  fixing matrix list\n");
  if(ecat_format==6) {
    if(verbose>2) ecat63PrintMatlist(&ml6); // print original matrixlist
    ecat63GatherMatlist(&ml6, do_planes, do_frames, do_gates, do_beds);
    if(verbose>2) ecat63PrintMatlist(&ml6); // print fixed matrixlist
  } else {
    if(verbose>2) ecat7PrintMatlist(&ml7); // print original matrixlist
    ecat7GatherMatlist(&ml7, do_planes, do_frames, do_gates, do_beds);  
    if(verbose>2) ecat7PrintMatlist(&ml7); // print fixed matrixlist
  }

  /*
   *  Replace ECAT data with new matrixlist:
   *  Write data with new matrix list in temp file
   *  Delete original file
   *  Rename temp file to original filename
   */
  if(verbose>1) printf("  writing fixed data in temporary file\n");
  /* Make tempfile name */    
  strcpy(outfile, ecatfile); strcat(outfile, "_efixplnr");
  if(verbose>3) printf("  tempfile := %s\n", outfile);
  if(ecat_format==6) {
    /* Update num_* in mainheader */
    ecat63GetNums(&ml6,
                        &ecat63_main_header.num_planes, // strict 2D format
                        &ecat63_main_header.num_frames,
                        &ecat63_main_header.num_gates,
                        &ecat63_main_header.num_bed_pos);
    ret=ecat63CopyFile(&ecat63_main_header, &ml6, fp, outfile);
  } else {
    /* Update num_* in mainheader */
    ecat7GetNums(&ml7, &ecat7_main_header, fp,
                       &ecat7_main_header.num_planes,
                       &ecat7_main_header.num_frames,
                       &ecat7_main_header.num_gates,
                       &ecat7_main_header.num_bed_pos);
    ret=ecat7CopyFile(&ecat7_main_header, &ml7, fp, outfile);
  }
  fclose(fp);
  if(ret) {fprintf(stderr, "Error: %s\n", imgStatus(ret)); return(11);}
  /* Remove input file */
  if(verbose>1) printf("  removing %s\n", ecatfile);
  ret=remove(ecatfile); if(ret) {
    fprintf(stderr, "Error: cannot overwrite file %s\n", ecatfile);
    return(13);
  }
  /* Rename temp file to input name */
  if(verbose>1) printf("  renaming %s to %s\n", outfile, ecatfile);
  ret=rename(outfile, ecatfile); if(ret) {
    fprintf(stderr, "Error: could not rename %s to %s\n", outfile, ecatfile);
    return(15);
  }

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @cond
/**
 *  main()
 */
int main(int argc, char **argv)
{
  int               ai, help=0, version=0, verbose=1;
  int               ret, ffi=0, fileNr=0, okNr=0;
  short int         do_planes=1, do_frames=0, do_gates=0, do_beds=0;
  char             *cptr;
  char              ecatfile[FILENAME_MAX];


  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  ecatfile[0]=(char)0;
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') { /* options */
    cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(cptr==NULL) continue;
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    if(strncasecmp(cptr, "PLANES=", 7)==0) {
      cptr+=7;
      if(strncasecmp(cptr, "YES", 1)==0 || strcasecmp(cptr, "ON")==0) {
        do_planes=1; continue;}
      if(strncasecmp(cptr, "NO", 1)==0 || strcasecmp(cptr, "OFF")==0) {
        do_planes=0; continue;}
    } else if(strncasecmp(cptr, "FRAMES=", 7)==0) {
      cptr+=7;
      if(strncasecmp(cptr, "YES", 1)==0 || strcasecmp(cptr, "ON")==0) {
        do_frames=1; continue;}
      if(strncasecmp(cptr, "NO", 1)==0 || strcasecmp(cptr, "OFF")==0) {
        do_frames=0; continue;}
    } else if(strncasecmp(cptr, "GATES=", 6)==0) {
      cptr+=6;
      if(strncasecmp(cptr, "YES", 1)==0 || strcasecmp(cptr, "ON")==0) {
        do_gates=1; continue;}
      if(strncasecmp(cptr, "NO", 1)==0 || strcasecmp(cptr, "OFF")==0) {
        do_gates=0; continue;}
    } else if(strncasecmp(cptr, "BEDS=", 5)==0) {
      cptr+=5;
      if(strncasecmp(cptr, "YES", 1)==0 || strcasecmp(cptr, "ON")==0) {
        do_beds=1; continue;}
      if(strncasecmp(cptr, "NO", 1)==0 || strcasecmp(cptr, "OFF")==0) {
        do_beds=0; continue;}
    }
    fprintf(stderr, "Error: invalid option '%s'\n", argv[ai]);
    return(1);
  } else break;
  
  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  /* Process other arguments, starting from the first non-option */
  for(; ai<argc; ai++) {
    if(ffi<1) ffi=ai;
    fileNr++;
  }

  /* Is something missing? */
  if(fileNr<1) {
    fprintf(stderr, "Error: missing command-line argument; use option --help\n");
    return(1);
  }

  /* In verbose mode print arguments and options */
  if(verbose>1) {
    printf("do_planes := %d\n", do_planes);
    printf("do_frames := %d\n", do_frames);
    printf("do_gates := %d\n", do_gates);
    printf("do_beds := %d\n", do_beds);
    printf("fileNr := %d\n", fileNr);
  }

  /* Anything to do? */
  if(do_planes==0 && do_frames==0 && do_gates==0 && do_beds==0) return(0);

  /* Fix all ECAT files */
  for(ai=ffi; ai<argc; ai++) {
    strcpy(ecatfile, argv[ai]);
    if(verbose>0 && fileNr>1) printf("fixing %s\n", ecatfile);
    ret=ecatFixMatrixlist(ecatfile, do_planes, do_frames, do_gates, do_beds,
                          verbose-2);
    if(ret==0) {
      if(verbose>1) fprintf(stdout, "  ok.\n");
      okNr++;
    } else {
      if(verbose>0) fprintf(stdout, "  failed (%d).\n", ret);
    }
  }
  if(okNr<fileNr) {
    fprintf(stderr, "Error: %d out of %d files failed.\n", fileNr-okNr, fileNr);
    return(1);
  } else {
    if(verbose>0) fprintf(stdout, "%d file(s) fixed.\n", okNr); 
  }
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
