/** @file eframe.c
 *  @brief Lists or sets time frames in ECAT 6.3 or 7 file.
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <math.h>
#include <string.h>
#include <time.h>
/*****************************************************************************/
#include "libtpcmisc.h"
#include "libtpcimgio.h"
#include "libtpcimgp.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "List the time frame information of an ECAT 6.3 or 7.x file.",
  "The frame start times and lengths can be saved in SIF.",
  " ",
  "If an existing SIF is specified, the frame times are read from it and",
  "written in the ECAT file; SIF and ECAT file must contain an equal number",
  "of frames. If ECAT file does not contain isotope or scan start time,",
  "they are copied from the SIF, too.",
  " ",
  "Warning! Make sure that SIF does not exist, unless you want to change",
  "the frame times in ECAT file.",
  " ",
  "Usage: @P [-Options] ecatfile [SIF]",
  " ",
  "Options:",
  " -ift",
  "     Frame file is written in interfile type format; currently",
  "     this format is not supported in changing the frame times.",
  "     Times are saved in min; use option -sec to save in sec.",
  " -plain",
  "     Frame file is written with only frame start times and lengths.",
  "     Times are saved in min; use option -sec to save in sec.",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "See also: esetstrt, egetstrt, ecattime, tacframe, imghead, imgdelfr",
  " ",
  "Keywords: image, ECAT, time, SIF",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int       ai, help=0, version=0, verbose=1;
  int       fi, ret, sec=0, frame_nr;
  int       output_format=2; // 0=plain, 1=ift, 2=sif
  char      ecatfile[FILENAME_MAX], datfile[FILENAME_MAX], tmp[512], *cptr;
  float     start=0., dur=0., end=0.;
  IMG       img;
  FILE     *fp=NULL;
  SIF       esif, nsif;
  double    hl;


  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  ecatfile[0]=datfile[0]=tmp[0]=(char)0;
  imgInit(&img); sifInit(&esif); sifInit(&nsif);
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') { /* options */
    cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(cptr==NULL) continue;
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    if(strcasecmp(cptr, "IFT")==0) {
      output_format=1; continue;
    } else if(strcasecmp(cptr, "SIF")==0) {
      output_format=2; continue;
    } else if(strcasecmp(cptr, "PLAIN")==0) {
      output_format=0; continue;
    } else if(strncasecmp(cptr, "SEC", 3)==0 || strcasecmp(cptr, "S")==0) {
      sec=1; continue;
    } else if(strncasecmp(cptr, "MIN", 3)==0 || strcasecmp(cptr, "M")==0) {
      sec=0; continue;
    }
    fprintf(stderr, "Error: invalid option '%s'.\n", argv[ai]);
    return(1);
    
  } else break;

  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  /* Process other arguments, starting from the first non-option */
  if(ai<argc) strlcpy(ecatfile, argv[ai++], FILENAME_MAX);
  if(ai<argc) strlcpy(datfile, argv[ai++], FILENAME_MAX);
  if(ai<argc) {fprintf(stderr, "Error: invalid argument '%s'.\n", argv[ai]); return(1);}

  /* Is something missing? */
  if(!ecatfile[0]) {
    fprintf(stderr, "Error: missing command-line argument; try %s --help\n", argv[0]);
    return(1);
  }

  /* In verbose mode print arguments and options */
  if(verbose>1) {
    printf("ecatfile := %s\n", ecatfile);
    printf("datfile := %s\n", datfile);
    printf("output_format := %d\n", output_format);
    printf("sec := %d\n", sec);
    fflush(stdout);
  }
  if(verbose>2) ECAT63_TEST=verbose-2; else ECAT63_TEST=0;
  if(verbose>2) ECAT7_TEST=verbose-2;  else ECAT7_TEST=0;
  if(verbose>2) IMG_TEST=verbose-2;  else IMG_TEST=0;


  /* If frame file is specified, check if it exists */
  int mode=1; // 0=set frames, 1=list frames, 2=save frames
  if(datfile[0]) {
    /* Check if file exists */
    if(access(datfile, 0) != -1) {
      mode=0; /* frame file exists; change frames */
      if(verbose>0) fprintf(stdout, "Setting frame times from %s\n", datfile);
    } else {
      mode=2; /* not existing; so create it */
      /* Check if frame times are to be saved in SIF or IFT format */
      char *cptr=strrchr(datfile, '.');
      if(cptr!=NULL && strcasecmp(cptr, ".SIF")==0) output_format=2;
      if(cptr!=NULL && strcasecmp(cptr, ".IFT")==0) output_format=1;
    }
  } else
    mode=1; /* frame file not specified; just list frames */
  if(verbose>1) printf("mode := %d\n", mode);
  fflush(stdout);


  /*
   *  Read information about input file
   */
  if(verbose>1) printf("reading ECAT file\n");
  ret=imgReadHeader(ecatfile, &img, IMG_UNKNOWN); if(verbose>10) imgInfo(&img);
  if(ret) {
    fprintf(stderr, "Error in reading %s: %s\n", ecatfile, imgStatus(ret));
    imgEmpty(&img); return(2);
  }
  frame_nr=img.dimt;
  if(verbose>1) printf("frame_nr := %d\n", frame_nr);
  /* Allocate SIF to store frame times */
  ret=sifSetmem(&esif, frame_nr); if(ret!=0) {
    fprintf(stderr, "Error: out of memory.\n");
    imgEmpty(&img); return(2);
  }
  esif.colNr=4; esif.version=1;
  /* Read frame information, one frame at a time */
  imgEmpty(&img);
  for(fi=0; fi<frame_nr; fi++) {
    if(verbose>2) printf("\nframe %d\n", fi+1);
    /* Read first/next frame in file */
    ret=imgReadFrame(ecatfile, fi+1, &img, 0);
    if(ret) {
      fprintf(stderr, "\nError in reading %s: %s\n", ecatfile, imgStatus(ret));
      imgEmpty(&img); return(3);
    }
    if(verbose>0 && frame_nr>2) {fprintf(stdout, "."); fflush(stdout);}
    /* Set SIF frame time */    
    esif.x1[fi]=img.start[0]; esif.x2[fi]=img.end[0];
    if(fi==0) {
      /* Set isotope */
      strcpy(esif.isotope_name, imgIsotope(&img) );
      /* Set studynumber */
      strcpy(esif.studynr, img.studyNr);
      /* Set scan start time */
      esif.scantime=img.scanStart;
    }
  } // next frame
  if(verbose>0 && frame_nr>2) printf("\n");
  if(verbose>10) sifPrint(&esif);

  if(mode==1 || (verbose>0 && (mode==0 || mode==2))) {
    printf("Frame Start  End    Dur (sec)   Start   End     Dur (min)\n");
    fflush(stdout);
    for(fi=0; fi<esif.frameNr; fi++) {
      printf("%5d %6.1f %6.1f %6.1f      %7.3f %7.3f %7.3f\n",
        fi+1, esif.x1[fi], esif.x2[fi], esif.x2[fi]-esif.x1[fi],
        esif.x1[fi]/60., esif.x2[fi]/60., (esif.x2[fi]-esif.x1[fi])/60. );
    }
    fflush(stdout);
    if(mode==1) {
      /* Ready... Quit */
      imgEmpty(&img); sifEmpty(&esif);
      return(0);
    }
  }



  /*
   *  List frame times on screen
   */
  if(mode>0) {
    if(output_format==2) { /* SIF */
      if(mode==1) strcpy(datfile, "stdout");
      int ret=sifWrite(&esif, datfile);
      if(ret!=0) {
        fprintf(stderr, "Error: cannot write frame times in %s\n", datfile); 
        imgEmpty(&img); sifEmpty(&esif); return(11);
      }
      if(verbose>0 && mode==2) {
        if(sec) fprintf(stdout, "Frame times (sec) written in %s\n", datfile);
        else fprintf(stdout, "Frame times (min) written in %s\n", datfile);
      }
      /* Ready... Quit */
      if(verbose>0 && mode==2) {
        if(sec) fprintf(stdout, "Frame times (sec) written in %s\n", datfile);
        else fprintf(stdout, "Frame times (min) written in %s\n", datfile);
      }
      imgEmpty(&img); sifEmpty(&esif);
      return(0);
    }

    FILE *fp=NULL;
    if(mode==1) {
      fp=stdout; strcpy(datfile, "stdout");
    } else { // mode==2
      /* Open file for writing */
      if((fp=fopen(datfile, "w"))==NULL ) {
        fprintf(stderr, "Error: cannot write file %s\n", datfile); 
        imgEmpty(&img); sifEmpty(&esif); return(11);
      }
    }
    if(output_format==0) { /* just frame times */
      for(int fi=0; fi<esif.frameNr; fi++) {
        if(sec) ret=fprintf(fp, "%g  %g\n", esif.x1[fi], (esif.x2[fi]-esif.x1[fi]) );
        else ret=fprintf(fp, "%g  %g\n", esif.x1[fi]/60.0, (esif.x2[fi]-esif.x1[fi])/60.0 );
        if(ret<4) {
          fprintf(stderr, "Error: cannot write frame times in %s.\n", datfile); 
          imgEmpty(&img); sifEmpty(&esif); if(mode==2) fclose(fp); 
          return(11);
        }
      }
    } else if(output_format==1) { /* Interfile type format */
      fprintf(fp, "filename := %s\n", ecatfile);
      fprintf(fp, "frame_nr := %d\n", esif.frameNr);
      if(sec) strcpy(tmp, "sec"); else strcpy(tmp, "min"); 
      fprintf(fp, "time_unit := %s\n", tmp);
      for(int fi=0; fi<esif.frameNr; fi++) {
        if(sec) sprintf(tmp, "%g", esif.x1[fi]); else sprintf(tmp, "%g", esif.x1[fi]/60.0);
        fprintf(fp, "frame_start[%d] := %s\n", fi+1, tmp);
        if(sec) sprintf(tmp, "%g", esif.x2[fi]-esif.x1[fi]);
        else sprintf(tmp, "%g", (esif.x2[fi]-esif.x1[fi])/60.0);
        ret=fprintf(fp, "frame_duration[%d] := %s\n", fi+1, tmp);
        if(ret<4) {
          fprintf(stderr, "Error: cannot write frame times.\n"); 
          imgEmpty(&img); sifEmpty(&esif); if(mode==2) fclose(fp);
          return(11);
        }
      }
    }
    if(mode==2) fclose(fp);

    /* Ready... Quit */
    if(verbose>0 && mode==2) {
      if(sec) fprintf(stdout, "Frame times (sec) written in %s\n", datfile);
      else fprintf(stdout, "Frame times (min) written in %s\n", datfile);
    }
    imgEmpty(&img); sifEmpty(&esif);
    return(0);
  }



  /*
   *  Read the new frame times
   */
  /* Try first to read it as a SIF */
  ret=sifRead(datfile, &nsif);
  if(ret==0) { // SIF could be read
    if(nsif.frameNr!=frame_nr) {
      fprintf(stderr, "Error: different frame number in %s and %s\n", datfile, ecatfile); 
      imgEmpty(&img); sifEmpty(&esif); sifEmpty(&nsif); return(7);
    }
  } else { // it was not SIF
    if(verbose>2) printf("%s is not SIF\n", datfile);
    /* Open frame file (file with just frame times) */
    if((fp=fopen(datfile, "r")) == NULL) {
      fprintf(stderr, "Error in opening %s\n", datfile);
      imgEmpty(&img); sifEmpty(&esif); return(5);
    }
    /* Allocate memory for frame data */
    ret=sifSetmem(&nsif, frame_nr); if(ret!=0) {
      fprintf(stderr, "Error: out of memory.\n");
      imgEmpty(&img); sifEmpty(&esif); fclose(fp);  return(6);
    }
    nsif.colNr=4; nsif.version=1;
    /* Read and get frames */
    int fi=0; end=0.0;
    while(fgets(tmp, 256, fp)!=NULL) {
      if(tmp[0]=='#') continue;
      ret=sscanf(tmp, "%f %f", &start, &dur); if(ret<=0) continue;
      if(verbose>22) printf("'%s' -> n=%d, start=%g\n", tmp, ret, start);
      if(fi==frame_nr) {fi=0; break;}
      if(ret==1) {
        nsif.x1[fi]=end; nsif.x2[fi]=nsif.x1[fi]+start;
      } else {
        nsif.x1[fi]=start; nsif.x2[fi]=nsif.x1[fi]+dur;
      }
      end=nsif.x2[fi];
      if(verbose>21) printf("fi=%d %g %g %g\n", fi, start, end, dur);
      fi++;
    } 
    fclose(fp);
    if(fi!=frame_nr) {
      fprintf(stderr, "Error: different frame number in %s and %s\n", datfile, ecatfile); 
      imgEmpty(&img); sifEmpty(&esif); sifEmpty(&nsif); return(7);
    }
    /* Convert times into seconds, if they are not seconds already */
    if(sec==0) for(fi=0; fi<nsif.frameNr; fi++) {nsif.x1[fi]*=60.; nsif.x2[fi]*=60.;}
  }
  if(verbose>0) fprintf(stdout, "Frame times read from %s\n", datfile);
  if(verbose>0) fflush(stdout);
  if(verbose>10) sifPrint(&esif);


  /*
   *  Change the frame times in ECAT file
   */
  /* Backup the original ECAT file */
  if(access(ecatfile, 0) != -1) {
    strcpy(tmp, ecatfile); strcat(tmp, ".bak");
    if(access(tmp, 0) != -1) remove(tmp);
    if(verbose>0) printf("Making backup %s -> %s\n", ecatfile, tmp);
    rename(ecatfile, tmp);
    if(access(ecatfile, 0) != -1) {
      fprintf(stderr, "Error: cannot overwrite %s\n", ecatfile);
      imgEmpty(&img); sifEmpty(&esif); sifEmpty(&nsif); return(12);
    }
  }
  if(verbose>0) printf("editing ECAT file\n");
  imgEmpty(&img);
  for(int fi=0; fi<frame_nr; fi++) {
    if(verbose>1) printf("\nframe %d\n", fi+1);
    /* Read first/next frame in file */
    /* Note: we are reading from backup file because file with original name 
       does not exist currently or contains just the output */
    ret=imgReadFrame(tmp, fi+1, &img, 0);
    if(ret) {
      fprintf(stderr, "\nError in reading %s: %s\n", ecatfile, imgStatus(ret));
      imgEmpty(&img); sifEmpty(&esif); sifEmpty(&nsif);
      return(3);
    }
    /* Set frame time from SIF */    
    img.start[0]=nsif.x1[fi]; img.end[0]=nsif.x2[fi];
    img.mid[0]=0.5*(img.start[0]+img.end[0]);
    /* Set isotope, if it does not exist */
    if(img.isotopeHalflife<=1.0) {
      hl=hlFromIsotope(nsif.isotope_name);
      if(hl>0) {
        if(verbose>0 && fi==0)
          fprintf(stdout, "\nnew halflife := %g [s]\n", 60.0*hl);
        img.isotopeHalflife=60.0*hl;
      }
    }
    /* Do not set studynumber : strcpy(img.studyNr, esif.studynr); */
    /* Set scan start time, if it does not exist */
    if(nsif.scantime>1000 && img.scanStart<1000) img.scanStart=nsif.scantime;
    /* Write frame in file */
    ret=imgWriteFrame(ecatfile, fi+1, &img, 0);
    if(ret) {
      fprintf(stderr, "\nError in writing %s: %s\n", ecatfile, imgStatus(ret));
      imgEmpty(&img); sifEmpty(&esif); sifEmpty(&nsif); return(13);
    }
    if(verbose>0 && frame_nr>2) {fprintf(stdout, "."); fflush(stdout);}
  } // next frame
  if(verbose>0 && frame_nr>2) printf("\n");


  /*
   *  List modified frame times on screen.
   */
  if(verbose>0) {
    printf("Modified frame times:\n");
    printf("Frame Start End   Dur (sec)  Start  End    Dur (min)\n");
    fflush(stdout);
    for(int fi=0; fi<nsif.frameNr; fi++) {
      printf("%5d %5.0f %5.0f %5.0f      %6.2f %6.2f %6.2f\n",
        fi+1, nsif.x1[fi], nsif.x2[fi], nsif.x2[fi]-nsif.x1[fi],
        nsif.x1[fi]/60., nsif.x2[fi]/60., (nsif.x2[fi]-nsif.x1[fi])/60. );
    }
  }

  imgEmpty(&img); sifEmpty(&esif); sifEmpty(&nsif);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
