/** @file esplit.c
 *  @brief Extracts specified ECAT 6.3 or ECAT 7 frames and planes to new file.
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 *  @test Add tests (although used when testing other applications).
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <math.h>
#include <string.h>
#include <time.h>
/*****************************************************************************/
#include "libtpcmisc.h"
#include "libtpcimgio.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Extract specified time frames and planes in ECAT 6.3 or 7.x matrix file",
  "to a new file.",
  "Note! The program interprets ECAT 7 files as if they had only one plane.",
  " ",
  "Usage: @P [Options] file frames planes outputfile ",
  " ",
  "Options:",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "Example:",
  "  @P a2345dy1.img 1-20 4,6,8,10 a2345dy1_part.img ",
  " ",
  "See also: lmlist, imgdelfr, imgdelpl, efixplnr, imgbox, imgslim, e63mreg",
  " ",
  "Keywords: image, ECAT, tool, cropping, modelling",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int                 ai, help=0, version=0, verbose=1;
  int                 bi, m, fi, pi, ret;
  int                 frameNr=0, planeNr=0, delNr=0, blkNr=0;
  int                 ecat_version=7, nxtblk;
  char                ecatfile[FILENAME_MAX], outfile[FILENAME_MAX],
                      tmp[FILENAME_MAX], *cptr, buf[MatBLKSIZE];
  INTEGER_LIST        planelist, framelist;
  FILE               *fp, *fp2;
  ECAT7_mainheader    mainheader7;
  ECAT63_mainheader   mainheader63;
  ECAT7_MATRIXLIST    mlist7;
  MATRIXLIST          mlist63;
  ECAT7_Matval        matval7;
  Matval              matval63;


  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  ecatfile[0]=outfile[0]=(char)0;
  ecat63InitMatlist(&mlist63); ecat7InitMatlist(&mlist7);
  integerListInit(&planelist); integerListInit(&framelist);
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') { /* options */
    cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(cptr==NULL) continue;
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    fprintf(stderr, "Error: invalid option '%s'.\n", argv[ai]);
    return(1);
  } else break;
  
  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  /* Process other arguments, starting from the first non-option */
  for(; ai<argc; ai++) {
    if(!ecatfile[0]) {
      /* check that file exists */
      if(access(argv[ai], 0) == -1) {
        fprintf(stderr, "Error: file '%s' does not exist.\n", argv[ai]);
        return(1);
      }
      strlcpy(ecatfile, argv[ai], FILENAME_MAX); continue;
    } else if(framelist.nr==0) {
      m=integerListExpandFromString(argv[ai], ",;", &framelist, 1); if(m>0) continue;
    } else if(planelist.nr==0) {
      m=integerListExpandFromString(argv[ai], ",;", &planelist, 1); if(m>0) continue;
    } else if(!outfile[0]) {
      strlcpy(outfile, argv[ai], FILENAME_MAX); continue;
    }
    fprintf(stderr, "Error: invalid argument '%s'.\n", argv[ai]);
    integerListEmpty(&planelist); integerListEmpty(&framelist); 
    return(1);
  } /* next argument */

  /* Is something missing? */
  if(!outfile[0]) {
    fprintf(stderr, "Error: missing command-line argument; try %s --help\n", argv[0]);
    integerListEmpty(&planelist); integerListEmpty(&framelist);
    return(1);
  }

  /* Check output filename */
  if(strcasecmp(ecatfile, outfile)==0) {
    fprintf(stderr, "Error: same name for input and output file.\n");
    integerListEmpty(&planelist); integerListEmpty(&framelist);
    return(1);
  }

  /* In verbose mode print arguments and options */
  if(verbose>1) {
    printf("ecatfile := %s\n", ecatfile);
    printf("outputfile := %s\n", outfile);
    printf("Frames:");
    for(fi=0; fi<framelist.nr; fi++) printf(" %d", framelist.list[fi]);
    printf("\n");
    printf("Planes:");
    for(pi=0; pi<planelist.nr; pi++) printf(" %d", planelist.list[pi]);
    printf("\n");
  }


  /*
   *  Read main header and matrix list
   */
  if(verbose>1) {fprintf(stdout, "reading %s\n", ecatfile); fflush(stdout);}

  /* Open file for read */
  if((fp=fopen(ecatfile, "rb")) == NULL) {
    fprintf(stderr, "Error: cannot read file %s\n", ecatfile);
    integerListEmpty(&planelist); integerListEmpty(&framelist);
    return(3);
  }

  /* Try to read ECAT 7.x main header */
  ret=ecat7ReadMainheader(fp, &mainheader7);
  if(ret) {
    fprintf(stderr, "Error %d: cannot read main header.\n", ret);
    integerListEmpty(&planelist); integerListEmpty(&framelist);
    fclose(fp); return(2);
  }
  /* If header could be read, check for magic number */
  if(strncmp(mainheader7.magic_number, ECAT7V_MAGICNR, 7)==0) {
    ecat_version=7;
  } else {
    /* Check if file is in ECAT 6.3 format */
    ret=ecat63ReadMainheader(fp, &mainheader63);
    if(ret) {
      fprintf(stderr, "Error %d: cannot read main header.\n", ret);
      integerListEmpty(&planelist); integerListEmpty(&framelist);
      fclose(fp); return(2);
    }
    ecat_version=6;
  }
  if(verbose>1) printf("  Format := %d\n", ecat_version);

  /* Read matrix list */
  if(ecat_version==7) {
    ret=ecat7ReadMatlist(fp, &mlist7, verbose-2);
    if(ret==0 && (mlist7.matrixNr<1 || ecat7CheckMatlist(&mlist7))) ret=99;
  } else {
    ret=ecat63ReadMatlist(fp, &mlist63, verbose-2);
    if(ret==0 && (mlist63.matrixNr<1 || ecat63CheckMatlist(&mlist63))) ret=99;
  }
  if(ret) {
    fprintf(stderr, "Error %d: invalid matrix list.\n", ret);
    integerListEmpty(&planelist); integerListEmpty(&framelist);
    fclose(fp); return(4);
  }
  if(verbose>20) {
    if(ecat_version==7) ecat7PrintMatlist(&mlist7);
    else ecat63PrintMatlist(&mlist63);
  }

  /*
   *  Mark non-listed matrices as deleted
   */
  delNr=0;
  if(ecat_version==7) {
    for(m=0; m<mlist7.matrixNr; m++) {
      ecat7_id_to_val(mlist7.matdir[m].id, &matval7);
      ret=0;
      for(pi=0; pi<planelist.nr; pi++) if(planelist.list[pi]==matval7.plane) {ret++; break;}
      for(fi=0; fi<framelist.nr; fi++) if(framelist.list[fi]==matval7.frame) {ret++; break;}
      if(ret<2) {mlist7.matdir[m].status=-1; delNr++; continue;}
      if(matval7.plane>planeNr) planeNr=matval7.plane;
      if(matval7.frame>frameNr) frameNr=matval7.frame;
    }
    ret=mlist7.matrixNr-delNr;
    mainheader7.num_planes=planeNr;
    mainheader7.num_frames=frameNr;
  } else {
    for(m=0; m<mlist63.matrixNr; m++) {
      mat_numdoc(mlist63.matdir[m].matnum, &matval63);
      ret=0;
      for(pi=0; pi<planelist.nr; pi++) if(planelist.list[pi]==matval63.plane) {ret++; break;}
      for(fi=0; fi<framelist.nr; fi++) if(framelist.list[fi]==matval63.frame) {ret++; break;}
      if(ret<2) {mlist63.matdir[m].matstat=-1; delNr++; continue;}
      if(matval63.plane>planeNr) planeNr=matval63.plane;
      if(matval63.frame>frameNr) frameNr=matval63.frame;
    }
    ret=mlist63.matrixNr-delNr;
    mainheader63.num_planes=planeNr;
    mainheader63.num_frames=frameNr;
  }
  if(ret==0) {
    fprintf(stderr, "Error: none of the specified matrices were found.\n");
    ecat7EmptyMatlist(&mlist7); ecat63EmptyMatlist(&mlist63);
    integerListEmpty(&planelist); integerListEmpty(&framelist);
    fclose(fp); return(5);
  }
  if(verbose>0) fprintf(stdout, "  extracting %d matrices\n", ret);
  if(verbose>9) {
    printf("After checking for matches:\n");
    if(ecat_version==7) ecat7PrintMatlist(&mlist7);
    else ecat63PrintMatlist(&mlist63);
  }
  integerListEmpty(&planelist); integerListEmpty(&framelist);


  /*
   *  Write output file
   */
  /* Check if output file exists; rename to backup file, if necessary */
  backupExistingFile(outfile, NULL, tmp);

  if(ecat_version==7) {
    /* Open output file */
    fp2=ecat7Create(outfile, &mainheader7);
    if(fp2==NULL) {
      fprintf(stderr, "Error: cannot write ECAT file.\n");
      ecat7EmptyMatlist(&mlist7); ecat63EmptyMatlist(&mlist63);
      fclose(fp); return(11);
    }
    /* Copy the matrices */
    for(m=0; m<mlist7.matrixNr; m++) if(mlist7.matdir[m].status==1) {
      blkNr=1+mlist7.matdir[m].endblk-mlist7.matdir[m].strtblk;
      /* Get block number for matrix header and data */
      nxtblk=ecat7EnterMatrix(fp2, mlist7.matdir[m].id, blkNr-1);
      if(nxtblk<1) {
        fprintf(stderr, "Error: cannot write ECAT matrix.\n");
        ecat7EmptyMatlist(&mlist7); ecat63EmptyMatlist(&mlist63);
        fclose(fp); fclose(fp2); remove(outfile); return(13);
      }
      if(verbose>3) printf("  m=%d blkNr=%d nxtblk=%d\n", m, blkNr, nxtblk);
      /* Copy each block */
      for(bi=mlist7.matdir[m].strtblk; bi<=mlist7.matdir[m].endblk; bi++) {
        /* Read block */
        fseek(fp, (bi-1)*MatBLKSIZE, SEEK_SET);
        if(ftell(fp)!=(bi-1)*MatBLKSIZE) {
          fprintf(stderr, "Error: cannot find matrix block %d.\n", bi);
          ecat7EmptyMatlist(&mlist7); ecat63EmptyMatlist(&mlist63);
          fclose(fp); fclose(fp2); remove(outfile); return(8);
        }
        if(fread(buf, MatBLKSIZE, 1, fp)<1) {
          fprintf(stderr, "Error: cannot read matrix block %d.\n", bi);
          ecat7EmptyMatlist(&mlist7); ecat63EmptyMatlist(&mlist63);
          fclose(fp); fclose(fp2); remove(outfile); return(9);
        }
        /* Write block */
        fseek(fp2, (nxtblk-1)*MatBLKSIZE, SEEK_SET);
        if(ftell(fp2)!=(nxtblk-1)*MatBLKSIZE) {
          fprintf(stderr, "Error: cannot find matrix block %d for write.\n", nxtblk);
          ecat7EmptyMatlist(&mlist7); ecat63EmptyMatlist(&mlist63);
          fclose(fp); fclose(fp2); remove(outfile); return(15);
        }
        if(fwrite(buf, 1, MatBLKSIZE, fp2)<1) {
          fprintf(stderr, "Error: cannot write matrix block %d.\n", nxtblk);
          ecat7EmptyMatlist(&mlist7); ecat63EmptyMatlist(&mlist63);
          fclose(fp); fclose(fp2); remove(outfile); return(16);
        }
        nxtblk++;
      }
    }
  } else {
    /* Open output file */
    fp2=ecat63Create(outfile, &mainheader63);
    if(fp2==NULL) {
      fprintf(stderr, "Error: cannot write ECAT file.\n");
      ecat7EmptyMatlist(&mlist7); ecat63EmptyMatlist(&mlist63);
      fclose(fp); return(11);
    }
    /* Copy the matrices */
    for(m=0; m<mlist63.matrixNr; m++) if(mlist63.matdir[m].matstat==1) {
      blkNr=1+mlist63.matdir[m].endblk-mlist63.matdir[m].strtblk;
      /* Get block number for matrix header and data */
      nxtblk=ecat63Matenter(fp2, mlist63.matdir[m].matnum, blkNr-1);
      if(nxtblk<1) {
        fprintf(stderr, "Error: cannot write ECAT matrix.\n");
        ecat7EmptyMatlist(&mlist7); ecat63EmptyMatlist(&mlist63);
        fclose(fp); fclose(fp2); remove(outfile); return(13);
      }
      if(verbose>3) printf("  m=%d blkNr=%d nxtblk=%d\n", m, blkNr, nxtblk);
      /* Copy each block */
      for(bi=mlist63.matdir[m].strtblk; bi<=mlist63.matdir[m].endblk; bi++) {
        /* Read block */
        fseek(fp, (bi-1)*MatBLKSIZE, SEEK_SET);
        if(ftell(fp)!=(bi-1)*MatBLKSIZE) {
          fprintf(stderr, "Error: cannot find matrix block %d.\n", bi);
          ecat7EmptyMatlist(&mlist7); ecat63EmptyMatlist(&mlist63);
          fclose(fp); fclose(fp2); remove(outfile); return(8);
        }
        if(fread(buf, MatBLKSIZE, 1, fp)<1) {
          fprintf(stderr, "Error: cannot read matrix block %d.\n", bi);
          ecat7EmptyMatlist(&mlist7); ecat63EmptyMatlist(&mlist63);
          fclose(fp); fclose(fp2); remove(outfile); return(9);
        }
        /* Write block */
        fseek(fp2, (nxtblk-1)*MatBLKSIZE, SEEK_SET);
        if(ftell(fp2)!=(nxtblk-1)*MatBLKSIZE) {
          fprintf(stderr, "Error: cannot find matrix block %d for write.\n", nxtblk);
          ecat7EmptyMatlist(&mlist7); ecat63EmptyMatlist(&mlist63);
          fclose(fp); fclose(fp2); remove(outfile); return(15);
        }
        if(fwrite(buf, 1, MatBLKSIZE, fp2)<1) {
          fprintf(stderr, "Error: cannot write matrix block %d.\n", nxtblk);
          ecat7EmptyMatlist(&mlist7); ecat63EmptyMatlist(&mlist63);
          fclose(fp); fclose(fp2); remove(outfile); return(16);
        }
        nxtblk++;
      }
    }
  }
   
  /* Close files */
  fclose(fp); fclose(fp2);
  ecat7EmptyMatlist(&mlist7); ecat63EmptyMatlist(&mlist63);

  if(verbose>0) fprintf(stdout, "done.\n");

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
