/** @file lmlist.c
 *  @brief List the matrices of an ECAT 6 or 7 file.
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include <time.h>
/*****************************************************************************/
#include "libtpcmisc.h"
#include "libtpcimgio.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "List the matrices of an ECAT 6.3 or 7.x file.",
  " ",
  "Usage: @P [Options] ecatfile",
  " ",
  "Options:",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "Example:",
  "  @P s2345dy1.v",
  " ",
  "See also: esplit, efixplnr, imgdim, lmhdr, lshdr, ecat2ana, imgmlist",
  " ",
  "Keywords: ECAT, matrixlist, header, tool",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int                     ai, help=0, version=0, verbose=1;
  int                     ret;
  char                   *cptr, ecatfile[FILENAME_MAX];
  FILE                   *fp;
  int                     ecat_format=0; // 6 or 7 
  ECAT63_mainheader       e6_mainheader;
  ECAT7_mainheader        e7_mainheader;
  static MATRIXLIST       e6_mlist;
  static ECAT7_MATRIXLIST e7_mlist;


  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  ecatfile[0]=(char)0;
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') { /* options */
    cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(cptr==NULL) continue;
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    fprintf(stderr, "Error: invalid option %s\n", argv[ai]);
    return(1);
  } else break;
  
  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  /* Process other arguments, starting from the first non-option */
  for(; ai<argc; ai++) {
    if(!ecatfile[0]) {
      strcpy(ecatfile, argv[ai]); continue;
    }
    fprintf(stderr, "Error: invalid argument '%s'\n", argv[ai]);
    return(1);
  }

  /* Is something missing? */
  if(!ecatfile[0]) {
    fprintf(stderr, "Error: missing command-line argument; use option --help\n");
    return(1);
  }


  /* In verbose mode print arguments and options */
  if(verbose>1) {
    printf("ecatfile := %s\n", ecatfile);
  }


  /*
   *  Open ECAT file
   */
  if((fp=fopen(ecatfile, "rb")) == NULL) {
    fprintf(stderr, "Error: cannot read file %s\n", ecatfile); 
    return(2);
  }


  /*
   *  Read main header
   */
  ecat_format=0;
  /* Try to read ECAT 7.x main header */
  ret=ecat7ReadMainheader(fp, &e7_mainheader);
  if(ret) {
    fprintf(stderr, "Error: unsupported file format (%s)\n", ecatfile);
    fclose(fp); return(3);
  }
  /* If header could be read, check for magic number */
  if(strncmp(e7_mainheader.magic_number, ECAT7V_MAGICNR, 7)==0) {
    ecat_format=7;
  } else { // maybe this is ECAT 6.3
    ret=ecat63ReadMainheader(fp, &e6_mainheader);
    if(ret==0) ecat_format=6;
  }
  if(ecat_format==0) {
    fprintf(stderr, "Error: unsupported file format (%s)\n", ecatfile);
    fclose(fp); return(3);  
  }
  if(verbose>1) printf("ecat_format := %d\n", ecat_format);
  if(verbose>5) {
    if(ecat_format==6)
      ecat63PrintMainheader(&e6_mainheader, stdout);
    else
      ecat7PrintMainheader(&e7_mainheader, stdout);
    printf("\n");
  }

  /*
   *  Read matrix list
   */
  if(ecat_format==6) {
    ecat63InitMatlist(&e6_mlist);
    ret=ecat63ReadMatlist(fp, &e6_mlist, verbose-1);
    if(ret) {
      fprintf(stderr, "Error (%d): cannot read matrix list.\n", ret);
      fclose(fp); return 3;
    }
    if(e6_mlist.matrixNr<=0) {
      fprintf(stderr, "Error: matrix list is empty.\n");
      fclose(fp); return 3;
    }
  } else {
    ecat7InitMatlist(&e7_mlist);
    ret=ecat7ReadMatlist(fp, &e7_mlist, verbose-1);
    if(ret) {
      fprintf(stderr, "Error (%d): cannot read matrix list.\n", ret);
      fclose(fp); return(4);
    }
    if(e7_mlist.matrixNr<=0) {
      fprintf(stderr, "Error: matrix list is empty.\n");
      fclose(fp); return(4);
    }
  }

  /*
   *  Close ECAT file
   */
  fclose(fp);

  /*
   *  Print matrix list, and free memory
   */
  if(ecat_format==6) {
    ecat63PrintMatlist(&e6_mlist);
    ecat63EmptyMatlist(&e6_mlist);
  } else {
    ecat7PrintMatlist(&e7_mlist);
    ecat7EmptyMatlist(&e7_mlist);
  }
  printf("\n");

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
