/** @file lshdr.c
 *  @brief Lists the contents of subheader(s) in ECAT file.
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include <unistd.h>
#include <time.h>
/*****************************************************************************/
#include "libtpcmisc.h"
#include "libtpcimgio.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "List the subheader(s) of an ECAT 6 or 7 file.",
  " ",
  "Usage: @P [Options] ecatfile",
  " ",
  "Options:",
  " -plane=<plane>",
  "     List only specified plane; by default all planes in 2D data.",
  " -frame=<frame>",
  "     List only specified frame; by default all frames.",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "Example:",
  "     @P -frame=1 s2345dy1.img",
  " ",
  "See also: lmhdr, lmlist, eframe, egetstrt, imgunit, e7evhdr",
  " ",
  "Keywords: ECAT, header",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int                 ai, help=0, version=0, verbose=1;
  int                 ret;
  int                 frame=-1, plane=-1;
  char               *cptr, ecatfile[FILENAME_MAX];
  FILE               *fp;
  ECAT63_mainheader   e6_mainheader;
  ECAT7_mainheader    e7_mainheader;
  int                 ecat_format=0; // 6 or 7 


  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  ecatfile[0]=(char)0;
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') { /* options */
    cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(cptr==NULL) continue;
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    if(strncasecmp(cptr, "PLANE=", 6)==0) {
      plane=atoi(cptr+6); if(plane>0) continue;
    } else if(strncasecmp(cptr, "FRAME=", 6)==0) {
      frame=atoi(cptr+6); if(frame>0) continue;
    }
    fprintf(stderr, "Error: invalid option %s\n", argv[ai]);
    return(1);
  } else break;
  
  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  /* Process other arguments, starting from the first non-option */
  for(; ai<argc; ai++) {
    if(!ecatfile[0]) {
      strcpy(ecatfile, argv[ai]); continue;
    } else if(frame<0) {
      frame=atoi(argv[ai]); if(frame>0) continue;
    } else if(plane<0) {
      plane=atoi(argv[ai]); if(plane>0) continue;
    }
    fprintf(stderr, "Error: invalid argument '%s'\n", argv[ai]);
    return(1);
  }


  /* Is something missing? */
  if(!ecatfile[0]) {
    fprintf(stderr, "Error: missing command-line argument; try %s --help\n", argv[0]);
    return(1);
  }


  /* In verbose mode print arguments and options */
  if(verbose>1) {
    printf("ecatfile := %s\n", ecatfile);
    printf("plane := %d\n", plane);
    printf("frame := %d\n", frame);
  }

  if(verbose>2) ECAT7_TEST=ECAT63_TEST=verbose-2;


  /*
   *  Open ECAT file
   */
  if((fp=fopen(ecatfile, "rb")) == NULL) {
    fprintf(stderr, "Error: cannot read file %s\n", ecatfile); 
    return(2);
  }

  /*
   *  Read main header
   */
  ecat_format=0;
  /* Try to read ECAT 7.x main header */
  ret=ecat7ReadMainheader(fp, &e7_mainheader);
  if(ret) {
    fprintf(stderr, "Error: unsupported file format (%s)\n", ecatfile);
    fclose(fp); return(3);
  }
  /* If header could be read, check for magic number */
  if(strncmp(e7_mainheader.magic_number, ECAT7V_MAGICNR, 7)==0) {
    ecat_format=7;
  } else { // maybe this is ECAT 6.3
    ret=ecat63ReadMainheader(fp, &e6_mainheader);
    if(ret==0) ecat_format=6;
  }
  if(ecat_format==0) {
    fprintf(stderr, "Error: unsupported file format (%s)\n", ecatfile);
    fclose(fp); return(3);  
  }
  if(verbose>1) printf("ecat_format := %d\n", ecat_format);
  if(verbose>5) {
    if(ecat_format==6)
      ecat63PrintMainheader(&e6_mainheader, stdout);
    else
      ecat7PrintMainheader(&e7_mainheader, stdout);
    printf("\n");
  }


  /*
   *  Read and print subheaders
   */
  if(ecat_format==6) {
    if(verbose>1) {
      char tmp[32];
      if(e6_mainheader.file_type==1) strcpy(tmp, "sinogram");
      else if(e6_mainheader.file_type==2) strcpy(tmp, "image");
      else if(e6_mainheader.file_type==3) strcpy(tmp, "attenuation");
      else if(e6_mainheader.file_type==4) strcpy(tmp, "normalization");
      else strcpy(tmp, "unknown");
      fprintf(stdout, "file_type := %d (%s)\n", e6_mainheader.file_type, tmp);
    }
    ret=ecat6PrintSubheader(e6_mainheader, fp, plane, frame, stdout);
  } else {
    if(verbose>1) fprintf(stdout, "file_type := %d (%s)\n",
      e7_mainheader.file_type, ecat7filetype(e7_mainheader.file_type) );
    ret=ecat7PrintSubheader(e7_mainheader, fp, plane, frame, stdout);
  }
  fclose(fp);
  printf("\n");

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
