/** @file imgweigh.c
 *  @brief Constructs a SIF based on dynamic PET image or sinogram, to be used 
     for weighting.
 *  @details Application name was previously cti2sif. 
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <ctype.h>
#include <math.h>
#include <time.h>
/*****************************************************************************/
#include "libtpcmisc.h"
#include "libtpcimgio.h"
#include "libtpcimgp.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Constructs a Scan Information File (SIF) from specified dynamic image",
  "in ECAT, Analyze, or NIfTI format, or sinogram in ECAT format,",
  "for providing weighting information to modelling software.",
  "SIF can also be used with Analyze and NIfTI image formats to retain",
  "the frame time information.", 
  "Note that the count values in the resulting SIF file are not the same as",
  "the values in SIF file retrieved from the scanner data, but similar",
  "enough for usual purposes.",
  " ",
  "Usage: @P [-Options] petfile sif",
  " ",
  "Options:",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "Example:",
  "  @P ub7761dy1.v ub7761dy1.sif",
  " ",
  "See also: eframe, tacweigh, imgthrs, imghead, tacframe, simframe",
  " ",
  "Keywords: image, SIF, weighting, time, ECAT, NIfTI, Analyze",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int        ai, help=0, version=0, verbose=1;
  int        ret;
  char      *cptr, petfile[FILENAME_MAX], siffile[FILENAME_MAX];

  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  petfile[0]=siffile[0]=(char)0;
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') { /* options */
    cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(cptr==NULL) continue;
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    /* We should not be here */
    fprintf(stderr, "Error: invalid option '%s'.\n", argv[ai]);
    return(1);
  } else break;
  
  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}
  
  /* Process other arguments, starting from the first non-option */
  if(ai<argc) {strlcpy(petfile, argv[ai], FILENAME_MAX); ai++;}
  if(ai<argc) {strlcpy(siffile, argv[ai], FILENAME_MAX); ai++;}
  if(ai<argc) {fprintf(stderr, "Error: too many arguments.\n"); return(1);}

  /* Is something missing or wrong? */
  if(!siffile[0]) {
    fprintf(stderr, "Error: missing command-line argument; use option --help\n");
    return(1);
  }
  if(strcasecmp(petfile, siffile)==0) {
    fprintf(stderr, "Error: check the output filename.\n");
    return(1);
  }


  /* In verbose mode print arguments and options */
  if(verbose>1) {
    printf("petfile := %s\n", petfile);
    printf("siffile := %s\n", siffile);
  }
  if(verbose>3) IMG_TEST=verbose-3; else IMG_TEST=0;
  if(verbose>5) {ECAT7_TEST=ECAT63_TEST=verbose>5;}



  /* Read the PET image or sinogram */
  IMG img; imgInit(&img);
  if(verbose>0) printf("reading PET data\n");
  ret=imgRead(petfile, &img);
  if(ret) {
    fprintf(stderr, "Error: %s\n", imgStatus(ret));
    if(verbose>0) imgInfo(&img);
    imgEmpty(&img); return(2);
  }
  if(verbose>1) printf("frame_nr := %d\n", img.dimt);

  /* check the file type */
  if(img.type!=IMG_TYPE_IMAGE && img.type!=IMG_TYPE_RAW) {
    fprintf(stderr, "Error: %s is not an image or scan file.\n", petfile);
    imgEmpty(&img);
    return(2);
  }

  /* Compute the SIF data */
  SIF sif; sifInit(&sif);
  if(verbose>0) printf("computing SIF\n");
  ret=img2sif(&img, &sif, 1, 1, 2, 2);
  if(ret) {
    fprintf(stderr, "Error: cannot create SIF from %s.\n", petfile);
    imgEmpty(&img); sifEmpty(&sif); return(3);
  }
  if(verbose>2) sifPrint(&sif);

  // Close the image
  imgEmpty(&img);

  /*
   *  Save the SIF file
   */
  if(verbose>1) printf("writing SIF\n");
  if(sifWrite(&sif, siffile)) {
    fprintf(stderr, "Error in writing '%s': %s\n", siffile, siferrmsg);
    sifEmpty(&sif); return(11);
  }
  if(verbose>0) fprintf(stderr, "SIF data written in %s\n", siffile);

  /* Prepare to quit */
  sifEmpty(&sif);

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
