#!/bin/bash
#: Title      : test_imglhk3
#: Date       : 2018-03-22
#: Author     : "Vesa Oikonen" <vesa.oikonen@utu.fi>
#: Options    : None

# Set the name of executable to test
if [ -n "${OS+1}" ] && [ "$OS" = "Windows_NT" ]; then
EXT=.exe;
else
EXT=;
fi

PROGRAM=../../imglhk3$EXT;
if [ ! -f $PROGRAM ]; then
  printf "Failed: executable does not exist.\n"
  exit 1
fi

printf "=====================================================================\n"
printf "creating test data for %s\n" $PROGRAM
printf "=====================================================================\n"


printf "\n verifying that required data exists \n"

if [ ! -f fdg_arterial_plasma.fit ] || [ ! -f shortframes.sif ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
if [ ! -f fdg.par ] || [ ! -f fdgvb.par ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi

if [ ! -f fdgplasma.tac ]; then
  rm -f fdg.v ana/fdg.* nii/fdg.*
  printf "Making 90 min FDG PTAC with 0.05 min (3 s) sample intervals \n"
  fit2dat -c=0,90,0.1 fdg_arterial_plasma.fit fdgplasma.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f fdg.v ] || [ ! -f fdgmask.v ]; then
  rm -f fdgmask.v correct_fdg_k3.v
  rm -f ana/fdg.* nii/fdg.*
  simcirc -format=float 32 16 circ.bin
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  flat2img -scanner=hrrt circ.bin circ.v 1 1 32 32
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  imgmove circ.v -8 -6 1 circ1.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  imgmove circ.v -4 -2 1 circ2.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  imgadd fdgmask.v circ1.v circ.v circ2.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  printf "Simulating FDG TTACs with K1=0.09, K1/k2=0.41, k3=0.11 \n"
  printf "Ki=K1*k3/(k2+k3)= 0.0300444 \n"
  printf "DV=K1/(k2+k3)=0.273131 \n"
  sim_3tcm fdg.par fdgplasma.tac none fdg.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  simframe fdg.tac shortframes.sif fdgtissue.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  dft2img -tpl=fdgmask.v fdgtissue.tac fdg.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi
  
if [ ! -f correct_fdg_k3.v ] || [ ! -f correct_fdg_k1.v ] || [ ! -f correct_fdg_k2.v ] || [ ! -f correct_fdg_dv.v ] || [ ! -f correct_fdg_ki.v ]; then
  printf "Making parametric images for comparison \n"
  printf "1 0.09\n# timeunit := min\n# unit := ml/(min*ml)\n" > k1.dat
  printf "1 0.11\n# timeunit := min\n# unit := 1/min\n" > k3.dat
  printf "1 0.2195122\n# timeunit := min\n# unit := 1/min\n" > k2.dat
  printf "1 0.273131\n# timeunit := min\n# unit := unitless\n" > dv.dat
  printf "1 0.0300444\n# timeunit := min\n# unit := ml/(min*ml)\n" > ki.dat
  dft2img -tpl=fdgmask.v k1.dat correct_fdg_k1.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  dft2img -tpl=fdgmask.v k3.dat correct_fdg_k3.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  dft2img -tpl=fdgmask.v k2.dat correct_fdg_k2.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  dft2img -tpl=fdgmask.v dv.dat correct_fdg_dv.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  dft2img -tpl=fdgmask.v ki.dat correct_fdg_ki.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi


if [ ! -f ana/fdg.img ] || [ ! -f ana/fdg.hdr ] || [ ! -f ana/fdg.sif ]; then
  printf "\n Making Analyze images \n"
  ecat2ana -sif -o=ana fdg.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  printf "\n ok \n"
fi

if [ ! -f nii/fdg.nii ] || [ ! -f nii/fdg.sif ]; then
  printf "\n Making NIfTI images \n"
  ecat2nii -sif -o=nii fdg.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  printf "\n ok \n"
fi


  
if [ ! -f fdgvb.v ] || [ ! -f fdgvbmask.v ]; then
  rm -f fdgvbmask.v correct_fdgvb_k3.v
  simcirc -format=float 64 60 circ.bin
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  flat2img -scanner=hrrt circ.bin circ.v 1 1 64 64
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  simcirc -format=float 64 40 circ2.bin
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  flat2img -scanner=hrrt circ2.bin circ2.v 1 1 64 64
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  imgmove circ2.v -8 -4
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  imgcalc circ.v + circ2.v ring1.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  simcirc -format=float 64 20 circ3.bin
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  flat2img -scanner=hrrt circ3.bin circ3.v 1 1 64 64
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  imgcalc ring1.v + circ3.v fdgvbmask.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  img2tif -rb fdgvbmask.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  printf "Simulating FDG TTACs with K1=0.09, K1/k2=0.41, k3=0.11, Vb=0 and Vb=0.05 \n"
  printf "Ki=K1*k3/(k2+k3)= 0.0300444 \n"
  printf "DV=K1/(k2+k3)=0.273131 \n"
  sim_3tcm fdgvb.par fdgplasma.tac fdgplasma.tac fdgvb.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacadd fdgvb.tac fdgplasma.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  simframe fdgvb.tac shortframes.sif fdgvbtissue.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  dft2img -tpl=fdgvbmask.v fdgvbtissue.tac fdgvb.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  img2tif -rb fdgvb.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f correct_fdgvb_k3.v ] || [ ! -f correct_fdgvb_vb.v ] || [ ! -f correct_fdgvb_k1.v ]; then
  printf "Making parametric images for comparison \n"
  printf "1 0 0.05 1.0\n# timeunit := min\n# unit := unitless\n" > vb.dat
  printf "1 0.09 0.09 0\n# timeunit := min\n# unit := ml/(min*ml)\n" > k1.dat
  printf "1 0.11 0.11 0\n# timeunit := min\n# unit := 1/min\n" > k3.dat
  dft2img -tpl=fdgvbmask.v vb.dat correct_fdgvb_vb.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  dft2img -tpl=fdgvbmask.v k1.dat correct_fdgvb_k1.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  dft2img -tpl=fdgvbmask.v k3.dat correct_fdgvb_k3.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f correct_fdgvb_k2.v ] || [ ! -f correct_fdgvb_dv.v ] || [ ! -f correct_fdgvb_ki.v ]; then
  printf "Making parametric images for comparison \n"
  printf "1 0.2195122 0.2195122 0\n# timeunit := min\n# unit := 1/min\n" > k2.dat
  printf "1 0.273131 0.273131 0\n# timeunit := min\n# unit := unitless\n" > dv.dat
  printf "1 0.0300444 0.0300444 0\n# timeunit := min\n# unit := ml/(min*ml)\n" > ki.dat
  dft2img -tpl=fdgvbmask.v k2.dat correct_fdgvb_k2.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  dft2img -tpl=fdgvbmask.v dv.dat correct_fdgvb_dv.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  dft2img -tpl=fdgvbmask.v ki.dat correct_fdgvb_ki.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f correct_fdgvb_k2k3.v ]; then
  printf "Making parametric images for comparison \n"
  printf "1 0.3295122 0.3295122 0\n# timeunit := min\n# unit := 1/min\n" > k2k3.dat
  dft2img -tpl=fdgvbmask.v k2k3.dat correct_fdgvb_k2k3.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi



printf "\n\n"
printf "=====================================================================\n"
printf "testing %s\n" $PROGRAM
printf "=====================================================================\n"



printf "\n 1.0.0 \n"
printf " Test case 1: Noiseless image simulated with FGD parameters with zero Vb. \n"
printf " Test case 2: Save all parametric images except Vb and k2+k3. \n"
printf " Expected result: Correct parametric images are calculated. \n\n"

rm -f k1.v k2.v k3.v dv.v ki.v

$PROGRAM -k1=k1.v -k2=k2.v -dv=dv.v -ki=ki.v fdgplasma.tac fdg.v k3.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

img2dft k3.v fdgmask.v k3.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
img2dft k1.v fdgmask.v k1.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
img2dft k2.v fdgmask.v k2.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
img2dft ki.v fdgmask.v ki.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
img2dft dv.v fdgmask.v dv.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n--- Results for k3: \n"
taclist k3.dft
printf "\n--- Results for K1: \n"
taclist k1.dft
printf "\n--- Results for k2: \n"
taclist k2.dft
printf "\n--- Results for Ki: \n"
taclist ki.dft


imgmatch -frames=n -rel=0.1 -abs=0.0001 correct_fdg_k3.v k3.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

imgmatch -frames=n -rel=0.1 -abs=0.0001 correct_fdg_k1.v k1.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
imgmatch -frames=n -rel=0.1 -abs=0.0001 correct_fdg_k2.v k2.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
imgmatch -frames=n -rel=0.1 -abs=0.0001 correct_fdg_ki.v ki.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
imgmatch -frames=n -rel=0.1 -abs=0.0001 correct_fdg_dv.v dv.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 1.0.1 \n"
printf " Test case: Same as 1.0.0 but Analyze format. \n"
printf " Expected result: Correct k3 image is calculated. \n\n"

rm -f ana/output.*

$PROGRAM fdgplasma.tac ana/fdg ana/output
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

imgmatch -frames=n -rel=0.1 -abs=0.0001 correct_fdg_k3.v ana/output
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 1.0.2 \n"
printf " Test case: Same as 1.0.0 but NIfTI format. \n"
printf " Expected result: Correct k3 image is calculated. \n\n"

rm -f nii/output.*

$PROGRAM fdgplasma.tac nii/fdg nii/output
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

imgmatch -frames=n -rel=0.1 -abs=0.0001 correct_fdg_k3.v nii/output
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 1.1.0 \n"
printf " Test case 1: Noiseless image simulated with FGD parameters with zero and 5 percent Vb. \n"
printf " Test case 2: Save all parametric images including Vb. \n"
printf " Expected result: Correct parametric images are calculated. \n\n"

rm -f k1.v k2.v k3.v dv.v ki.v vb.v k2k3.v

$PROGRAM -vb=vb.v -k1=k1.v -k2=k2.v -dv=dv.v -k2k3=k2k3.v -ki=ki.v fdgplasma.tac fdgvb.v k3.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

img2dft k3.v fdgvbmask.v k3.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
img2dft vb.v fdgvbmask.v vb.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
img2dft k1.v fdgvbmask.v k1.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
img2dft k2.v fdgvbmask.v k2.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
img2dft k2k3.v fdgvbmask.v k2k3.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
img2dft ki.v fdgvbmask.v ki.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
img2dft dv.v fdgvbmask.v dv.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n--- Results for k3: \n"
taclist k3.dft
printf "\n--- Results for Vb: \n"
taclist vb.dft
printf "\n--- Results for K1: \n"
taclist k1.dft
printf "\n--- Results for k2: \n"
taclist k2.dft
printf "\n--- Results for k2+k3: \n"
taclist k2k3.dft
printf "\n--- Results for Ki: \n"
taclist ki.dft
printf "\n--- Results for Vd: \n"
taclist dv.dft

printf "\n verifying k3 image... \n"
imgmatch -frames=n -rel=0.1 -abs=0.0001 correct_fdgvb_k3.v k3.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n verifying Vb image... \n"
imgmatch -frames=n -rel=0.1 -abs=0.001 correct_fdgvb_vb.v vb.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n verifying K1 image... \n"
imgmatch -frames=n -rel=0.1 -abs=0.0005 correct_fdgvb_k1.v k1.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

printf "\n verifying Ki image... \n"
imgmatch -frames=n -rel=0.1 -abs=0.0001 correct_fdgvb_ki.v ki.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

printf "\n verifying Vd image... \n"
imgmatch -frames=n -rel=0.1 -abs=0.0001 correct_fdgvb_dv.v dv.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

printf "\n verifying k2 image... \n"
imgmatch -frames=n -rel=0.1 -abs=0.001 correct_fdgvb_k2.v k2.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n verifying k2+k3 image... \n"
imgmatch -frames=n -rel=0.1 -abs=0.001 correct_fdgvb_k2k3.v k2k3.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n===================================================================\n"





printf "\n 2.0.0 \n"
printf " Test case: Reduce fit time using option -end. \n"
printf " Expected result: Data inconsistency after fit time does not affect results. \n\n"

imgmove fdgvb.v -10 +10 36-38 movement.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

rm -f k1.v k2.v k3.v dv.v ki.v vb.v
rm -f output.ift

$PROGRAM -end=60 -vb=vb.v -k1=k1.v -k2=k2.v -dv=dv.v -ki=ki.v fdgplasma.tac movement.v k3.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

printf "\n Checking frame length in parametric image... \n"
eframe -ift k3.v output.ift
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.001 output.ift frame_start[1] 0
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.001 output.ift frame_duration[1] 60
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

img2dft k3.v fdgvbmask.v k3.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
img2dft vb.v fdgvbmask.v vb.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
img2dft k1.v fdgvbmask.v k1.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
img2dft k2.v fdgvbmask.v k2.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
img2dft ki.v fdgvbmask.v ki.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
img2dft dv.v fdgvbmask.v dv.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n--- Results for k3: \n"
taclist k3.dft
printf "\n--- Results for Vb: \n"
taclist vb.dft
printf "\n--- Results for K1: \n"
taclist k1.dft
printf "\n--- Results for k2: \n"
taclist k2.dft
printf "\n--- Results for Ki: \n"
taclist ki.dft
printf "\n--- Results for Vd: \n"
taclist dv.dft


printf "\n verifying k3 image... \n"
imgmatch -frames=n -rel=0.1 -abs=0.001 correct_fdgvb_k3.v k3.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n verifying Vb image... \n"
imgmatch -frames=n -rel=0.1 -abs=0.001 correct_fdgvb_vb.v vb.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n verifying K1 image... \n"
imgmatch -frames=n -rel=0.1 -abs=0.0005 correct_fdgvb_k1.v k1.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

printf "\n verifying Ki image... \n"
imgmatch -frames=n -rel=0.1 -abs=0.0001 correct_fdgvb_ki.v ki.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

printf "\n verifying Vd image... \n"
imgmatch -frames=n -rel=0.1 -abs=0.001 correct_fdgvb_dv.v dv.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

printf "\n verifying k2 image... \n"
imgmatch -frames=n -rel=0.1 -abs=0.002 correct_fdgvb_k2.v k2.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"




printf "\n===================================================================\n"
printf "   All passed! \n"
printf "===================================================================\n"
exit 0



printf "\n 2.1.0 \n"
printf " Test case: Alternative k3 estimation method. \n"
printf " Expected result: Correct parametric images are calculated. \n\n"

rm -f k1.v k2.v k3.v dv.v ki.v vb.v

$PROGRAM -alt -d3 -vb=vb.v -k1=k1.v -k2=k2.v -dv=dv.v -ki=ki.v fdgplasma.tac fdgvb.v k3.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

img2dft k3.v fdgvbmask.v k3.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
img2dft vb.v fdgvbmask.v vb.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
img2dft k1.v fdgvbmask.v k1.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
img2dft k2.v fdgvbmask.v k2.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
img2dft ki.v fdgvbmask.v ki.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
img2dft dv.v fdgvbmask.v dv.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n--- Results for k3: \n"
taclist k3.dft
printf "\n--- Results for Vb: \n"
taclist vb.dft
printf "\n--- Results for K1: \n"
taclist k1.dft
printf "\n--- Results for k2: \n"
taclist k2.dft
printf "\n--- Results for Ki: \n"
taclist ki.dft
printf "\n--- Results for Vd: \n"
taclist dv.dft

printf "\n verifying k3 image... \n"
imgmatch -frames=n -rel=0.1 -abs=0.0001 correct_fdgvb_k3.v k3.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n verifying Vb image... \n"
imgmatch -frames=n -rel=0.1 -abs=0.001 correct_fdgvb_vb.v vb.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n verifying K1 image... \n"
imgmatch -frames=n -rel=0.1 -abs=0.0005 correct_fdgvb_k1.v k1.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

printf "\n verifying Ki image... \n"
imgmatch -frames=n -rel=0.1 -abs=0.0001 correct_fdgvb_ki.v ki.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

printf "\n verifying Vd image... \n"
imgmatch -frames=n -rel=0.1 -abs=0.0001 correct_fdgvb_dv.v dv.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

printf "\n verifying k2 image... \n"
imgmatch -frames=n -rel=0.1 -abs=0.001 correct_fdgvb_k2.v k2.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

