/** @file imgfrdyn.c
 *  @brief Estimate the dynamics in dynamic PET image by looking the difference between frame values.
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include <unistd.h>
#include <time.h>
/*****************************************************************************/
#include "libtpcmisc.h"
#include "libtpcimgio.h"
#include "libtpcimgp.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Estimate the dynamics in PET image by counting the numbers of frames where",
  "pixel values are increasing or decreasing as compared to the previous frame.",
  "An image containing number of increases minus number of decreases is saved.",
  " ",
  "Usage: @P [Options] imgfile dynimgfile",
  " ",
  "Options:",
  " -dmi | -decr | -incr",
  "     Instead of the default operation, calculate the number of decreases",
  "     minus increases (-dmi), number of increases (-incr), or number of",
  "     decreases (-decr).",
  " -noneg",
  "     Negative numbers are set to zero.",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "See also: imgfrdif, imgledif, imgpeak, imgmask, imgthrs, img2tif",
  " ",
  "Keywords: image, dynamics, time, mask",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int     ai, help=0, version=0, verbose=1;
  char    imgfile[FILENAME_MAX], outfile[FILENAME_MAX];
  int     mode=0; // 0=incr-decr, 1=decr-incr, 2=incr, 3=decr
  int     workNegatives=0; // 0=leave, 1=set to zero 
  int     ret;


  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  imgfile[0]=outfile[0]=(char)0;
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') { /* options */
    char *cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(cptr==NULL) continue;
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    if(strcasecmp(cptr, "IMD")==0) {
      mode=0; continue;
    } else if(strcasecmp(cptr, "DMI")==0) {
      mode=1; continue;
    } else if(strcasecmp(cptr, "INCR")==0) {
      mode=2; continue;
    } else if(strcasecmp(cptr, "DECR")==0) {
      mode=3; continue;
    } else if(strncasecmp(cptr, "NONEGATIVES", 5)==0) {
      workNegatives=1; continue;
    }
    fprintf(stderr, "Error: invalid option '%s'.\n", argv[ai]); 
    return(1);
  } else break;

  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}
  
  /* Process other arguments, starting from the first non-option */
  if(ai<argc) {strlcpy(imgfile, argv[ai], FILENAME_MAX); ai++;}
  if(ai<argc) {strlcpy(outfile, argv[ai], FILENAME_MAX); ai++;}
  if(ai<argc) {fprintf(stderr, "Error: too many arguments.\n"); return(1);}

  /* Is something missing or wrong? */
  if(!outfile[0]) {
    fprintf(stderr, "Error: missing command-line argument; use option --help\n");
    return(1);
  }

  /* In verbose mode print arguments and options */
  if(verbose>1) {
    printf("imgfile := %s\n", imgfile);
    printf("outfile := %s\n", outfile);
    printf("mode := %d\n", mode);
    printf("workNegatives := %d\n", workNegatives);
    fflush(stdout);
  }


  /*
   *  Read dynamic image
   */
  if(verbose>0) {printf("reading dynamic image %s\n", imgfile); fflush(stdout);}
  IMG img; imgInit(&img);
  ret=imgRead(imgfile, &img);
  if(ret) {
    fprintf(stderr, "Error: %s\n", img.statmsg);
    if(verbose>1) printf("ret := %d\n", ret);
    return(2);
  }
  if(img.dimt<2) {
    fprintf(stderr, "Error: %s contains only 1 time frame.\n", imgfile);
    imgEmpty(&img); return(2);
  }
  if(verbose>0) {
    fprintf(stdout, "  image contains %d frames and %d planes.\n", img.dimt, img.dimz);
    fflush(stdout);
  }
  if(imgNaNs(&img, 1)>0)
    if(verbose>0) fprintf(stderr, "Warning: missing pixel values.\n");


  /*
   *  Calculate the numbers of increasing and decreasing frames
   */
  if(verbose>0) {printf("computing frame differences\n"); fflush(stdout);}
  IMG iimg; imgInit(&iimg);
  IMG dimg; imgInit(&dimg);
  ret=imgGetFrameDyn(&img, &iimg, &dimg, verbose-1);
  imgEmpty(&img);
  if(ret) {
    fprintf(stderr, "Error: cannot compute frame differences.\n");
    imgEmpty(&dimg); imgEmpty(&iimg);
    return(4);
  }
  /* Based on the mode, calculate final number */
  if(mode==0) {
    if(verbose>1) {printf("computing increases minus decreases\n"); fflush(stdout);}
    for(int zi=0; zi<dimg.dimz; zi++)
      for(int yi=0; yi<dimg.dimy; yi++)
        for(int xi=0; xi<dimg.dimx; xi++)
          iimg.m[zi][yi][xi][0]-=dimg.m[zi][yi][xi][0];
  } else if(mode==1) {
    if(verbose>1) {printf("computing decreases minus increases\n"); fflush(stdout);}
    for(int zi=0; zi<dimg.dimz; zi++)
      for(int yi=0; yi<dimg.dimy; yi++)
        for(int xi=0; xi<dimg.dimx; xi++)
          iimg.m[zi][yi][xi][0]=dimg.m[zi][yi][xi][0]-iimg.m[zi][yi][xi][0];
  } else if(mode==3) {
    for(int zi=0; zi<dimg.dimz; zi++)
      for(int yi=0; yi<dimg.dimy; yi++)
        for(int xi=0; xi<dimg.dimx; xi++)
          iimg.m[zi][yi][xi][0]=dimg.m[zi][yi][xi][0];
  }
  imgEmpty(&dimg);

  /* If requested, set negative difference to zero */
  if(workNegatives==1) imgCutoff(&iimg, 0.0, 1);

  /*
   *  Save result image
   */
  if(verbose>1) {printf("writing result image\n"); fflush(stdout);}
  ret=imgWrite(outfile, &iimg);
  if(ret) {
    fprintf(stderr, "Error: %s\n", iimg.statmsg);
    imgEmpty(&iimg);
    return(11);
  }
  if(verbose>0) fprintf(stdout, "Image %s saved.\n", outfile);

  imgEmpty(&iimg);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
