/** @file imgmax.c
 *  @brief Find maximum pixel value in PET image file(s).
 *  @details Previous application name ecatmax. 
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include <unistd.h>
#include <time.h>
//#include <float.h>
/*****************************************************************************/
#include "libtpcmisc.h"
#include "libtpcimgio.h"
#include "libtpcimgp.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Find maximum pixel value in PET image file(s) in ECAT 6.3 or 7.x, NIfTI-1,",
  "or Analyze 7.5 format.",
  " ",
  "Usage: @P [Options] imgfile(s)",
  " ",
  "Options:",
  " -min",
  "     Minimum values are printed instead of maximum values.", 
  " -both",
  "     Both max and min values are printed.",
  " -clean",
  "     Only the max and/or min value of all specified files is printed.",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "Example: get the maximum and minimum values inside all ECAT 7 image files",
  "in the current folder:"
  "  @P -both *.v",
  " ",
  "See also: imgmaxp, imgunit, lmhdr, lshdr, eframe, img2tif, img2flat",
  " ",
  "Keywords: image, ECAT, NIfTI, Analyze, max, min",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int     ai, help=0, version=0, verbose=1;
  int     ret, fileNr=0, firstfile=0;
  int     cleanOutput=0;
  int     differing_units=0;
  int     print_what=0; // 0=max, 1=min, 2=both
  char    imgfile[FILENAME_MAX], *cptr=NULL, imgunit[128];
  IMG     img;
  float   imgmax=-FLT_MAX, allmax=-FLT_MAX;
  float   imgmin=FLT_MAX,  allmin=FLT_MAX;


  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  imgfile[0]=(char)0; strcpy(imgunit, "");
  imgInit(&img);
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') { /* options */
    cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(cptr==NULL) continue;
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    if(strncasecmp(cptr, "CLEAN", 1)==0) {
      cleanOutput=1; continue;
    } else if(strncasecmp(cptr, "MINIMUM", 3)==0) {
      print_what=1; continue;
    } else if(strncasecmp(cptr, "BOTH", 2)==0) {
      print_what=2; continue;
    }
    fprintf(stderr, "Error: invalid option '%s'.\n", argv[ai]); 
    return(1);
  } else break;
  
  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  /* Process other arguments, starting from the first non-option */
  for(; ai<argc; ai++) {
    if(firstfile==0) firstfile=ai;
    fileNr++;
  }
  /* Did we get all the information that we need? */
  if(fileNr==0) {tpcPrintUsage(argv[0], info, stderr); return(1);}


  /* In verbose mode print options */
  if(verbose>1) {
    printf("cleanOutput := %d\n", cleanOutput);
    printf("fileNr := %d\n", fileNr);
    printf("print_what := %d\n", print_what);
  }

  if(verbose>2) IMG_TEST=ECAT63_TEST=ECAT7_TEST=verbose-2;
  else IMG_TEST=ECAT63_TEST=ECAT7_TEST=0;

  /*
   *  Process each file
   */
  if(verbose>0) printf("processing %d file(s)...\n", fileNr);
  fileNr=0;
  for(ai=firstfile; ai<argc; ai++) {

    strlcpy(imgfile, argv[ai], FILENAME_MAX); 
    imgmax=-FLT_MAX; imgmin=FLT_MAX;

    /* Read Image file */
    if(!cleanOutput) {fprintf(stdout, "%s :\n", imgfile); fflush(stdout);}
    ret=imgRead(imgfile, &img);
    if(ret) {
      fprintf(stderr, "Warning: %s\n", img.statmsg);
      if(verbose>1) printf("ret := %d\n", ret);
      continue;
    }

    /* Get maximum pixel value */
    ret=imgMinMax(&img, &imgmin, &imgmax);
    if(ret) {
      fprintf(stderr, "Warning: cannot read maximum.\n");
      imgEmpty(&img);
      continue;
    }
    cptr=imgUnit(img.unit);
    if(imgunit[0] && strcasecmp(imgunit, cptr)!=0) differing_units=1;
    strcpy(imgunit, imgUnit(img.unit));
    if(!cleanOutput) {
      if(print_what==0 || print_what==2)
        fprintf(stdout, " max := %g %s\n", imgmax, imgunit);
      if(print_what==1 || print_what==2)
        fprintf(stdout, " min := %g %s\n", imgmin, imgunit);
    }
    if(imgmax>allmax) allmax=imgmax;
    if(imgmin<allmin) allmin=imgmin;
    imgEmpty(&img);
    fileNr++;
  } /* next file */
  if(fileNr==0) {
    fprintf(stderr, "Error: no file(s) could be read.\n");
    return(3);
  }


  /*
   *  Print the max of all files
   */
  if(cleanOutput) {
    if(print_what==0 || print_what==2)
      fprintf(stdout, "%g\n", allmax);
    if(print_what==1 || print_what==2)
      fprintf(stdout, "%g\n", allmin);
  } else if(fileNr>1) {
    if(differing_units) strcpy(imgunit, "");
    if(print_what==0 || print_what==2)
      fprintf(stdout, "Maximum of all files := %g %s\n", allmax, imgunit );
    if(print_what==1 || print_what==2)
      fprintf(stdout, "Minimum of all files := %g %s\n", allmin, imgunit );
  }

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
