/** @file imgposv.c
 *  @brief Calculate the volume percentage voxels with positive value.
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include <unistd.h>
#include <time.h>
/*****************************************************************************/
#include "libtpcmisc.h"
#include "libtpcimgio.h"
//#include "libtpcimgp.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Calculate the volume percentage of voxels with positive (>0) value in",
  "a static PET image file in ECAT 6.3 or 7.x, NIfTI-1, or Analyze 7.5 format.",
  " ",
  "Usage: @P [Options] imgfile [resfile]",
  " ",
  "Options:",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "See also: imghist, imgthrs, imgmask, imgmax, imginteg, imgslim",
  " ",
  "Keywords: image, threshold, mask",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int     ai, help=0, version=0, verbose=1;
  char    imgfile[FILENAME_MAX], outfile[FILENAME_MAX];
  int     ret;
  char   *cptr; 


  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  imgfile[0]=outfile[0]=(char)0;
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') { /* options */
    cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(cptr==NULL) continue;
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    fprintf(stderr, "Error: invalid option '%s'.\n", argv[ai]); 
    return(1);
  } else break;
  
  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}
  
  /* Process other arguments, starting from the first non-option */
  if(ai<argc) {strlcpy(imgfile, argv[ai], FILENAME_MAX); ai++;}
  if(ai<argc) {strlcpy(outfile, argv[ai], FILENAME_MAX); ai++;}
  if(ai<argc) {fprintf(stderr, "Error: too many arguments.\n"); return(1);}

  /* Is something missing or wrong? */
  if(!imgfile[0]) {
    fprintf(stderr, "Error: missing command-line argument; use option --help\n");
    return(1);
  }

  /* In verbose mode print arguments and options */
  if(verbose>1) {
    printf("imgfile := %s\n", imgfile);
    if(outfile[0]) printf("outfile := %s\n", outfile);
  }


  /*
   *  Read dynamic image
   */
  if(verbose>0) printf("reading static image %s\n", imgfile);
  IMG img; imgInit(&img);
  ret=imgRead(imgfile, &img);
  if(ret) {
    fprintf(stderr, "Error: %s\n", img.statmsg);
    if(verbose>1) printf("ret := %d\n", ret);
    return(2);
  }
  if(img.dimt!=1) {
    fprintf(stderr, "Error: not a static image.\n");
    imgEmpty(&img); return(2);
  }
  if(verbose>1) { 
    printf("dimx := %d\n", img.dimx);
    printf("dimy := %d\n", img.dimy);
    printf("dimz := %d\n", img.dimz);
  }
  if(imgNaNs(&img, 1)>0)
    if(verbose>0) fprintf(stderr, "Warning: missing pixel values.\n");


  /*
   *  Check each pixel value
   */
  if(verbose>0) printf("counting positive voxels...\n");
  long long total_n=0, posit_n=0;
  int zi, yi, xi;
  for(zi=0; zi<img.dimz; zi++)
    for(yi=0; yi<img.dimy; yi++)
      for(xi=0; xi<img.dimx; xi++) {
        total_n++;
        if(img.m[zi][yi][xi][0]>0.0) posit_n++;
      }
  double volper=100.0*(double)posit_n/(double)total_n;
  imgEmpty(&img);

  /* Report results */
  if(!outfile[0] || verbose>0) {
    printf("positive_volume_percentage := %g\n", volper);
  }
  if(!outfile[0]) return(0);
  /* Save results */
  char tmp[64]; sprintf(tmp, "%g", volper);
  IFT ift; iftInit(&ift);
  ret=iftPut(&ift, "positive_volume_percentage", tmp, NULL, 0);
  if(ret==0) ret=iftWrite(&ift, outfile, 0);
  iftEmpty(&ift);
  if(ret) {
    fprintf(stderr, "Error: cannot save the result.\n");
    return(11);
  }
  if(verbose>0) printf("done.\n");

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
