/** @file imgdelfr.c
 *  @brief Delete specified frames from dynamic PET image.
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <math.h>
#include <time.h>
/*****************************************************************************/
#include "libtpcmisc.h"
#include "libtpcmodel.h"
#include "libtpccurveio.h"
#include "libtpcimgio.h"
#include "libtpcimgp.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Delete specified frames from PET image in ECAT 6.3, 7.x, NIfTI-1, or",
  "Analyze 7.5 format.",
  "Analyze and NIfTI image must have the SIF in the same folder.",
  " ",
  "Usage: @P [Options] image startframe endframe newimage",
  " ",
  "Options:",
  " -zero",
  "     Frames are not actually deleted, but pixel values are set to zero.",
  " -sec",
  "     Start and end times (in sec) are given instead of frame numbers.",
  " -min",
  "     Start and end times (in min) are given instead of frame numbers.",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "See also: esplit, ecatcat, eframe, ecattime, imgbox, imgdelpl, imgdim",
  " ",
  "Keywords: image, cropping, compression, time",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int      ai, help=0, version=0, verbose=1;
  int      doCrop=1; // 0=zeroes; 1=crop
  int      times=-1; // -1=frames; 0=sec, 1=min
  char     petfile[FILENAME_MAX], outfile[FILENAME_MAX];
  int      ret;
  int      fstart, fstop;
  double   tstart, tstop;
  
  
  
  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  petfile[0]=outfile[0]=(char)0;
  /* Get options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') {
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    char *cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(cptr==NULL) continue;
    if(strcasecmp(cptr, "CROP")==0) {
      doCrop=1; continue;
    } else if(strcasecmp(cptr, "ZERO")==0) {
      doCrop=0; continue;
    } else if(strncasecmp(cptr, "SECONDS", 3)==0) {
      times=0; continue;
    } else if(strncasecmp(cptr, "MINUTES", 3)==0) {
      times=1; continue;
    }
    fprintf(stderr, "Error: invalid option '%s'.\n", argv[ai]);
    return(1);
  } else break;
  
  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  /* Process other arguments, starting from the first non-option */
  if(ai<argc) {strlcpy(petfile, argv[ai++], FILENAME_MAX);}
  if(ai<argc) {
    if(times<0) ret=atoi_with_check(argv[ai], &fstart);    
    else if(times==0) ret=atof_with_check(argv[ai], &tstart);    
    else {ret=atof_with_check(argv[ai], &tstart); tstart*=60.0;}
    if(ret!=0) {fprintf(stderr, "Error: invalid cut start '%s'.\n", argv[ai]); return(1);}
    ai++;
  }
  if(ai<argc) {
    if(times<0) ret=atoi_with_check(argv[ai], &fstop);    
    else if(times==0) ret=atof_with_check(argv[ai], &tstop);
    else {ret=atof_with_check(argv[ai], &tstop); tstop*=60.0; times=0;}
    if(ret!=0) {fprintf(stderr, "Error: invalid cut stop '%s'.\n", argv[ai]); return(1);}
    ai++;
  }
  if(ai<argc) {strlcpy(outfile, argv[ai++], FILENAME_MAX);}
  if(ai<argc) {
    fprintf(stderr, "Error: invalid argument '%s'.\n", argv[ai]);
    return(1);
  }

  /* Did we get all the information that we need? */
  if(!outfile[0]) {
    fprintf(stderr, "Error: missing command-line argument; use option --help\n");
    return(1);
  }
  if(times<0 && fstart>fstop) {
    fprintf(stderr, "Error: invalid frame number range.\n");
    return(1);
  }
  if(times>=0 && tstart>=tstop) {
    fprintf(stderr, "Error: invalid frame time range.\n");
    return(1);
  }
  if(strcasecmp(outfile, petfile)==0) {
    fprintf(stderr, "Error: check the output filename.\n");
    return(1);
  }


  /* In verbose mode print arguments and options */
  if(verbose>1) {
    printf("petfile := %s\n", petfile);
    printf("outfile := %s\n", outfile);
    printf("doCrop := %d\n", doCrop);
    if(times<0) {
      printf("cut_start_frame := %d\n", fstart);
      printf("cut_stop_frame := %d\n", fstop);
    } else {
      printf("cut_start_time_in_sec := %g\n", tstart);
      printf("cut_stop_time_in_sec := %g\n", tstop);
    }
  }
  if(verbose>9) IMG_TEST=verbose-9; else IMG_TEST=0;
  

  /*
   *  Read image
   */
  if(verbose>0) fprintf(stdout, "reading image %s\n", petfile);
  IMG img; imgInit(&img);
  ret=imgRead(petfile, &img);
  if(ret) {
    fprintf(stderr, "Error: %s\n", img.statmsg); if(verbose>1) imgInfo(&img);
    return(2);
  }

  /* Check that frame times are available, if frame time range was given */
  if(times>=0 && !imgExistentTimes(&img)) {
    fprintf(stderr, "Error: image does not contain frame times.\n");
    imgEmpty(&img); return(2);
  }

  /* If frame time range was given, get frame number range */
  if(times>=0) {
    if(verbose>1) printf("PET time range: %g - %g s\n", img.start[0], img.end[img.dimt-1]);
    fstart=fstop=-1;
    for(int fi=0; fi<img.dimt; fi++) if(img.mid[fi]>=tstart) {fstart=fi; break;} 
    for(int fi=img.dimt-1; fi>=0; fi--) if(img.mid[fi]<=tstop) {fstop=fi; break;} 
    if(fstart<0 || fstop<0) {
      fprintf(stderr, "Error: invalid time range.\n");
      imgEmpty(&img); return(2);
    }
    if(verbose>1) {
      printf("cut_start_frame := %d\n", 1+fstart);
      printf("cut_stop_frame := %d\n", 1+fstop);
    }
  }
  /* If frame number range was given, check the range, and convert to indices */
  if(times<0) {
    if(fstart<1) fstart=1;
    if(fstop>img.dimt) fstop=img.dimt;
    fstart--; fstop--;
  }
  if(fstart>fstop || fstart>=img.dimt || fstop<0) {
    fprintf(stderr, "Error: invalid frame time range.\n");
    imgEmpty(&img); return(1);
  }
  if(verbose>1) {
    printf("Selected PET time range: %g - %g s\n", img.start[fstart], img.end[fstop]);
  }


  /*
   *  If contents of selected frames are just to be set to zero,
   *  then do it in place, save the image in new file, and exit.
   */
  if(doCrop==0) {
    if(verbose>1) printf("setting selected pixel values to zero...\n");
    int fi, zi, yi, xi;
    for(fi=fstart; fi<=fstop; fi++)
      for(zi=0; zi<img.dimz; zi++)
        for(yi=0; yi<img.dimy; yi++)
          for(xi=0; xi<img.dimx; xi++)
            img.m[zi][yi][xi][fi]=0.0;
    if(verbose>1) fprintf(stdout, "writing image...\n");
    ret=imgWrite(outfile, &img);
    if(ret) {
      fprintf(stderr, "Error: %s\n", img.statmsg);
      imgEmpty(&img); return(11);
    }
    if(verbose>0) fprintf(stdout, "%s written.\n", outfile);
    imgEmpty(&img);
    return(0);
  }


  /*
   *  Create new IMG without the frames marked to be deleted,
   *  and save it.
   */
  if(verbose>1) printf("deleting selected frames...\n");
  /* Calculate the nr of remaining frames */
  int frameNr=img.dimt-(1+fstop-fstart);
  if(frameNr<1) {
    fprintf(stderr, "Error: all frames would be cropped.\n"); 
    imgEmpty(&img);
    return(4);
  }
  if(verbose>2) fprintf(stdout, "allocating memory\n");
  IMG out; imgInit(&out);
  ret=imgAllocateWithHeader(&out, img.dimz, img.dimy, img.dimx, frameNr, &img);
  if(ret) {
    fprintf(stderr, "Error: cannot allocate memory for new image.\n"); 
    imgEmpty(&img);
    return(5);
  }
  /* Copy data from the original image */
  if(verbose>2) fprintf(stdout, "copying data\n");
  int fi, fj=0, zi, yi, xi;
  for(fi=0; fi<fstart; fi++, fj++) {
    for(zi=0; zi<img.dimz; zi++)
      for(yi=0; yi<img.dimy; yi++)
        for(xi=0; xi<img.dimx; xi++)
          out.m[zi][yi][xi][fj]=img.m[zi][yi][xi][fi];
    out.start[fj]=img.start[fi];
    out.end[fj]=img.end[fi];
    out.mid[fj]=img.mid[fi];
  }
  for(fi=fstop+1; fi<img.dimt; fi++, fj++) {
    for(zi=0; zi<img.dimz; zi++)
      for(yi=0; yi<img.dimy; yi++)
        for(xi=0; xi<img.dimx; xi++)
          out.m[zi][yi][xi][fj]=img.m[zi][yi][xi][fi];
    out.start[fj]=img.start[fi];
    out.end[fj]=img.end[fi];
    out.mid[fj]=img.mid[fi];
  }
  imgEmpty(&img);
  if(verbose>1) fprintf(stdout, "writing cropped image...\n");
  ret=imgWrite(outfile, &out);
  if(ret) {
    fprintf(stderr, "Error: %s\n", out.statmsg);
    imgEmpty(&out); return(11);
  }
  if(verbose>0) fprintf(stdout, "%s written.\n", outfile);
  imgEmpty(&out);

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
