/** @file imginteg.c
 *  @brief Calculates an AUC image from dynamic PET image data.
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <math.h>
#include <time.h>
/*****************************************************************************/
#include "libtpcmisc.h"
#include "libtpcmodel.h"
#include "libtpccurveio.h"
#include "libtpcimgio.h"
#include "libtpcimgp.h"
#include "libtpcmodext.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Calculates AUC (integral over time) image from PET image in",
  "in ECAT 6.3, 7.x, NIfTI-1, or Analyze 7.5 format.",
  "Analyze and NIfTI image must have SIF in the same folder.",
  "In case of a static image, the matrix values are simply multiplied by the",
  "frame duration.",
  "The pixel values of resulting integral image will be in units of",
  "radioactivity concentration times sec.",
  " ",
  "Usage: @P [Options] image starttime duration aucimage",
  " ",
  "AUC start time and duration (not stop time) must be entered in seconds;",
  "program will automatically set the integration start time or duration",
  "based on the time range in the image, if either is set to zero.",
  " ",
  "Options:",
  " -avg",
  "     Average during specified range is calculated instead of AUC.",
  " -min",
  "     AUC times are given in minutes instead of seconds.",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "Example 1: calculate integral image between 32 and 302 seconds after",
  "scan start with command",
  "  @P s5998dy1.v 32 270 s5998int.v",
  " ",
  "Example 2: integrate a static or dynamic image from the start time of",
  "the first frame to the end of the last frame:",
  "  @P a773dy1.v 0 0 a773int.v", 
  " ",
  "Example 3: integrate a dynamic image from 600 seconds to the end of",
  "the last frame:",
  "  @P a773dy1.v 600 0 a773int.v", 
  " ",
  "See also: imgunit, ecattime, ecatsum, imgsuv, imgledif, imgfur, imglkup",
  " ",
  "Keywords: image, AUC, modelling, autoradiography, perfusion, SUV",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int      ai, help=0, version=0, verbose=1;
  int      ret, isStat=0;
  int      calc_avg=0; // 0=integral, 1=average
  int      times_in_min=0; // 0=sec, 1=min
  char     petfile[FILENAME_MAX], intfile[FILENAME_MAX];
  char    *cptr, tmp[512];
  IMG      pet, rout;
  float    tstart, tstop, tdur;
  
  
  
  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  petfile[0]=intfile[0]=(char)0;
  tstart=tdur=tstop=-1.0;
  imgInit(&pet); imgInit(&rout);
  /* Get options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') {
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(cptr==NULL) continue;
    if(strcasecmp(cptr, "AVG")==0 || strcasecmp(cptr, "MEAN")==0) {
      calc_avg=1; continue;
    } else if(strncasecmp(cptr, "MINUTES", 3)==0) {
      times_in_min=1; continue;
    } else if(strncasecmp(cptr, "SECONDS", 3)==0) {
      times_in_min=0; continue;
    }
    fprintf(stderr, "Error: invalid option '%s'.\n", argv[ai]);
    return(1);
  } else break;
  
  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  /* Process other arguments, starting from the first non-option */
  if(ai<argc) {strlcpy(petfile, argv[ai], FILENAME_MAX); ai++;}
  if(ai<argc) {tstart=atof_dpi(argv[ai]); ai++;}
  if(ai<argc) {tdur=atof_dpi(argv[ai]); tstop=tstart+tdur; ai++;}
  if(ai<argc) {strlcpy(intfile, argv[ai], FILENAME_MAX); ai++;}
  if(ai<argc) {fprintf(stderr, "Error: invalid argument '%s'.\n", argv[ai]); return(1);}

  /* Did we get all the information that we need? */
  if(!intfile[0]) {
    fprintf(stderr, "Error: missing command-line argument; use option --help\n");
    return(1);
  }
  if(strcasecmp(intfile, petfile)==0) {
    fprintf(stderr, "Error: check the output file name.\n");
    return(1);
  }
  if(tstop<tstart || tstart<0.0) {
    fprintf(stderr, "Error: invalid time range arguments.\n");
    return(1);
  }
  /* Convert AUC times to seconds, if necessary */
  if(times_in_min==1) {tstart*=60.0; tstop*=60.0; tdur*=60.0;}


  /* In verbose mode print arguments and options */
  if(verbose>1) {
    printf("petfile := %s\n", petfile);
    printf("intfile := %s\n", intfile);
    printf("calc_avg := %d\n", calc_avg);
    printf("times_in_min := %d\n", times_in_min);
  }
  if(verbose>9) IMG_TEST=verbose-9; else IMG_TEST=0;
  

  /*
   *  Read image
   */
  if(verbose>0) fprintf(stdout, "reading image %s\n", petfile);
  ret=imgRead(petfile, &pet);
  if(ret) {
    fprintf(stderr, "Error: %s\n", pet.statmsg); if(verbose>1) imgInfo(&pet);
    return(2);
  }
  /* Check if PET data is raw or image */
  if(pet.type!=IMG_TYPE_IMAGE) {
    fprintf(stderr, "Error: %s is not an image.\n", petfile);
    imgEmpty(&pet); return(2);
  }
  if(pet.dimt<2) {
    isStat=1;
    fprintf(stderr, "Warning: %s is assumed to be a static image.\n", petfile);
  }
  if(verbose>1) printf("isStat := %d\n", isStat);

  /* Check that frame times are available */
  if(!imgExistentTimes(&pet)) {
    fprintf(stderr, "Error: image does not contain frame times.\n");
    imgEmpty(&pet); return(2);
  }

  /* Make sure that there is no overlap in image frames */ 
  if(pet.dimt>1) {
    if(verbose>1) fprintf(stdout, "checking frame overlap in %s\n", petfile);
    ret=imgDeleteFrameOverlap(&pet); 
    if(ret) {
      fprintf(stderr, "Error: image %s has overlapping frame times.\n", petfile);
      imgEmpty(&pet); return(2);
    }
  }

  /* If both integration start time and duration were set to zero, then take these from the image */
  if(tstart<1.0E-03 && tdur<1.0E-02) {
    tstart=pet.start[0]; tstop=pet.end[pet.dimt-1]; tdur=tstop-tstart;
  } else if(tdur<1.0E-02) {
    /* If duration was set to zero, then take it from the image */
    tstop=pet.end[pet.dimt-1]; tdur=tstop-tstart;
  }
  if(verbose>1) {
    printf("tstart := %g\n", tstart);
    printf("tstop := %g\n", tstop);
    printf("tdur := %g\n", tdur);
  }


  /*
   *  Calculate the integral image (or average)
   */
  if(verbose>1) fprintf(stdout, "calculating integral image\n");
  ret=imgTimeIntegral(&pet, tstart, tstop, &rout, calc_avg, tmp, verbose-2);
  if(ret!=0) {
    fprintf(stderr, "Error: %s.\n", tmp);
    imgEmpty(&pet); imgEmpty(&rout); return(6);
  }
  if(verbose>0) fprintf(stdout, "%s.\n", tmp);
  imgEmpty(&pet);


  /*
   *  Save parametric image(s)
   */
  if(verbose>1) fprintf(stdout, "writing integral image\n");
  ret=imgWrite(intfile, &rout);
  if(ret) {
    fprintf(stderr, "Error: %s\n", rout.statmsg);
    imgEmpty(&rout); return(11);
  }
  if(verbose>0) {
    if(calc_avg) fprintf(stdout, "mean image written in %s\n", intfile);
    else fprintf(stdout, "integral image written in %s\n", intfile);
  }
  imgEmpty(&rout);

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
