/** @file imgmask.c
 *  @brief Apply mask image to PET image file.
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include <unistd.h>
#include <time.h>
/*****************************************************************************/
#include "libtpcmisc.h"
#include "libtpcimgio.h"
#include "libtpcimgp.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Apply mask image(s) to PET image file in ECAT 6.3 or 7.x, NIfTI-1, or",
  "Analyze 7.5 format. Mask and PET image must have the same dimensions, except",
  "if z and/or t dimension in mask image is 1, it is applied to all PET image",
  "planes and frames, respectively.",
  "By default, those pixels that have value 0 in the mask image will be set to",
  "zero in the PET image.",
  "Notice that PET image file is overwritten; make sure that backup exists.",
  " ",
  "Usage: @P [Options] imgfile maskfile [maskedfile]",
  " ",
  "Options:",
  " -inv",
  "     Pixels that have non-zero value in the mask image will be set to zero",
  "     in the PET image.", 
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "See also: imgcalc, imgslim, imgadd, simcirc, flat2img, imgmax, maskinv",
  " ",
  "Keywords: image, mask, threshold",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int     ai, help=0, version=0, verbose=1;
  int     ret;
  int     invert=0, overwrite=0;
  char    imgfile[FILENAME_MAX], maskfile[FILENAME_MAX], outfile[FILENAME_MAX];
  char   *cptr=NULL;
  IMG     img, mask;


  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  imgfile[0]=maskfile[0]=outfile[0]=(char)0;
  imgInit(&img); imgInit(&mask);
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') { /* options */
    cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(cptr==NULL) continue;
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    if(strncasecmp(cptr, "INVERT", 3)==0) {
      invert=1; continue;
    }
    fprintf(stderr, "Error: invalid option '%s'.\n", argv[ai]); 
    return(1);
  } else break;
  
  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  /* Process other arguments, starting from the first non-option */
  if(ai<argc) {strlcpy(imgfile, argv[ai], FILENAME_MAX); ai++;}
  if(ai<argc) {strlcpy(maskfile, argv[ai], FILENAME_MAX); ai++;}
  if(ai<argc) {strlcpy(outfile, argv[ai], FILENAME_MAX); ai++;}
  if(ai<argc) {fprintf(stderr, "Error: too many arguments.\n"); return(1);}

  /* Did we get all the information that we need? */
  if(!maskfile[0]) {
    fprintf(stderr, "Error: missing command-line argument; use option --help\n");
    return(1);
  }
  if(!outfile[0]) {strcpy(outfile, imgfile); overwrite=1;}


  /* In verbose mode print options */
  if(verbose>1) {
    printf("imgfile := %s\n", imgfile);
    printf("maskfile := %s\n", maskfile);
    printf("outfile := %s\n", outfile);
    printf("invert := %d\n", invert);
  }


  /*
   *  Read PET and mask image
   */
  if(verbose>0) printf("reading %s\n", imgfile);
  ret=imgRead(imgfile, &img);
  if(ret) {
    fprintf(stderr, "Error: %s\n", img.statmsg);
    if(verbose>1) printf("ret := %d\n", ret);
    return(2);
  }
  if(verbose>0) printf("reading %s\n", maskfile);
  ret=imgRead(maskfile, &mask);
  if(ret) {
    fprintf(stderr, "Error: %s\n", mask.statmsg);
    if(verbose>1) printf("ret := %d\n", ret);
    imgEmpty(&img); return(3);
  }
  /* Check that dimensions are compatible */
  if(img.dimx!=mask.dimx || img.dimy!=mask.dimy) {
    fprintf(stderr, "Error: different image dimensions.\n");
    if(verbose>0) {
      printf("image := %d x %d\n", img.dimx, img.dimy);
      printf("mask := %d x %d\n", mask.dimx, mask.dimy);
    }
    imgEmpty(&img); imgEmpty(&mask); return(4);
  }
  if(mask.dimz!=1 && img.dimz!=mask.dimz) {
    fprintf(stderr, "Error: different image dimensions.\n");
    if(verbose>0) {
      printf("image := %d x %d x %d\n", img.dimx, img.dimy, img.dimz);
      printf("mask := %d x %d x %d\n", mask.dimx, mask.dimy, mask.dimz);
    }
    imgEmpty(&img); imgEmpty(&mask); return(4);
  }
  if(mask.dimt!=1 && img.dimt!=mask.dimt) {
    fprintf(stderr, "Error: different image dimensions.\n");
    if(verbose>0) {
      printf("image := %d x %d x %d x %d\n", 
             img.dimx, img.dimy, img.dimz, img.dimt);
      printf("mask := %d x %d x %d x %d\n", 
             mask.dimx, mask.dimy, mask.dimz, mask.dimt);
    }
    imgEmpty(&img); imgEmpty(&mask); return(4);
  }


  /*
   *  Process the image matrix
   */
  if(verbose>0) printf("processing image...\n");
  int ti, zi, yi, xi, tti, tzi;
  long long n=0;
  for(zi=0; zi<img.dimz; zi++) {
    if(mask.dimz==1) tzi=0; else tzi=zi;
    for(yi=0; yi<img.dimy; yi++)
      for(xi=0; xi<img.dimx; xi++)
        for(ti=0; ti<img.dimt; ti++) {
          if(mask.dimt==1) tti=0; else tti=ti;
          if(invert==0 && (fabs(mask.m[tzi][yi][xi][tti])<0.5 || !isfinite(mask.m[tzi][yi][xi][tti]))) {
            img.m[zi][yi][xi][ti]=0.0; n++;
          } else if(invert==1 && fabs(mask.m[tzi][yi][xi][tti])>0.5) {
            img.m[zi][yi][xi][ti]=0.0; n++;
          }
        }
  }
  imgEmpty(&mask);
  if(n==0) {
    fprintf(stderr, "Warning: no pixels modified.\n");
    if(overwrite) { // no reason to overwrite file when data not modified
      imgEmpty(&img);
      return(0);
    }
  }


  /*
   *  Save the modified image
   */
  if(verbose>0) printf("writing image %s\n", outfile);
  ret=imgWrite(outfile, &img);
  if(ret) {
    fprintf(stderr, "Error: %s\n", img.statmsg);
    imgEmpty(&img); return(11);
  }
  imgEmpty(&img);
  if(verbose>0) printf("done.\n\n");

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
