/** @file imgslim.c
 *  @brief Slice off empty parts in dynamic PET image.
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include <unistd.h>
#include <time.h>
/*****************************************************************************/
#include "libtpcmisc.h"
#include "libtpcimgio.h"
#include "libtpcimgp.h"
#include "libtpcmodel.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Slice off empty parts (rows, columns, planes) in dynamic PET image file in",
  "ECAT, NIfTI, or Analyze format.",
  " ",
  "Usage: @P [Options] imgfile [outputfile]",
  " ",
  "Options:",
  " -limit=<value>",
  "     Slices, where average pixel value is less than the limit value,",
  "     are cut off; by default 0.1.",
  " -keepframes | -slimframes",
  "     Selects whether time frames can be sliced off",
  "     (by default, frames are always kept).",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "Example:",
  "     @P b123dy1.v b123dy1_slim.v",
  " ",
  "See also: imgdim, imgshrink, imgbkgrm, imgthrs, imgbox",
  " ",
  "Keywords: image, compression, cropping, threshold",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int      ai, help=0, version=0, verbose=1;
  char     petfile[FILENAME_MAX], outfile[FILENAME_MAX];
  double   limit_mean=nan("");
  int      keep_frames=1; // 0=off; 1=on


  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  petfile[0]=outfile[0]=(char)0;
  /* Get options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') {
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    char *cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(cptr==NULL) continue;
    if(strncasecmp(cptr, "KEEPFRAMES", 5)==0) {
      keep_frames=1; continue;
    } else if(strncasecmp(cptr, "SLIMFRAMES", 5)==0) {
      keep_frames=0; continue;
    } else if(strncasecmp(cptr, "LIMIT=", 6)==0) {
      if(!atof_with_check(cptr+6, &limit_mean)) continue;
    }
    fprintf(stderr, "Error: invalid option '%s'.\n", argv[ai]);
    return(1);
  } else break;
  
  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}
  
  /* Process other arguments, starting from the first non-option */
  if(ai<argc) {strlcpy(petfile, argv[ai], FILENAME_MAX); ai++;}
  if(ai<argc) {strlcpy(outfile, argv[ai], FILENAME_MAX); ai++;}
  if(ai<argc) {fprintf(stderr, "Error: too many arguments.\n"); return(1);}


  /* Did we get all the information that we need? */
  if(!petfile[0]) {
    fprintf(stderr, "Error: missing command-line argument; use option --help\n");
    return(1);
  }
  if(!outfile[0]) strcpy(outfile, petfile);


  /* In verbose mode print arguments and options */
  if(verbose>1) {
    printf("petfile := %s\n", petfile);
    printf("outfile := %s\n", outfile);
    if(!isnan(limit_mean)) printf("limit_mean := %g\n", limit_mean);
    printf("keep_frames := %d\n", keep_frames);
  }

  if(isnan(limit_mean)) limit_mean=0.1;


  /*
   *  Read the contents of the PET file to img data structure
   */
  IMG img; imgInit(&img);
  if(verbose>0) printf("reading %s\n", petfile);
  int ret=imgRead(petfile, &img);
  if(ret) {
    fprintf(stderr, "Error: %s\n", img.statmsg);
    if(verbose>1) printf("ret=%d\n", ret);
    imgEmpty(&img); return(2);
  }
  /* Check that PET data is image */
  if(img.type!=IMG_TYPE_IMAGE) {
    fprintf(stderr, "Error: %s is not an image.\n", petfile);
    imgEmpty(&img); return(2);
  }
  /* Check that we have dynamic data */
  if(img.dimt<2) {
    fprintf(stderr, "Error: method works only for dynamic data.\n");
    imgEmpty(&img); return(2);
  }
  if(imgNaNs(&img, 1)>0)
    if(verbose>0) fprintf(stderr, "Warning: missing pixel values.\n");
  /*
   *  Print original image dimensions
   */
  if(verbose>0) {
    fprintf(stdout, "image_dimensions := %d %d %d\n", img.dimx, img.dimy, img.dimz);
    fprintf(stdout, "frame_nr := %d\n", img.dimt);
  }


  /*
   *  Set tables for slimming
   */
  typedef struct {
    double sum;
    double weight;
    char keep;
  } SLIM_LIST;

  SLIM_LIST xlist[img.dimx];
  SLIM_LIST ylist[img.dimy];
  SLIM_LIST zlist[img.dimz];
  SLIM_LIST tlist[img.dimt];

  for(int xi=0; xi<img.dimx; xi++) {
    xlist[xi].sum=0.0;
    xlist[xi].weight=0.0;
    xlist[xi].keep=1;
  }
  for(int yi=0; yi<img.dimy; yi++) {
    ylist[yi].sum=0.0;
    ylist[yi].weight=0.0;
    ylist[yi].keep=1;
  }
  for(int zi=0; zi<img.dimz; zi++) {
    zlist[zi].sum=0.0;
    zlist[zi].weight=0.0;
    zlist[zi].keep=1;
  }
  for(int ti=0; ti<img.dimt; ti++) {
    tlist[ti].sum=0.0;
    tlist[ti].weight=0.0;
    tlist[ti].keep=1;
  }

  /* Fill the slim tables */
  for(int zi=0; zi<img.dimz; zi++) {
    for(int yi=0; yi<img.dimy; yi++) {
      for(int xi=0; xi<img.dimx; xi++) {
        for(int fi=0; fi<img.dimt; fi++) {
          xlist[xi].sum+=img.m[zi][yi][xi][fi]; xlist[xi].weight+=1.0;
          ylist[yi].sum+=img.m[zi][yi][xi][fi]; ylist[yi].weight+=1.0;
          zlist[zi].sum+=img.m[zi][yi][xi][fi]; zlist[zi].weight+=1.0;
          tlist[fi].sum+=img.m[zi][yi][xi][fi]; tlist[fi].weight+=1.0;
        }
      }
    }
  }

  /*
   *  Determine which slices are to be kept and which are not
   */
  /* check values against the limits */
  for(int xi=0; xi<img.dimx; xi++) {
    xlist[xi].keep=(xlist[xi].sum/xlist[xi].weight<limit_mean) ? 0 : 1 ;
  }
  for(int yi=0; yi<img.dimy; yi++) {
    ylist[yi].keep=(ylist[yi].sum/ylist[yi].weight<limit_mean) ? 0 : 1 ;
  }
  for(int zi=0; zi<img.dimz; zi++) {
    zlist[zi].keep=(zlist[zi].sum/zlist[zi].weight<limit_mean) ? 0 : 1 ;
  }
  for(int ti=0; ti<img.dimt; ti++) {
    tlist[ti].keep=(tlist[ti].sum/tlist[ti].weight<limit_mean) ? 0 : 1 ;
  }
  if(verbose>3) { /* Show the initial slim lists */
    printf("\nxlist:\n");
    for(int xi=0; xi<img.dimx; xi++) {
      printf("  %04d  %20.5f  %20.5f  %d\n",
        xi+1, xlist[xi].sum, xlist[xi].weight, xlist[xi].keep);
    }
    printf("ylist:\n");
    for(int yi=0; yi<img.dimy; yi++) {
      printf("  %04d  %20.5f  %20.5f  %d\n",
        yi+1, ylist[yi].sum, ylist[yi].weight, ylist[yi].keep);
    }
    printf("zlist:\n");
    for(int zi=0; zi<img.dimz; zi++) {
      printf("  %04d  %20.5f  %20.5f  %d\n",
        zi+1, zlist[zi].sum, zlist[zi].weight, zlist[zi].keep);
    }
    printf("tlist:\n");
    for(int ti=0; ti<img.dimt; ti++) {
      printf("  %04d  %20.5f  %20.5f  %d\n",
        ti+1, tlist[ti].sum, tlist[ti].weight, tlist[ti].keep);
    }
  }

  /* keep all slices that are between slices that are kept */
  int newdimx, newdimy, newdimz, newdimt;

  {
  int i, first, last;
  i=0; while(i<img.dimx && xlist[i].keep==0) i++;
  first=i;
  i=img.dimx-1; while(i>=0 && xlist[i].keep==0) i--;
  last=i;
  if(first>last) {first=0; last=img.dimx-1;}
  for(i=first; i<=last; i++) xlist[i].keep=1;
  newdimx=1+last-first;
  if(verbose>1) printf("x_first := %d\nx_last := %d", first, last);
  }

  {
  int i, first, last;
  i=0; while(i<img.dimy && ylist[i].keep==0) i++;
  first=i;
  i=img.dimy-1; while(i>=0 && ylist[i].keep==0) i--;
  last=i;
  if(first>last) {first=0; last=img.dimy-1;}
  for(i=first; i<=last; i++) ylist[i].keep=1;
  newdimy=1+last-first;
  if(verbose>1) printf("y_first := %d\ny_last := %d", first, last);
  }

  {
  int i, first, last;
  i=0; while(i<img.dimz && zlist[i].keep==0) i++;
  first=i;
  i=img.dimz-1; while(i>=0 && zlist[i].keep==0) i--;
  last=i;
  if(first>last) {first=0; last=img.dimz-1;}
  for(i=first; i<=last; i++) zlist[i].keep=1;
  newdimz=1+last-first;
  if(verbose>1) printf("z_first := %d\nz_last := %d", first, last);
  }

  {
  int i, first, last;
  i=0; while(i<img.dimt && tlist[i].keep==0) i++;
  first=i;
  i=img.dimt-1; while(i>=0 && tlist[i].keep==0) i--;
  last=i;
  if(first>last) {first=0; last=img.dimt-1;}
  for(i=first; i<=last; i++) tlist[i].keep=1;
  newdimt=1+last-first;
  if(verbose>1) printf("t_first := %d\nt_last := %d", first, last);
  }

  /* If user wants to keep all time frames, then put them back */
  if(keep_frames && newdimt<img.dimt) {
    if(verbose>1) printf("keeping all time frames\n");
    for(int ti=0; ti<img.dimt; ti++) tlist[ti].keep=1;
    newdimt=img.dimt;
  }

  if(verbose>2) { /* Show the final slim lists */
    printf("\nxlist:\n");
    for(int xi=0; xi<img.dimx; xi++) {
      printf("  %04d  %20.5f  %20.5f  %d\n",
        xi+1, xlist[xi].sum, xlist[xi].weight, xlist[xi].keep);
    }
    printf("ylist:\n");
    for(int yi=0; yi<img.dimy; yi++) {
      printf("  %04d  %20.5f  %20.5f  %d\n",
        yi+1, ylist[yi].sum, ylist[yi].weight, ylist[yi].keep);
    }
    printf("zlist:\n");
    for(int zi=0; zi<img.dimz; zi++) {
      printf("  %04d  %20.5f  %20.5f  %d\n",
        zi+1, zlist[zi].sum, zlist[zi].weight, zlist[zi].keep);
    }
    printf("tlist:\n");
    for(int ti=0; ti<img.dimt; ti++) {
      printf("  %04d  %20.5f  %20.5f  %d\n",
        ti+1, tlist[ti].sum, tlist[ti].weight, tlist[ti].keep);
    }
  }
  if(verbose>0) {
    printf("  new_dimx := %d\n", newdimx);
    printf("  new_dimy := %d\n", newdimy);
    printf("  new_dimz := %d\n", newdimz);
    printf("  new_dimt := %d\n", newdimt);
  }
  /* Check if anything needs to be done */
  if(newdimx==img.dimx && newdimy==img.dimy && newdimz==img.dimz && newdimt==img.dimt) {
    fprintf(stderr, "Error: cannot slim-off anything from %s\n", petfile);
    imgEmpty(&img); return(4);
  }
  /* Check that something is left */
  if(newdimx==0 || newdimy==0 || newdimz==0 || newdimt==0) {
    fprintf(stderr, "Error: this would result in empty image.\n");
    imgEmpty(&img); return(5);
  }


  /*
   *  Create IMG for output data
   */
  IMG out; imgInit(&out);
  if(imgAllocateWithHeader(&out, newdimz, newdimy, newdimx, newdimt, &img)!=0) {
    fprintf(stderr, "Error: cannot make output image.\n");
    imgEmpty(&img); imgEmpty(&out); return(8);
  }
  /* Copy pixel data */
  for(int ti=0, nti=0; ti<img.dimt; ti++) if(tlist[ti].keep==1) {
    for(int zi=0, nzi=0; zi<img.dimz; zi++) if(zlist[zi].keep==1) {
      for(int yi=0, nyi=0; yi<img.dimy; yi++) if(ylist[yi].keep==1) {
        for(int xi=0, nxi=0; xi<img.dimx; xi++) if(xlist[xi].keep==1) {
          out.m[nzi][nyi][nxi][nti]=img.m[zi][yi][xi][ti];
          nxi++;
        }
        nyi++;
      }
      nzi++;
    }
    out.start[nti]=img.start[ti]; out.end[nti]=img.end[ti];
    nti++;
  }

  /* No need for original image */
  imgEmpty(&img);


  /*
   *  Save the slim image
   */
  if(verbose>1) fprintf(stdout, "writing image...\n");
  ret=imgWrite(outfile, &out);
  if(ret) {
    fprintf(stderr, "Error: %s\n", out.statmsg);
    imgEmpty(&out); return(11);
  }
  if(verbose==1) printf("Written image %s\n", outfile);

  imgEmpty(&out);
  if(verbose>0) printf("done.\n");
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
